package ru.yandex.intranet.d.services.transfer;

import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.transfers.QuotaTransfer;
import ru.yandex.intranet.d.model.transfers.ResourceQuotaTransfer;
import ru.yandex.intranet.d.model.transfers.TransferApplicationDetails;
import ru.yandex.intranet.d.model.transfers.TransferNotified;
import ru.yandex.intranet.d.model.transfers.TransferParameters;
import ru.yandex.intranet.d.model.transfers.TransferRequestByFolderModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByServiceModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestEventType;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryFields;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestStatus;
import ru.yandex.intranet.d.model.transfers.TransferRequestSubtype;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.model.transfers.TransferResponsible;
import ru.yandex.intranet.d.model.transfers.TransferVote;
import ru.yandex.intranet.d.model.transfers.TransferVotes;
import ru.yandex.intranet.d.model.transfers.VoteType;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.transfer.model.CreateTransferRequestConfirmationData;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedReserveTransfer;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedTransferRequestParameters;
import ru.yandex.intranet.d.services.transfer.model.PutTransferRequestConfirmationData;
import ru.yandex.intranet.d.services.transfer.model.QuotaTransferPreApplicationData;
import ru.yandex.intranet.d.services.transfer.model.ResponsibleAndNotified;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestIndices;
import ru.yandex.intranet.d.services.transfer.model.ValidatedCreateTransferRequest;
import ru.yandex.intranet.d.services.transfer.model.ValidatedPutTransferRequest;
import ru.yandex.intranet.d.services.transfer.model.ValidatedQuotaResourceTransfer;
import ru.yandex.intranet.d.services.transfer.model.ValidatedReserveTransfer;
import ru.yandex.intranet.d.services.transfer.model.ValidatedReserveTransferParameters;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontPutTransferRequestDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static ru.yandex.intranet.d.services.transfer.TransferRequestQuotaService.addResponsibleIndex;
import static ru.yandex.intranet.d.services.transfer.TransferRequestQuotaService.copyVotes;
import static ru.yandex.intranet.d.services.transfer.TransferRequestQuotaService.prepareConfirmationVote;
import static ru.yandex.intranet.d.services.transfer.TransferRequestQuotaService.prepareNotifiedUsersCreate;
import static ru.yandex.intranet.d.services.transfer.TransferRequestQuotaService.prepareNotifiedUsersPut;
import static ru.yandex.intranet.d.services.transfer.TransferRequestQuotaService.updatedQuotaTransferRequestMustBeApplied;

/**
 * Transfer request reserve service implementation.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
@Component
public class TransferRequestReserveService {

    private final TransferRequestValidationService validationService;
    private final TransferRequestStoreService storeService;
    private final TransferRequestIndicesService indicesService;
    private final TransferRequestResponsibleAndNotifyService transferRequestResponsibleAndNotifyService;
    private final MessageSource messages;

    public TransferRequestReserveService(
            TransferRequestValidationService validationService,
            TransferRequestStoreService storeService,
            TransferRequestIndicesService indicesService,
            TransferRequestResponsibleAndNotifyService transferRequestResponsibleAndNotifyService,
            @Qualifier("messageSource") MessageSource messages) {
        this.validationService = validationService;
        this.storeService = storeService;
        this.indicesService = indicesService;
        this.transferRequestResponsibleAndNotifyService = transferRequestResponsibleAndNotifyService;
        this.messages = messages;
    }

    public Mono<Result<ValidatedCreateTransferRequest>> validateReserveTransfer(
            YdbTxSession txSession,
            FrontCreateTransferRequestDto transferRequest,
            YaUserDetails currentUser,
            Locale locale,
            boolean publicApi,
            boolean delayValidation
    ) {
        PreValidatedTransferRequestParameters.Builder preValidatedRequestBuilder
                = PreValidatedTransferRequestParameters.builder();
        ErrorCollection.Builder preErrorsBuilder = ErrorCollection.builder();

        validationService.validateCreateFields(transferRequest::getDescription, transferRequest::getParameters,
                preValidatedRequestBuilder, preErrorsBuilder, locale);

        if (transferRequest.getLoanParameters().isPresent()) {
            preErrorsBuilder.addError("loanParameters", TypedError.invalid(messages
                    .getMessage("errors.loan.is.not.supported", null, locale)));
        }

        if (transferRequest.getParameters().isPresent()) {
            PreValidatedReserveTransfer.Builder preValidatedReserveTransferBuilder =
                    PreValidatedReserveTransfer.builder();

            validationService.preValidateReserveTransferParameters(transferRequest.getParameters().get(),
                    preValidatedReserveTransferBuilder, preErrorsBuilder, locale, publicApi);

            if (preErrorsBuilder.hasAnyErrors()) {
                return Mono.just(Result.failure(preErrorsBuilder.build()));
            }

            PreValidatedReserveTransfer preValidatedReserveTransfer = preValidatedReserveTransferBuilder.build();
            PreValidatedTransferRequestParameters preValidatedRequest = preValidatedRequestBuilder.build();

            return validateReserveTransfer(txSession, preValidatedReserveTransfer, currentUser, locale, publicApi)
                    .flatMap(r -> r.andThenMono(parameters ->
                            validationService.validateReserveTransferApplication(txSession, parameters, locale,
                                            delayValidation)
                                    .flatMap(preApplicationR -> preApplicationR.andThenMono(preApplication ->
                                            transferRequestResponsibleAndNotifyService
                                                    .calculateForReserveTransferCreate(txSession, parameters,
                                                            currentUser)
                                                    .map(responsible -> validationService.validateResponsible(
                                                            responsible, transferRequest::getAddConfirmation,
                                                            currentUser, locale).apply(confirmation ->
                                                            prepareCreateReserveTransferRequest(preValidatedRequest,
                                                                    parameters, preApplication, responsible,
                                                                    confirmation)))))));
        } else {
            return Mono.just(Result.failure(preErrorsBuilder.build()));
        }
    }

    Mono<Result<ValidatedReserveTransferParameters>> validateReserveTransfer(
            YdbTxSession txSession, PreValidatedReserveTransfer preValidatedReserveTransfer, YaUserDetails currentUser,
            Locale locale, boolean publicApi) {
        Set<Long> serviceIds = new HashSet<>();
        preValidatedReserveTransfer.getDestinationServiceId().ifPresent(serviceIds::add);
        Set<String> resourceIds = new HashSet<>();
        preValidatedReserveTransfer.getResourceTransfers()
                .forEach(resourceTransfer -> resourceIds.add(resourceTransfer.getResourceId()));

        String destinationFolderId = preValidatedReserveTransfer.getDestinationFolderId();
        return storeService.loadProvider(txSession, preValidatedReserveTransfer.getProviderId()).flatMap(provider ->
                getReserveFolder(txSession, provider).flatMap(reserveFolder ->
                        storeService.loadFolder(txSession, destinationFolderId).flatMap(folder -> {
                            Set<Long> moreServiceIds = Stream.of(folder.map(FolderModel::getServiceId),
                                    provider.map(ProviderModel::getServiceId))
                                    .filter(Optional::isPresent)
                                    .map(Optional::get).collect(Collectors.toSet());
                            Set<Long> serviceIdsToLoad = Sets.union(serviceIds, moreServiceIds);
                            return storeService.loadServices(txSession, serviceIdsToLoad).flatMap(services ->
                                    storeService.loadResources(txSession, resourceIds).flatMap(resources -> {
                                        Set<String> unitsEnsemblesIds = resources.stream()
                                                .map(ResourceModel::getUnitsEnsembleId)
                                                .collect(Collectors.toSet());
                                        return storeService.loadUnitsEnsembles(txSession, unitsEnsemblesIds)
                                                .flatMap(unitsEnsembles ->
                                                        validationService.validateReserveTransferParameters(
                                                                preValidatedReserveTransfer, currentUser, locale,
                                                                folder, reserveFolder, services, resources, provider,
                                                                unitsEnsembles, publicApi)
                                                );
                                    }));
                        })));
    }

    private Mono<Optional<FolderModel>> getReserveFolder(YdbTxSession txSession, Optional<ProviderModel> provider) {
        if (provider.isEmpty()) {
            return Mono.just(Optional.empty());
        }
        return storeService.loadReserveFolder(txSession, provider.get().getServiceId())
                .map(folderModels -> folderModels.size() == 1
                        ? Optional.ofNullable(folderModels.get(0))
                        : Optional.empty());
    }

    private ValidatedCreateTransferRequest prepareCreateReserveTransferRequest(
            PreValidatedTransferRequestParameters preValidatedRequest,
            ValidatedReserveTransferParameters validatedParameters,
            QuotaTransferPreApplicationData preApplicationData,
            ResponsibleAndNotified responsibleAndNotified,
            CreateTransferRequestConfirmationData confirmation) {
        ValidatedReserveTransfer transfer = validatedParameters.getTransfer();
        String destinationFolderId = transfer.getDestinationFolder().getId();
        long destinationServiceId = transfer.getDestinationService().getId();
        String reserveFolderId = transfer.getReserveFolder().getId();
        long reserveServiceId = transfer.getReserveFolder().getServiceId();
        TransferResponsible responsible = responsibleAndNotified.getResponsible();
        Instant now = Instant.now();

        ValidatedCreateTransferRequest.Builder validatedBuilder = ValidatedCreateTransferRequest.builder();
        validatedBuilder.quotas(preApplicationData.getQuotas());
        validatedBuilder.folders(validatedParameters.getFolders());
        validatedBuilder.services(validatedParameters.getServices());
        validatedBuilder.resources(validatedParameters.getResources());
        validatedBuilder.unitsEnsembles(validatedParameters.getUnitsEnsembles());
        validatedBuilder.providers(List.of(transfer.getProvider()));
        validatedBuilder.now(now);
        validatedBuilder.apply(confirmation.isApply());

        TransferRequestModel.Builder transferRequestBuilder = TransferRequestModel.builder();
        TransferParameters.Builder parametersBuilder = TransferParameters.builder();
        List<ValidatedQuotaResourceTransfer> transfers = transfer.getTransfers();
        parametersBuilder.addQuotaTransfer(getQuotaTransfer(destinationFolderId, destinationServiceId, transfers,
                true));
        parametersBuilder.addQuotaTransfer(getQuotaTransfer(reserveFolderId, reserveServiceId, transfers, false));

        TransferVotes.Builder votesBuilder = TransferVotes.builder();
        if (confirmation.isConfirm()) {
            if (confirmation.isProviderResponsibleAutoConfirm()) {
                // See DISPENSER-3540
                TransferVote vote = prepareProviderAutoConfirmationVote(now, confirmation.getUser(),
                        validatedParameters);
                votesBuilder.addVote(vote);
            } else {
                TransferVote vote = prepareConfirmationVote(responsible, now, confirmation.getUser());
                votesBuilder.addVote(vote);
            }
        }

        TransferApplicationDetails.Builder applicationDetailsBuilder = TransferApplicationDetails.builder();
        if (confirmation.isApply()) {
            addFolderOpLogId(destinationFolderId, validatedBuilder::putPreGeneratedFolderOpLogId,
                    applicationDetailsBuilder);
            addFolderOpLogId(reserveFolderId, validatedBuilder::putPreGeneratedFolderOpLogId,
                    applicationDetailsBuilder);
        }

        TransferRequestHistoryModel.Builder historyBuilder = TransferRequestHistoryModel.builder();
        String transferRequestId = UUID.randomUUID().toString();
        TransferParameters transferParameters = parametersBuilder.build();
        TransferVotes transferVotes = votesBuilder.build();
        TransferApplicationDetails transferApplicationDetails = applicationDetailsBuilder.build();

        historyBuilder.id(UUID.randomUUID().toString());
        historyBuilder.tenantId(Tenants.DEFAULT_TENANT_ID);
        historyBuilder.transferRequestId(transferRequestId);
        historyBuilder.type(TransferRequestEventType.CREATED);
        historyBuilder.timestamp(now);
        historyBuilder.authorId(confirmation.getUser().getId());
        historyBuilder.oldFields(null);
        historyBuilder.oldParameters(null);
        historyBuilder.oldResponsible(null);
        historyBuilder.oldVotes(null);
        historyBuilder.oldApplicationDetails(null);
        String summary = generateSummary(transfer.getProvider(), transfer.getDestinationFolder(),
                transfer.getDestinationService());
        historyBuilder.newFields(TransferRequestHistoryFields.builder()
                .version(0L)
                .summary(summary)
                .description(preValidatedRequest.getDescription().orElse(null))
                .type(TransferRequestType.RESERVE_TRANSFER)
                .status(confirmation.isApply()
                        ? TransferRequestStatus.APPLIED : TransferRequestStatus.PENDING)
                .build());
        historyBuilder.newParameters(transferParameters);
        historyBuilder.newVotes(transferVotes);
        historyBuilder.newResponsible(responsible);
        historyBuilder.newApplicationDetails(confirmation.isApply() ? transferApplicationDetails : null);
        historyBuilder.order(0L);

        transferRequestBuilder.id(transferRequestId);
        transferRequestBuilder.tenantId(Tenants.DEFAULT_TENANT_ID);
        transferRequestBuilder.version(0L);
        transferRequestBuilder.summary(summary);
        transferRequestBuilder.description(preValidatedRequest.getDescription().orElse(null));
        transferRequestBuilder.type(TransferRequestType.RESERVE_TRANSFER);
        transferRequestBuilder.subtype(TransferRequestSubtype.DEFAULT_RESERVE_TRANSFER);
        transferRequestBuilder.status(confirmation.isApply()
                ? TransferRequestStatus.APPLIED : TransferRequestStatus.PENDING);
        transferRequestBuilder.createdBy(confirmation.getUser().getId());
        transferRequestBuilder.updatedBy(null);
        transferRequestBuilder.createdAt(now);
        transferRequestBuilder.updatedAt(null);
        transferRequestBuilder.appliedAt(confirmation.isApply() ? now : null);
        transferRequestBuilder.parameters(transferParameters);
        transferRequestBuilder.responsible(responsible);
        transferRequestBuilder.votes(transferVotes);
        transferRequestBuilder.applicationDetails(confirmation.isApply() ? transferApplicationDetails : null);
        transferRequestBuilder.nextHistoryOrder(1L);
        transferRequestBuilder.transferNotified(TransferNotified.from(responsibleAndNotified));

        validatedBuilder.transferRequest(transferRequestBuilder.build());
        validatedBuilder.history(historyBuilder.build());

        addFolderAndServiceIndex(confirmation, destinationFolderId,
                destinationServiceId, now, validatedBuilder, transferRequestId);
        addFolderAndServiceIndex(confirmation, reserveFolderId,
                reserveServiceId, now, validatedBuilder, transferRequestId);

        addResponsibleIndex(confirmation, responsible, now, validatedBuilder, transferRequestId);
        prepareNotifiedUsersCreate(confirmation, validatedBuilder, responsibleAndNotified);
        return validatedBuilder.build();
    }

    static QuotaTransfer getQuotaTransfer(
            String folderId, long serviceId, List<ValidatedQuotaResourceTransfer> transfers, boolean plus) {
        QuotaTransfer.Builder quotaTransferBuilder = QuotaTransfer.builder();
        quotaTransferBuilder.folderId(folderId);
        quotaTransferBuilder.serviceId(serviceId);
        transfers.forEach(resourceTransfer -> {
            ResourceQuotaTransfer.Builder resourceQuotaTransferBuilder = ResourceQuotaTransfer.builder();
            resourceQuotaTransferBuilder.resourceId(resourceTransfer.getResource().getId());
            resourceQuotaTransferBuilder.delta(getDeltaWithSign(resourceTransfer.getDelta(), plus));
            quotaTransferBuilder.addTransfer(resourceQuotaTransferBuilder.build());
        });
        return quotaTransferBuilder.build();
    }

    private static long getDeltaWithSign(long l, boolean plus) {
        return plus ? l : -l;
    }

    private TransferVote prepareProviderAutoConfirmationVote(Instant now,
                                                             UserModel user,
                                                             ValidatedReserveTransferParameters validatedParameters) {
        TransferVote.Builder voteBuilder = TransferVote.builder();
        voteBuilder.userId(user.getId());
        voteBuilder.type(VoteType.CONFIRM);
        voteBuilder.timestamp(now);
        ValidatedReserveTransfer transfer = validatedParameters.getTransfer();
        voteBuilder.addFolderId(transfer.getDestinationFolder().getId());
        voteBuilder.addFolderId(transfer.getReserveFolder().getId());
        voteBuilder.addProviderIds(Set.of(transfer.getProvider().getId()));
        return voteBuilder.build();
    }

    private void addFolderOpLogId(String folderId, BiConsumer<String, String> preGeneratedFolderOpLogIdConsumer,
                                  TransferApplicationDetails.Builder applicationDetailsBuilder) {
        String opLogId = UUID.randomUUID().toString();
        preGeneratedFolderOpLogIdConsumer.accept(folderId, opLogId);
        applicationDetailsBuilder.addFolderOpLogId(folderId, opLogId);
    }

    private void addFolderAndServiceIndex(
            CreateTransferRequestConfirmationData confirmation, String folderId, long serviceId, Instant now,
            ValidatedCreateTransferRequest.Builder validatedBuilder, String transferRequestId) {
        validatedBuilder.addFolderIndex(TransferRequestByFolderModel.builder()
                .tenantId(Tenants.DEFAULT_TENANT_ID)
                .folderId(folderId)
                .status(confirmation.isApply()
                        ? TransferRequestStatus.APPLIED : TransferRequestStatus.PENDING)
                .createdAt(now)
                .transferRequestId(transferRequestId)
                .build());
        validatedBuilder.addServiceIndex(TransferRequestByServiceModel.builder()
                .tenantId(Tenants.DEFAULT_TENANT_ID)
                .serviceId(serviceId)
                .status(confirmation.isApply()
                        ? TransferRequestStatus.APPLIED : TransferRequestStatus.PENDING)
                .createdAt(now)
                .transferRequestId(transferRequestId)
                .build());
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Result<ValidatedPutTransferRequest>> validatePutQuotaTransfer(
            YdbTxSession txSession, FrontPutTransferRequestDto putTransferRequest, TransferRequestModel transferRequest,
            ResponsibleAndNotified responsible, YaUserDetails currentUser, Locale locale, boolean publicApi,
            boolean delayValidation
    ) {
        PreValidatedTransferRequestParameters.Builder preValidatedRequestBuilder
                = PreValidatedTransferRequestParameters.builder();
        ErrorCollection.Builder preErrorsBuilder = ErrorCollection.builder();
        validationService.validateCreateFields(putTransferRequest::getDescription,
                putTransferRequest::getParameters, preValidatedRequestBuilder, preErrorsBuilder, locale);
        if (putTransferRequest.getLoanParameters().isPresent()) {
            preErrorsBuilder.addError("loanParameters", TypedError.invalid(messages
                    .getMessage("errors.loan.is.not.supported", null, locale)));
        }
        if (putTransferRequest.getParameters().isPresent()) {
            PreValidatedReserveTransfer.Builder preValidatedReserveTransferBuilder =
                    PreValidatedReserveTransfer.builder();

            validationService.preValidateReserveTransferParameters(putTransferRequest.getParameters().get(),
                    preValidatedReserveTransferBuilder, preErrorsBuilder, locale, publicApi);

            if (preErrorsBuilder.hasAnyErrors()) {
                return Mono.just(Result.failure(preErrorsBuilder.build()));
            }

            PreValidatedReserveTransfer preValidatedReserveTransfer = preValidatedReserveTransferBuilder.build();
            PreValidatedTransferRequestParameters preValidatedRequest = preValidatedRequestBuilder.build();

            return validateReserveTransfer(txSession, preValidatedReserveTransfer, currentUser, locale, publicApi)
                    .flatMap(r -> r.andThenMono(parameters ->
                            validationService.validateReserveTransferApplication(txSession, parameters, locale,
                                            delayValidation)
                                    .flatMap(preApplicationR -> preApplicationR.andThenMono(preApplication ->
                                            validationService.validateResponsiblePut(responsible,
                                                    putTransferRequest::getAddConfirmation, currentUser, locale)
                                                    .applyMono(confirmation ->
                                                            indicesService.loadTransferRequestIndices(txSession,
                                                                    transferRequest)
                                                                    .map(indices ->
                                                                            preparePutReserveTransferRequest(
                                                                                    preValidatedRequest,
                                                                                    parameters, preApplication,
                                                                                    responsible, confirmation,
                                                                                    indices,
                                                                                    transferRequest)))))));
        } else {
            return Mono.just(Result.failure(preErrorsBuilder.build()));
        }


    }

    private ValidatedPutTransferRequest preparePutReserveTransferRequest(
            PreValidatedTransferRequestParameters preValidatedRequest,
            ValidatedReserveTransferParameters validatedParameters,
            QuotaTransferPreApplicationData preApplicationData,
            ResponsibleAndNotified responsibleAndNotified,
            PutTransferRequestConfirmationData confirmation,
            TransferRequestIndices transferRequestIndices,
            TransferRequestModel currentTransferRequest) {
        TransferResponsible responsible = responsibleAndNotified.getResponsible();
        Instant now = Instant.now();
        ValidatedPutTransferRequest.Builder validatedBuilder = ValidatedPutTransferRequest.builder();
        validatedBuilder.quotas(preApplicationData.getQuotas());
        validatedBuilder.folders(validatedParameters.getFolders());
        validatedBuilder.services(validatedParameters.getServices());
        validatedBuilder.resources(validatedParameters.getResources());
        validatedBuilder.unitsEnsembles(validatedParameters.getUnitsEnsembles());
        ValidatedReserveTransfer transfer = validatedParameters.getTransfer();
        validatedBuilder.providers(List.of(transfer.getProvider()));
        validatedBuilder.now(now);
        validatedBuilder.oldTransferRequest(currentTransferRequest);
        TransferRequestModel.Builder transferRequestBuilder = TransferRequestModel.builder(currentTransferRequest);
        TransferParameters.Builder parametersBuilder = TransferParameters.builder();

        String destinationFolderId = transfer.getDestinationFolder().getId();
        long destinationServiceId = transfer.getDestinationService().getId();
        String reserveFolderId = transfer.getReserveFolder().getId();
        long reserveServiceId = transfer.getReserveFolder().getServiceId();
        List<ValidatedQuotaResourceTransfer> transfers = transfer.getTransfers();
        parametersBuilder.addQuotaTransfer(getQuotaTransfer(destinationFolderId, destinationServiceId, transfers,
                true));
        parametersBuilder.addQuotaTransfer(getQuotaTransfer(reserveFolderId, reserveServiceId, transfers, false));

        TransferParameters updatedTransferParameters = parametersBuilder.build();
        boolean parametersUpdated = !updatedTransferParameters.equals(currentTransferRequest.getParameters());
        boolean responsibleUpdated = !responsible.equals(currentTransferRequest.getResponsible());
        TransferVotes.Builder votesBuilder = TransferVotes.builder();

        if (confirmation.isConfirm()) {
            TransferVote vote = prepareConfirmationVote(responsible, now, confirmation.getUser());
            votesBuilder.addVote(vote);
        }
        if (!parametersUpdated) {
            copyVotes(responsible, confirmation, currentTransferRequest, votesBuilder);
        }

        TransferVotes updatedTransferVotes = votesBuilder.build();
        boolean votesUpdated = !updatedTransferVotes.equals(currentTransferRequest.getVotes());
        boolean apply = updatedQuotaTransferRequestMustBeApplied(updatedTransferVotes, updatedTransferParameters);
        TransferRequestStatus targetStatus = apply
                ? TransferRequestStatus.APPLIED : TransferRequestStatus.PENDING;
        TransferApplicationDetails.Builder applicationDetailsBuilder = TransferApplicationDetails.builder();
        if (apply) {
            addFolderOpLogId(destinationFolderId, validatedBuilder::putPreGeneratedFolderOpLogId,
                    applicationDetailsBuilder);
            addFolderOpLogId(reserveFolderId, validatedBuilder::putPreGeneratedFolderOpLogId,
                    applicationDetailsBuilder);
        }

        TransferRequestHistoryModel.Builder historyBuilder = TransferRequestHistoryModel.builder();
        String transferRequestId = currentTransferRequest.getId();
        TransferApplicationDetails transferApplicationDetails = applicationDetailsBuilder.build();
        historyBuilder.id(UUID.randomUUID().toString());
        historyBuilder.tenantId(Tenants.DEFAULT_TENANT_ID);
        historyBuilder.transferRequestId(transferRequestId);
        historyBuilder.type(TransferRequestEventType.UPDATED);
        historyBuilder.timestamp(now);
        historyBuilder.authorId(confirmation.getUser().getId());
        TransferRequestHistoryFields.Builder oldFields = TransferRequestHistoryFields.builder()
                .version(currentTransferRequest.getVersion());
        TransferRequestHistoryFields.Builder newFields = TransferRequestHistoryFields.builder()
                .version(currentTransferRequest.getVersion() + 1L);
        if (!currentTransferRequest.getDescription().equals(preValidatedRequest.getDescription())) {
            oldFields.description(currentTransferRequest.getDescription().orElse(null));
            newFields.description(preValidatedRequest.getDescription().orElse(null));
        }
        if (!currentTransferRequest.getStatus().equals(targetStatus)) {
            oldFields.status(currentTransferRequest.getStatus());
            newFields.status(targetStatus);
        }
        oldFields.type(null);
        newFields.type(null);
        historyBuilder.oldFields(oldFields.build());
        historyBuilder.newFields(newFields.build());
        if (parametersUpdated) {
            historyBuilder.oldParameters(currentTransferRequest.getParameters());
            historyBuilder.newParameters(updatedTransferParameters);
        } else {
            historyBuilder.oldParameters(null);
            historyBuilder.newParameters(null);
        }
        if (responsibleUpdated) {
            historyBuilder.oldResponsible(currentTransferRequest.getResponsible());
            historyBuilder.newResponsible(responsible);
        } else {
            historyBuilder.oldResponsible(null);
            historyBuilder.newResponsible(null);
        }
        if (votesUpdated) {
            historyBuilder.oldVotes(currentTransferRequest.getVotes());
            historyBuilder.newVotes(updatedTransferVotes);
        } else {
            historyBuilder.oldVotes(null);
            historyBuilder.newVotes(null);
        }
        historyBuilder.oldApplicationDetails(null);
        if (apply) {
            historyBuilder.newApplicationDetails(transferApplicationDetails);
        } else {
            historyBuilder.newApplicationDetails(null);
        }
        historyBuilder.order(currentTransferRequest.getNextHistoryOrder());
        transferRequestBuilder.version(currentTransferRequest.getVersion() + 1L);
        transferRequestBuilder.description(preValidatedRequest.getDescription().orElse(null));
        transferRequestBuilder.status(targetStatus);
        transferRequestBuilder.updatedBy(confirmation.getUser().getId());
        transferRequestBuilder.updatedAt(now);
        transferRequestBuilder.appliedAt(apply ? now : null);
        transferRequestBuilder.parameters(updatedTransferParameters);
        transferRequestBuilder.responsible(responsible);
        transferRequestBuilder.votes(updatedTransferVotes);
        transferRequestBuilder.applicationDetails(apply ? transferApplicationDetails : null);
        transferRequestBuilder.nextHistoryOrder(currentTransferRequest.getNextHistoryOrder() + 1L);
        transferRequestBuilder.transferNotified(TransferNotified.from(responsibleAndNotified));
        if (!transferRequestBuilder.hasChanges(currentTransferRequest)) {
            validatedBuilder.apply(false);
            validatedBuilder.transferRequest(null);
            validatedBuilder.history(null);
            return validatedBuilder.build();
        }
        validatedBuilder.apply(apply);
        TransferRequestModel newTransferRequest = transferRequestBuilder.build();
        validatedBuilder.transferRequest(newTransferRequest);
        validatedBuilder.history(historyBuilder.build());

        TransferRequestIndices newTransferRequestIndices =
                TransferRequestIndicesService.calculateTransferRequestIndices(newTransferRequest);
        TransferRequestIndices.Difference difference =
                TransferRequestIndicesService.difference(transferRequestIndices, newTransferRequestIndices);
        validatedBuilder.indicesDifference(difference);

        prepareNotifiedUsersPut(confirmation, validatedBuilder, responsibleAndNotified, parametersUpdated, apply,
                currentTransferRequest);
        return validatedBuilder.build();
    }

    private String generateSummary(ProviderModel provider, FolderModel destinationFolder,
                                   ServiceMinimalModel destinationService) {
        return String.format("Выдача квоты из резерва %s в %s:%s",
                provider.getNameRu(), destinationService.getSlug(), destinationFolder.getDisplayName());
    }

}
