package ru.yandex.intranet.d.services.transfer;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.Sets;
import it.unimi.dsi.fastutil.longs.LongSet;
import it.unimi.dsi.fastutil.longs.LongSets;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountReserveType;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.transfers.FoldersResponsible;
import ru.yandex.intranet.d.model.transfers.ProviderResponsible;
import ru.yandex.intranet.d.model.transfers.ProvisionTransfer;
import ru.yandex.intranet.d.model.transfers.QuotaTransfer;
import ru.yandex.intranet.d.model.transfers.ReserveResponsibleModel;
import ru.yandex.intranet.d.model.transfers.ServiceResponsible;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferResponsible;
import ru.yandex.intranet.d.model.users.AbcServiceMemberModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.model.users.UserServiceRoles;
import ru.yandex.intranet.d.services.transfer.model.ResponsibleAndNotified;
import ru.yandex.intranet.d.services.transfer.model.ValidatedProvisionTransfer;
import ru.yandex.intranet.d.services.transfer.model.ValidatedProvisionTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.ValidatedQuotaTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.ValidatedReserveTransfer;
import ru.yandex.intranet.d.services.transfer.model.ValidatedReserveTransferParameters;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * TransferRequestResponsibleAndNotifyService.
 * Сервис для работы с ответственными и теми кого надо уведомить по TransferRequest.
 * Позволяет пересчитывать актуальных ответственных и необходимых к уведомлению за заявку.
 *
 * @author Petr Surkov <petrsurkov@yandex-team.ru>
 */
@Component
public class TransferRequestResponsibleAndNotifyService {
    private final TransferRequestStoreService storeService;
    private final long quotaManagerRoleId;
    private final long serviceProductHeadRoleId;
    private final long serviceProductDeputyHeadRoleId;
    private final long serviceResponsibleRoleId;
    private final long responsibleOfProvider;

    public TransferRequestResponsibleAndNotifyService(TransferRequestStoreService storeService,
                                                      @Value("${abc.roles.quotaManager}")
                                                              long quotaManagerRoleId,
                                                      @Value("${abc.roles.serviceProductHead}")
                                                              long serviceProductHeadRoleId,
                                                      @Value("${abc.roles.serviceProductDeputyHead}")
                                                              long serviceProductDeputyHeadRoleId,
                                                      @Value("${abc.roles.serviceResponsible}")
                                                              long serviceResponsibleRoleId,
                                                      @Value("${abc.roles.responsibleOfProvider}")
                                                              long responsibleOfProvider) {
        this.storeService = storeService;
        this.quotaManagerRoleId = quotaManagerRoleId;
        this.serviceProductHeadRoleId = serviceProductHeadRoleId;
        this.serviceProductDeputyHeadRoleId = serviceProductDeputyHeadRoleId;
        this.serviceResponsibleRoleId = serviceResponsibleRoleId;
        this.responsibleOfProvider = responsibleOfProvider;
    }

    /**
     * Вычисляет новых ответственных и кого следует уведомить по transferRequest
     */
    public Mono<ResponsibleAndNotified> calculateForTransferRequestModel(YdbTxSession txSession,
                                                                         TransferRequestModel transferRequest,
                                                                         boolean background) {
        switch (transferRequest.getType()) {
            case QUOTA_TRANSFER:
                return calculateForQuotaTransfer(txSession, collectFoldersByService(
                        transferRequest.getParameters().getQuotaTransfers().stream(),
                        QuotaTransfer::getDestinationServiceId,
                        QuotaTransfer::getDestinationFolderId
                ), background);
            case RESERVE_TRANSFER:
                ReserveResponsibleModel reserveResponsibleModel =
                        transferRequest.getResponsible().getReserveResponsibleModel().orElseThrow();
                List<QuotaTransfer> nonReservedDestinations = transferRequest.getParameters()
                        .getQuotaTransfers()
                        .stream()
                        .filter(quotaTransfer ->
                                !quotaTransfer.getDestinationFolderId().equals(
                                        reserveResponsibleModel.getFolderId()))
                        .toList();
                if (nonReservedDestinations.size() != 1) {
                    throw new UnsupportedOperationException(
                            "Unsupported transferRequest " + transferRequest +
                                    (nonReservedDestinations.isEmpty() ? " without " : "with many ") +
                                    "non-reserved destinations"
                    );
                }

                return calculateForReserveTransfer(
                        txSession,
                        nonReservedDestinations.get(0).getDestinationFolderId(),
                        nonReservedDestinations.get(0).getDestinationServiceId(),
                        reserveResponsibleModel.getProviderId(),
                        reserveResponsibleModel.getFolderId(),
                        reserveResponsibleModel.getServiceId(),
                        background
                );
            case PROVISION_TRANSFER:
                return calculateForProvisionTransfer(txSession, transferRequest, background);
            default:
                throw new IllegalStateException("Unexpected value: " + transferRequest.getType());
        }
    }

    private Mono<ResponsibleAndNotified> calculateForProvisionTransfer(
            YdbTxSession txSession,
            TransferRequestModel transferRequest,
            boolean background
    ) {
        //todo загружать ответственных в зависимости от подтипа
        Set<String> accountIds = transferRequest.getParameters().getProvisionTransfers().stream()
                .flatMap(pt -> Stream.of(pt.getSourceAccountId(), pt.getDestinationAccountId()))
                .collect(Collectors.toSet());
        return storeService.loadAccounts(txSession, accountIds, background).flatMap(accounts -> {
            Map<String, AccountModel> accountById = accounts.stream()
                    .collect(Collectors.toMap(AccountModel::getId, x -> x));
            boolean hasProviderReserveAsSource = transferRequest.getParameters()
                    .getProvisionTransfers()
                    .stream()
                    .anyMatch(pt -> accountById.get(pt.getSourceAccountId()).getReserveType()
                            .map(art -> art == AccountReserveType.PROVIDER).orElse(false));
            if (hasProviderReserveAsSource) {
                Set<String> providerIds = transferRequest.getParameters().getProvisionTransfers()
                        .stream()
                        .map(ProvisionTransfer::getProviderId)
                        .collect(Collectors.toSet());
                return storeService.loadProviders(txSession, providerIds, background)
                        .flatMap(providers -> {
                            Map<Long, List<ProviderModel>> providersByServiceId = providers.stream()
                                    .collect(Collectors.groupingBy(ProviderModel::getServiceId));
                            return calculateForReserveProvisionTransfer(txSession, providersByServiceId,
                                    false, background);
                        });
            } else {
                return calculateForQuotaTransfer(txSession,
                        collectFoldersByService(transferRequest.getParameters().getProvisionTransfers()),
                        background);
            }
        });
    }

    /**
     * Вычисляет ответственных и кого следует уведомить по QuotaTransfer
     * с учётом автоподтверждения от переданного юзера
     */
    public Mono<ResponsibleAndNotified> calculateForQuotaTransferCreate(
            YdbTxSession txSession, ValidatedQuotaTransferParameters parameters, YaUserDetails currentUser) {
        // See DISPENSER-3540
        UserModel user = currentUser.getUser().get();
        Set<String> transferRequestProviderIds = new HashSet<>();
        parameters.getTransfers().forEach(t -> t.getTransfers()
                .forEach(v -> transferRequestProviderIds.add(v.getResource().getProviderId())));
        Map<String, ProviderModel> providersById = parameters.getProviders().stream()
                .collect(Collectors.toMap(ProviderModel::getId, Function.identity()));
        Set<Long> transferRequestProviderServiceIds = new HashSet<>();
        transferRequestProviderIds.forEach(providerId -> transferRequestProviderServiceIds
                .add(providersById.get(providerId).getServiceId()));
        boolean autoConfirm = !transferRequestProviderServiceIds.isEmpty()
                && user.getRoles().getOrDefault(UserServiceRoles.RESPONSIBLE_OF_PROVIDER, Set.of())
                .containsAll(transferRequestProviderServiceIds);
        if (!autoConfirm) {
            return calculateForQuotaTransfer(txSession, parameters);
        }
        TransferResponsible.Builder builder = TransferResponsible.builder();
        builder.addProviderResponsible(ProviderResponsible.builder()
                .responsibleId(user.getId())
                .addProviderIds(transferRequestProviderIds)
                .build());
        return Mono.just(new ResponsibleAndNotified(builder.build(), Set.of(), true));
    }

    /**
     * Вычисляет ответственных и кого следует уведомить по ReserveTransfer
     * с учётом автоподтверждения от переданного юзера
     */
    public Mono<ResponsibleAndNotified> calculateForReserveTransferCreate(
            YdbTxSession txSession, ValidatedReserveTransferParameters parameters, YaUserDetails currentUser) {
        // See DISPENSER-3540
        UserModel user = currentUser.getUser().orElseThrow();
        ProviderModel provider = parameters.getTransfer().getProvider();
        long providerServiceId = provider.getServiceId();
        String providerId = provider.getId();
        boolean autoConfirm = user.getRoles().getOrDefault(UserServiceRoles.RESPONSIBLE_OF_PROVIDER, Set.of())
                .contains(providerServiceId);
        if (!autoConfirm) {
            return calculateForReserveTransfer(txSession, parameters);
        }
        TransferResponsible.Builder builder = TransferResponsible.builder();
        builder.addProviderResponsible(ProviderResponsible.builder()
                .responsibleId(user.getId())
                .addProviderId(providerId)
                .build());
        return Mono.just(new ResponsibleAndNotified(builder.build(), Set.of(), true));
    }

    public Mono<ResponsibleAndNotified> calculateForProvisionTransferCreate(
            YdbTxSession txSession, ValidatedProvisionTransferParameters parameters, YaUserDetails currentUser) {
        UserModel user = currentUser.getUser().orElseThrow();
        Set<String> providerIds = new HashSet<>();
        for (ValidatedProvisionTransfer provisionTransfer : parameters.getProvisionTransfers()) {
            providerIds.add(provisionTransfer.getSourceAccount().getProviderId());
        }
        Map<String, ProviderModel> providerById = parameters.getProviders()
                .stream()
                .collect(Collectors.toMap(ProviderModel::getId, Function.identity()));
        Set<Long> providerServiceIds = new HashSet<>();
        providerIds.forEach(p -> providerServiceIds.add(providerById.get(p).getServiceId()));
        boolean autoConfirm = !providerServiceIds.isEmpty()
                && user.getRoles().getOrDefault(UserServiceRoles.RESPONSIBLE_OF_PROVIDER, Set.of())
                .containsAll(providerServiceIds);
        if (!autoConfirm) {
            return calculateForProvisionTransfer(txSession, parameters);
        }
        TransferResponsible.Builder builder = TransferResponsible.builder();
        builder.addProviderResponsible(ProviderResponsible.builder()
                .responsibleId(user.getId())
                .addProviderIds(providerIds)
                .build());
        return Mono.just(new ResponsibleAndNotified(builder.build(), Set.of(), true));
    }

    public Mono<ResponsibleAndNotified> calculateForReserveProvisionTransfer(
            YdbTxSession txSession,
            ValidatedProvisionTransferParameters parameters,
            YaUserDetails currentUser
    ) {
        UserModel user = currentUser.getUser().orElseThrow();
        Set<String> providerIds = new HashSet<>();
        for (ValidatedProvisionTransfer provisionTransfer : parameters.getProvisionTransfers()) {
            providerIds.add(provisionTransfer.getSourceAccount().getProviderId());
        }
        Map<Long, List<ProviderModel>> providersByServiceId = parameters.getProviders().stream()
                .filter(p -> providerIds.contains(p.getId()))
                .collect(Collectors.groupingBy(ProviderModel::getServiceId));
        Set<Long> providerServiceIds = providersByServiceId.keySet();
        boolean autoConfirm = !providerServiceIds.isEmpty()
                && user.getRoles().getOrDefault(UserServiceRoles.RESPONSIBLE_OF_PROVIDER, Set.of())
                .containsAll(providerServiceIds);
        return calculateForReserveProvisionTransfer(txSession, providersByServiceId, autoConfirm, false);
    }

    @NotNull
    private Mono<ResponsibleAndNotified> calculateForReserveProvisionTransfer(
            YdbTxSession txSession,
            Map<Long, List<ProviderModel>> providersByServiceId,
            boolean autoConfirm,
            boolean background
    ) {
        return storeService.loadServiceRoles(txSession, providersByServiceId.keySet(), Set.of(responsibleOfProvider),
                        background)
                .flatMap(roles -> {
                    Set<Long> staffIds = roles.stream()
                            .map(AbcServiceMemberModel::getStaffId)
                            .collect(Collectors.toSet());
                    return storeService.loadUsersByStaffIds(txSession, staffIds, background).map(users -> {
                        Map<Long, UserModel> userByStaffId = users.stream()
                                .collect(Collectors.toMap(u -> u.getStaffId().orElseThrow(), x -> x));
                        Map<Long, List<AbcServiceMemberModel>> rolesByServiceId = roles.stream()
                                .collect(Collectors.groupingBy(AbcServiceMemberModel::getServiceId));
                        Map<String, Set<String>> providerIdsByUserId = new HashMap<>();
                        rolesByServiceId.forEach((serviceId, serviceRoles) -> {
                            Set<String> serviceProviderIds = providersByServiceId.getOrDefault(serviceId, List.of())
                                    .stream()
                                    .map(ProviderModel::getId)
                                    .collect(Collectors.toSet());
                            for (AbcServiceMemberModel serviceRole : serviceRoles) {
                                UserModel responsible = userByStaffId.get(serviceRole.getStaffId());
                                providerIdsByUserId.computeIfAbsent(responsible.getId(), x -> new HashSet<>())
                                        .addAll(serviceProviderIds);
                            }
                        });
                        TransferResponsible.Builder builder = TransferResponsible.builder();
                        providerIdsByUserId.forEach((userId, pids) -> builder.addProviderResponsible(
                                ProviderResponsible.builder()
                                        .responsibleId(userId)
                                        .addProviderIds(pids)
                                        .build())
                        );
                        return new ResponsibleAndNotified(builder.build(), Set.copyOf(users), autoConfirm);
                    });
                });
    }

    public Mono<ResponsibleAndNotified> calculateForProvisionTransfer(
            YdbTxSession txSession,
            ValidatedProvisionTransferParameters parameters
    ) {
        Map<Long, Set<String>> folderIdsByService = new HashMap<>();
        for (ValidatedProvisionTransfer provisionTransfer : parameters.getProvisionTransfers()) {
            folderIdsByService.computeIfAbsent(provisionTransfer.getSourceService().getId(), x -> new HashSet<>())
                    .add(provisionTransfer.getSourceFolder().getId());
            folderIdsByService.computeIfAbsent(provisionTransfer.getDestinationService().getId(), x -> new HashSet<>())
                    .add(provisionTransfer.getDestinationFolder().getId());
        }
        return calculateForQuotaTransfer(txSession, folderIdsByService, false);
    }

    /**
     * Вычисляет ответственных и кого следует уведомить по QuotaTransfer
     *
     * @see #calculateForQuotaTransferCreate(YdbTxSession, ValidatedQuotaTransferParameters, YaUserDetails)
     */
    public Mono<ResponsibleAndNotified> calculateForQuotaTransfer(YdbTxSession txSession,
                                                                  ValidatedQuotaTransferParameters parameters) {
        Map<Long, Set<String>> foldersByService = parameters.getTransfers().stream()
                .collect(Collectors.groupingBy(quotaTransfer -> quotaTransfer.getDestinationService().getId(),
                        Collectors.mapping(x -> x.getDestinationFolder().getId(), Collectors.toSet())
                ));
        return calculateForQuotaTransfer(txSession, foldersByService, false);
    }

    /**
     * Вычисляет ответственных и кого следует уведомить по ReserveTransfer
     *
     * @see #calculateForReserveTransferCreate(YdbTxSession, ValidatedReserveTransferParameters, YaUserDetails)
     */
    public Mono<ResponsibleAndNotified> calculateForReserveTransfer(YdbTxSession txSession,
                                                                    ValidatedReserveTransferParameters parameters) {
        ValidatedReserveTransfer transfer = parameters.getTransfer();
        return calculateForReserveTransfer(txSession,
                transfer.getDestinationFolder().getId(),
                transfer.getDestinationService().getId(),
                transfer.getProvider().getId(),
                transfer.getReserveFolder().getId(),
                transfer.getProvider().getServiceId(),
                false);
    }

    /**
     * Исключает из списка юзеров для уведомления тех, что уже были уведомлены.
     * Если это старый трансфер реквест и данные notify не сохраненны, то использует для исключения
     * список ответственных трансфер реквеста.
     */
    public static Set<UserModel> excludeAlreadyNotifiedUsers(TransferRequestModel oldTransferRequest,
                                                             Set<UserModel> notifiedUsers) {
        Set<String> forExclude;
        if (oldTransferRequest.getTransferNotified().isEmpty()) {
            // старый трансфер реквест, в котором ещё не записаны нотифайнутые юзеры
            forExclude = oldTransferRequest.getResponsible().getResponsible().stream()
                    .flatMap(r -> r.getResponsible().stream().flatMap(v -> v.getResponsibleIds().stream()))
                    .collect(Collectors.toSet());
            oldTransferRequest.getResponsible().getReserveResponsibleModel()
                    .map(ReserveResponsibleModel::getResponsibleIds).ifPresent(forExclude::addAll);
        } else {
            forExclude = oldTransferRequest.getTransferNotified().get().getNotifiedUserIds();
        }
        return notifiedUsers.stream()
                .filter(u -> !forExclude.contains(u.getId()))
                .collect(Collectors.toSet());
    }

    public static Set<UserModel> excludeAlreadyNotifiedUsers(TransferRequestModel oldTransferRequest,
                                                             Set<UserModel> notifiedUsers,
                                                             YaUserDetails currentUser) {
        Set<UserModel> userModels = excludeAlreadyNotifiedUsers(oldTransferRequest, notifiedUsers);
        return currentUser.getUser()
                .map(current -> userModels.stream()
                        .filter(u -> !u.equals(current))
                        .collect(Collectors.toSet()))
                .orElse(userModels);
    }

    public static boolean isResponsible(UserModel user, TransferResponsible responsible) {
        return getResponsibleIds(responsible).contains(user.getId());
    }

    public static Set<String> getResponsibleIds(TransferRequestModel transferRequestModel) {
        return getResponsibleIds(transferRequestModel.getResponsible());
    }

    public static Set<String> getResponsibleIds(TransferResponsible responsible) {
        Set<String> res = new HashSet<>();
        responsible.getResponsible()
                .stream()
                .flatMap(fr -> fr.getResponsible().stream().flatMap(sr -> sr.getResponsibleIds().stream()))
                .forEach(res::add);
        responsible.getProviderResponsible()
                .forEach(pr -> res.add(pr.getResponsibleId()));
        responsible.getReserveResponsibleModel()
                .ifPresent(rr -> res.addAll(rr.getResponsibleIds()));
        return res;
    }

    private Mono<ResponsibleAndNotified> calculateForQuotaTransfer(YdbTxSession txSession,
                                                                   Map<Long, Set<String>> foldersByService,
                                                                   boolean background) {
        Set<Long> serviceHeadsRoleIds = Set.of(serviceProductHeadRoleId, serviceProductDeputyHeadRoleId,
                serviceResponsibleRoleId);
        Set<Long> quotaManagerRoleIds = Set.of(quotaManagerRoleId);
        Set<Long> allMatchingRoleIds = Sets.union(serviceHeadsRoleIds, quotaManagerRoleIds);
        return storeService.loadServicesParents(txSession,
                new ArrayList<>(foldersByService.keySet()), background).flatMap(parents -> {
            Set<Long> serviceIdsToLoadRoles = new HashSet<>(foldersByService.keySet());
            parents.values().forEach(serviceIdsToLoadRoles::addAll);
            return storeService.loadServiceRoles(txSession, serviceIdsToLoadRoles, allMatchingRoleIds, background)
                    .flatMap(roles -> {
                        Set<Long> allStaffIds = new HashSet<>();
                        roles.forEach(role -> allStaffIds.add(role.getStaffId()));
                        return storeService.loadUsersByStaffIds(txSession, allStaffIds, background).map(users -> {
                            Map<Long, UserModel> usersByStaffId = users.stream()
                                    .collect(Collectors.toMap(u -> u.getStaffId().get(), Function.identity()));
                            Map<Long, Set<UserModel>> matchingUsersByService = new HashMap<>();
                            Map<Long, Set<UserModel>> quotaManagersByService = new HashMap<>();
                            Map<Long, Set<UserModel>> serviceHeadsByService = new HashMap<>();
                            roles.forEach(role -> {
                                UserModel user = usersByStaffId.get(role.getStaffId());
                                if (user != null) {
                                    matchingUsersByService.computeIfAbsent(role.getServiceId(),
                                            k -> new HashSet<>()).add(user);
                                    if (quotaManagerRoleIds.contains(role.getRoleId())) {
                                        quotaManagersByService.computeIfAbsent(role.getServiceId(),
                                                k -> new HashSet<>()).add(user);
                                    }
                                    if (serviceHeadsRoleIds.contains(role.getRoleId())) {
                                        serviceHeadsByService.computeIfAbsent(role.getServiceId(),
                                                k -> new HashSet<>()).add(user);
                                    }
                                }
                            });
                            return calculateForQuotaTransfer(foldersByService, parents,
                                    matchingUsersByService,
                                    quotaManagersByService, serviceHeadsByService);
                        });
                    });
        });
    }

    private ResponsibleAndNotified calculateForQuotaTransfer(Map<Long, Set<String>> foldersByService,
                                                             Map<Long, LongSet> parents,
                                                             Map<Long, Set<UserModel>> matchingUsersByService,
                                                             Map<Long, Set<UserModel>> quotaManagersByService,
                                                             Map<Long, Set<UserModel>> serviceHeadsByService) {
        TransferResponsible.Builder builder = TransferResponsible.builder();
        foldersByService.forEach((serviceId, folders) -> {
            Set<Long> serviceParents = parents.getOrDefault(serviceId, LongSets.EMPTY_SET);
            FoldersResponsible.Builder foldersResponsibleBuilder = FoldersResponsible.builder();
            folders.forEach(foldersResponsibleBuilder::addFolderId);
            foldersResponsibleBuilder.addResponsible(ServiceResponsible.builder()
                    .serviceId(serviceId)
                    .addResponsibleIds(matchingUsersByService.getOrDefault(serviceId, Collections.emptySet())
                            .stream().map(UserModel::getId).collect(Collectors.toList()))
                    .build());
            serviceParents.forEach(serviceParentId -> {
                foldersResponsibleBuilder.addResponsible(ServiceResponsible.builder()
                        .serviceId(serviceParentId)
                        .addResponsibleIds(matchingUsersByService.getOrDefault(serviceParentId, Collections.emptySet())
                                .stream().map(UserModel::getId).collect(Collectors.toList()))
                        .build());
            });
            builder.addResponsibles(foldersResponsibleBuilder.build());
        });
        Set<UserModel> notified = calculateNotifiedForQuotaTransfer(foldersByService, quotaManagersByService,
                serviceHeadsByService);
        return new ResponsibleAndNotified(builder.build(), notified, false);
    }

    private Set<UserModel> calculateNotifiedForQuotaTransfer(Map<Long, Set<String>> foldersByService,
                                                             Map<Long, Set<UserModel>> quotaManagersByService,
                                                             Map<Long, Set<UserModel>> serviceHeadsByService) {
        Map<String, UserModel> notified = new HashMap<>();
        foldersByService.keySet().forEach(serviceId -> {
            Set<UserModel> quotaManagers = quotaManagersByService.getOrDefault(serviceId, Collections.emptySet());
            if (!quotaManagers.isEmpty()) {
                quotaManagers.forEach(u -> notified.put(u.getId(), u));
            } else {
                Set<UserModel> serviceHeads = serviceHeadsByService.getOrDefault(serviceId, Collections.emptySet());
                serviceHeads.forEach(u -> notified.put(u.getId(), u));
            }
        });
        return new HashSet<>(notified.values());
    }

    @SuppressWarnings("ParameterNumber")
    private Mono<ResponsibleAndNotified> calculateForReserveTransfer(YdbTxSession txSession,
                                                                     String destinationFolderId,
                                                                     long destinationServiceId,
                                                                     String providerId,
                                                                     String reserveFolderId,
                                                                     long providerServiceId,
                                                                     boolean background) {
        Set<Long> serviceHeadsRoleIds = Set.of(serviceProductHeadRoleId, serviceProductDeputyHeadRoleId,
                serviceResponsibleRoleId);
        Set<Long> quotaManagerRoleIds = Set.of(quotaManagerRoleId);
        Set<Long> responsibleOfProviderIds = Set.of(responsibleOfProvider);
        Set<Long> allMatchingRoleIds = Sets.union(Sets.union(serviceHeadsRoleIds, quotaManagerRoleIds),
                responsibleOfProviderIds);
        return storeService.loadServiceParents(txSession, destinationServiceId, background).flatMap(parents -> {
            Set<Long> serviceIdsToLoadRoles = new HashSet<>(List.of(destinationServiceId, providerServiceId));
            serviceIdsToLoadRoles.addAll(parents);
            return storeService.loadServiceRoles(txSession, serviceIdsToLoadRoles, allMatchingRoleIds, background)
                    .flatMap(roles -> {
                        Set<Long> allStaffIds = new HashSet<>();
                        roles.forEach(role -> allStaffIds.add(role.getStaffId()));
                        return storeService.loadUsersByStaffIds(txSession, allStaffIds, background).map(users -> {
                            Map<Long, UserModel> usersByStaffId = users.stream()
                                    .collect(Collectors.toMap(u -> u.getStaffId().orElseThrow(), Function.identity()));
                            Map<Long, Set<UserModel>> matchingUsersByService = new HashMap<>();
                            Set<UserModel> providerResponsibleSet = new HashSet<>();
                            Map<Long, Set<UserModel>> quotaManagersByService = new HashMap<>();
                            Map<Long, Set<UserModel>> serviceHeadsByService = new HashMap<>();
                            roles.forEach(role -> {
                                UserModel user = usersByStaffId.get(role.getStaffId());
                                if (user != null) {
                                    if (responsibleOfProviderIds.contains(role.getRoleId())) {
                                        if (role.getServiceId() == providerServiceId) {
                                            providerResponsibleSet.add(user);
                                        }
                                    } else {
                                        matchingUsersByService.computeIfAbsent(role.getServiceId(),
                                                k -> new HashSet<>()).add(user);
                                        if (quotaManagerRoleIds.contains(role.getRoleId())) {
                                            quotaManagersByService.computeIfAbsent(role.getServiceId(),
                                                    k -> new HashSet<>()).add(user);
                                        }
                                        if (serviceHeadsRoleIds.contains(role.getRoleId())) {
                                            serviceHeadsByService.computeIfAbsent(role.getServiceId(),
                                                    k -> new HashSet<>()).add(user);
                                        }
                                    }
                                }
                            });
                            return calculateForReserveTransfer(destinationFolderId, destinationServiceId,
                                    providerId, reserveFolderId, providerServiceId, parents, matchingUsersByService,
                                    quotaManagersByService, serviceHeadsByService, providerResponsibleSet);
                        });
                    });
        });
    }

    @SuppressWarnings("checkstyle:ParameterNumber")
    private ResponsibleAndNotified calculateForReserveTransfer(String destinationFolderId,
                                                               long destinationServiceId,
                                                               String providerId,
                                                               String reserveFolderId,
                                                               long providerServiceId,
                                                               Set<Long> parents,
                                                               Map<Long, Set<UserModel>> matchingUsersByService,
                                                               Map<Long, Set<UserModel>> quotaManagersByService,
                                                               Map<Long, Set<UserModel>> serviceHeadsByService,
                                                               Set<UserModel> providerResponsibleSet) {
        TransferResponsible.Builder builder = TransferResponsible.builder();
        FoldersResponsible.Builder foldersResponsibleBuilder = FoldersResponsible.builder();
        foldersResponsibleBuilder.addFolderId(destinationFolderId)
                .addResponsible(ServiceResponsible.builder()
                        .serviceId(destinationServiceId)
                        .addResponsibleIds(matchingUsersByService.getOrDefault(destinationServiceId,
                                Collections.emptySet())
                                .stream().map(UserModel::getId).collect(Collectors.toList()))
                        .build());
        parents.forEach(serviceParentId -> foldersResponsibleBuilder.addResponsible(ServiceResponsible.builder()
                .serviceId(serviceParentId)
                .addResponsibleIds(matchingUsersByService.getOrDefault(serviceParentId, Collections.emptySet())
                        .stream().map(UserModel::getId).collect(Collectors.toList()))
                .build()));
        builder.addResponsibles(foldersResponsibleBuilder.build())
                .reserveResponsibleModel(ReserveResponsibleModel.builder()
                        .addResponsibleIds(providerResponsibleSet
                                .stream().map(UserModel::getId).collect(Collectors.toList()))
                        .providerId(providerId)
                        .folderId(reserveFolderId)
                        .serviceId(providerServiceId)
                        .build());
        Set<UserModel> notified = calculateNotifiedForReserveTransfer(destinationServiceId, quotaManagersByService,
                serviceHeadsByService,
                providerResponsibleSet);
        return new ResponsibleAndNotified(builder.build(), notified, false);
    }

    private Set<UserModel> calculateNotifiedForReserveTransfer(long destinationServiceId,
                                                               Map<Long, Set<UserModel>> quotaManagersByService,
                                                               Map<Long, Set<UserModel>> serviceHeadsByService,
                                                               Set<UserModel> providerResponsibleSet) {
        Map<String, UserModel> notified = new HashMap<>();
        providerResponsibleSet.forEach(userModel -> notified.put(userModel.getId(), userModel));
        Set<UserModel> quotaManagers = quotaManagersByService.getOrDefault(destinationServiceId,
                Collections.emptySet());
        if (!quotaManagers.isEmpty()) {
            quotaManagers.forEach(u -> notified.put(u.getId(), u));
        } else {
            Set<UserModel> serviceHeads = serviceHeadsByService.getOrDefault(destinationServiceId,
                    Collections.emptySet());
            serviceHeads.forEach(u -> notified.put(u.getId(), u));
        }
        return new HashSet<>(notified.values());
    }

    private static <T> Map<Long, Set<String>> collectFoldersByService(Stream<T> transfers,
                                                                      Function<T, Long> serviceId,
                                                                      Function<T, String> folderId) {
        return transfers.collect(
                Collectors.groupingBy(serviceId, Collectors.mapping(folderId, Collectors.toSet()))
        );
    }

    private static Map<Long, Set<String>> collectFoldersByService(Set<ProvisionTransfer> transfers) {
        Map<Long, Set<String>> result = new HashMap<>();
        transfers.forEach(t -> {
            result.computeIfAbsent(t.getSourceServiceId(), x -> new HashSet<>()).add(t.getSourceFolderId());
            result.computeIfAbsent(t.getDestinationServiceId(), x -> new HashSet<>()).add(t.getDestinationFolderId());
        });
        return result;
    }
}
