package ru.yandex.intranet.d.services.transfer;

import java.util.HashSet;
import java.util.Locale;
import java.util.Set;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.transfer.model.ValidatedProvisionTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.ValidatedQuotaTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.ValidatedReserveTransferParameters;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static ru.yandex.intranet.d.util.result.TypedError.forbidden;

/**
 * Transfer request security service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class TransferRequestSecurityService {

    private final SecurityManagerService securityManagerService;
    private final MessageSource messages;

    public TransferRequestSecurityService(SecurityManagerService securityManagerService,
                                          @Qualifier("messageSource") MessageSource messages) {
        this.securityManagerService = securityManagerService;
        this.messages = messages;
    }

    public Mono<Result<TransferRequestModel>> checkReadPermissions(TransferRequestModel transferRequest,
                                                                   YaUserDetails currentUser, Locale locale) {
        // Permissions are checked using denormalized data from request itself
        // so this check matches request search indices
        // It is assumed that either service permission or folder permission
        // is sufficient for transfer request to be accessible
        Set<Long> serviceIds = new HashSet<>();
        Set<String> folderIds = new HashSet<>();
        transferRequest.getParameters().getFolderTransfers().forEach(t -> {
            serviceIds.add(t.getDestinationServiceId());
            serviceIds.add(t.getSourceServiceId());
            folderIds.add(t.getFolderId());
        });
        transferRequest.getParameters().getQuotaTransfers().forEach(t -> {
            folderIds.add(t.getDestinationFolderId());
            serviceIds.add(t.getDestinationServiceId());
        });
        transferRequest.getParameters().getAccountTransfers().forEach(t -> {
            folderIds.add(t.getSourceFolderId());
            folderIds.add(t.getDestinationFolderId());
            serviceIds.add(t.getSourceServiceId());
            serviceIds.add(t.getDestinationServiceId());
        });
        transferRequest.getParameters().getProvisionTransfers().forEach(t -> {
            folderIds.add(t.getSourceFolderId());
            folderIds.add(t.getDestinationFolderId());
            serviceIds.add(t.getSourceServiceId());
            serviceIds.add(t.getDestinationServiceId());
        });
        if (serviceIds.isEmpty() && folderIds.isEmpty()) {
            return Mono.just(Result.failure(ErrorCollection.builder()
                    .addError(forbidden(messages.getMessage("errors.access.denied", null, locale)))
                    .build()));
        }
        Flux<Result<Void>> destinationServicePermissionChecks = Flux.empty();
        if (!serviceIds.isEmpty()) {
            destinationServicePermissionChecks = Flux.fromIterable(serviceIds)
                    .flatMap(serviceId -> securityManagerService.checkReadPermissions(serviceId.intValue(),
                            currentUser, locale), 3);
        }
        Flux<Result<Void>> folderPermissionChecks = Flux.empty();
        if (!folderIds.isEmpty()) {
            folderPermissionChecks = Flux.fromIterable(folderIds)
                    .flatMap(folderId -> securityManagerService
                            .checkReadPermissions(folderId, currentUser, locale, null), 3);
        }
        return Flux.concat(destinationServicePermissionChecks, folderPermissionChecks).any(Result::isSuccess)
                .flatMap(success -> {
                    if (success) {
                        return Mono.just(Result.success(transferRequest));
                    } else {
                        return Mono.just(Result.failure(ErrorCollection.builder()
                                .addError(forbidden(messages.getMessage("errors.access.denied", null, locale)))
                                .build()));
                    }
                });
    }

    public Mono<Result<ValidatedQuotaTransferParameters>> checkQuotaTransferReadPermissions(
            ValidatedQuotaTransferParameters validatedParameters, YaUserDetails currentUser, Locale locale) {
        Set<FolderModel> folders = new HashSet<>();
        validatedParameters.getTransfers().forEach(t -> folders.add(t.getDestinationFolder()));
        Flux<Result<Void>> folderPermissionChecks = Flux.fromIterable(folders)
                .flatMap(folder -> securityManagerService
                        .checkReadPermissions(folder.getId(), currentUser, locale, null), 3);
        return folderPermissionChecks.any(Result::isSuccess)
                .flatMap(success -> {
                    if (success) {
                        return Mono.just(Result.success(validatedParameters));
                    } else {
                        return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError
                                .forbidden(messages.getMessage("errors.access.denied", null, locale)))
                                .build()));
                    }
                });
    }

    public Mono<Result<ValidatedReserveTransferParameters>> checkReserveTransferReadPermissions(
            ValidatedReserveTransferParameters validatedParameters, YaUserDetails currentUser, Locale locale) {
        FolderModel destinationFolder = validatedParameters.getTransfer().getDestinationFolder();
        return securityManagerService.checkReadPermissions(destinationFolder.getId(), currentUser, locale, null)
                .flatMap(result -> result.isSuccess()
                        ? Mono.just(Result.success(validatedParameters))
                        : Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError
                        .forbidden(messages.getMessage("errors.access.denied", null, locale)))
                        .build()))
                );
    }

    public Mono<Result<ValidatedProvisionTransferParameters>> checkProvisionTransferReadPermissions(
            ValidatedProvisionTransferParameters validatedParameters, YaUserDetails currentUser, Locale locale) {
        return Flux.fromIterable(validatedParameters.getProvisionTransfers())
                .flatMap(pt -> Flux.just(pt.getSourceFolder(), pt.getDestinationFolder()))
                .flatMap(f -> securityManagerService.checkReadPermissions(f, currentUser, locale, validatedParameters))
                .all(Result::isSuccess)
                .map(success -> {
                    if (success) {
                        return Result.success(validatedParameters);
                    } else {
                        return Result.failure(ErrorCollection.builder().addError(TypedError
                                        .forbidden(messages.getMessage("errors.access.denied", null, locale)))
                                .build());
                    }
                });
    }

}
