package ru.yandex.intranet.d.services.transfer;

import java.time.Instant;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.transfers.TransferRequestByFolderModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByResponsibleModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByServiceModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestEventType;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryFields;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestStatus;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.model.transfers.TransferResponsible;
import ru.yandex.intranet.d.model.transfers.VoteType;
import ru.yandex.intranet.d.services.quotas.MoveProvisionApplicationContext;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.transfer.model.ExpandedTransferRequests;
import ru.yandex.intranet.d.services.transfer.model.RefreshTransferRequest;
import ru.yandex.intranet.d.services.transfer.model.ResponsibleAndNotified;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestCreationResult;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestProviderData;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestRefreshResult;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestsSearchParameters;
import ru.yandex.intranet.d.services.transfer.model.ValidatedCreateTransferRequest;
import ru.yandex.intranet.d.services.transfer.model.ValidatedPutTransferRequest;
import ru.yandex.intranet.d.services.transfer.model.ValidatedVoteTransferRequest;
import ru.yandex.intranet.d.services.transfer.ticket.TransferRequestTicketService;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontPutTransferRequestDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestVotingDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

import static ru.yandex.intranet.d.services.transfer.TransferRequestPermissionService.canUserPutRequest;
import static ru.yandex.intranet.d.util.result.TypedError.forbidden;

/**
 * Transfer request service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class TransferRequestService {

    private static final Set<TransferRequestStatus> REQUEST_FINAL_STATUSES = EnumSet.of(
            TransferRequestStatus.APPLIED,
            TransferRequestStatus.PARTLY_APPLIED,
            TransferRequestStatus.FAILED,
            TransferRequestStatus.CANCELLED,
            TransferRequestStatus.REJECTED
    );

    private static final Logger LOG = LoggerFactory.getLogger(TransferRequestService.class);

    private final SecurityManagerService securityManagerService;
    private final MessageSource messages;
    private final TransferRequestStoreService storeService;
    private final TransferRequestValidationService validationService;
    private final TransferRequestSecurityService securityService;
    private final TransferRequestAnswerService answerService;
    private final TransferRequestQuotaService quotaService;
    private final TransferRequestReserveService reserveService;
    private final TransferRequestNotificationsService transferRequestNotificationsService;
    private final YdbTableClient tableClient;
    private final RefreshTransferRequestService refreshTransferRequestService;
    private final TransferRequestTicketService transferRequestTicketService;
    private final TransferRequestProvisionService provisionService;

    @SuppressWarnings("ParameterNumber")
    public TransferRequestService(SecurityManagerService securityManagerService,
                                  @Qualifier("messageSource") MessageSource messages,
                                  TransferRequestStoreService storeService,
                                  TransferRequestValidationService validationService,
                                  TransferRequestSecurityService securityService,
                                  TransferRequestAnswerService answerService,
                                  TransferRequestQuotaService quotaService,
                                  TransferRequestReserveService reserveService,
                                  TransferRequestNotificationsService transferRequestNotificationsService,
                                  YdbTableClient tableClient,
                                  RefreshTransferRequestService refreshTransferRequestService,
                                  TransferRequestTicketService transferRequestTicketService,
                                  TransferRequestProvisionService provisionService) {
        this.securityManagerService = securityManagerService;
        this.messages = messages;
        this.storeService = storeService;
        this.validationService = validationService;
        this.securityService = securityService;
        this.answerService = answerService;
        this.quotaService = quotaService;
        this.reserveService = reserveService;
        this.transferRequestNotificationsService = transferRequestNotificationsService;
        this.tableClient = tableClient;
        this.refreshTransferRequestService = refreshTransferRequestService;
        this.transferRequestTicketService = transferRequestTicketService;
        this.provisionService = provisionService;
    }

    public Mono<Result<ExpandedTransferRequests<TransferRequestModel>>> getOne(String id, YaUserDetails currentUser,
                                                                               Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale)
                .flatMap(ra -> ra.andThen(vb -> validationService.validateId(id, locale)).andThenMono(vc ->
                        tableClient.usingSessionMonoRetryable(session ->
                                storeService.getById(immediateTx(session), id)
                        .map(requestO -> validationService.validateExists(requestO.orElse(null), locale))
                        .flatMap(requestR -> requestR.andThenMono(request ->
                                securityService.checkReadPermissions(request, currentUser, locale)))
                        .flatMap(requestR -> requestR.applyMono(request -> answerService.expand(session, request))))));
    }

    public Mono<Result<ExpandedTransferRequests<Page<TransferRequestModel>>>> search(
            TransferRequestsSearchParameters searchParameters,
            PageRequest pageRequest,
            YaUserDetails currentUser,
            Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(ra -> ra.andThenMono(vb ->
                validationService.validatePageRequest(pageRequest, locale).andThenMono(validatedPage ->
                        tableClient.usingSessionMonoRetryable(session -> validationService.validateSearchParameters(
                                session, searchParameters, currentUser, locale).flatMap(validatedParamsR ->
                                        validatedParamsR.applyMono(validatedParams ->
                                                storeService.searchTransferRequests(immediateTx(session),
                                                        validatedParams, validatedPage, currentUser)
                                                        .flatMap(requests -> answerService.expandTransferRequests(
                                                                session, validatedPage, requests))))))));
    }

    public <T> Mono<Result<T>> closeIfNeeded(TransferRequestModel transferRequestModel, T returnData) {
        TransferRequestStatus status = transferRequestModel.getStatus();
        if (REQUEST_FINAL_STATUSES.contains(status)) {
            return transferRequestTicketService.closeTicket(transferRequestModel, returnData);
        } else {
            return Mono.just(Result.success(returnData));
        }
    }

    public <T> Mono<Result<T>> completeRefresh(YdbSession session,
                                                TransferRequestRefreshResult<Result<T>> refreshResult,
                                                boolean updateTicketOnSuccess) {
        if (refreshResult.isRefreshed()) {
            Mono<Result<T>> resultMono = Mono.just(refreshResult.getData());
            if (updateTicketOnSuccess || refreshResult.getData().isFailure()) {
                resultMono = updateTicketAfterRefresh(session, refreshResult);
            }
            return resultMono
                    .doOnNext(r -> transferRequestNotificationsService.sendNotifications(
                            refreshResult.getNotifiedUsers(), refreshResult.getRequest().getId()));
        }
        return Mono.just(refreshResult.getData());
    }

    @NotNull
    private <T> Mono<Result<T>> updateTicketAfterRefresh(YdbSession session,
                                                         TransferRequestRefreshResult<Result<T>> refreshResult) {
        return answerService.expand(session, refreshResult.getRequest())
                .flatMap(er -> transferRequestTicketService
                        .updateTicket(er, refreshResult.getData())
                        .map(ur -> ur.match(Tuple2::getT2, e -> {
                            LOG.error("Failed to refresh tracker issue for transfer request {}: {}",
                                    refreshResult.getRequest().getId(), e);
                            return refreshResult.getData();
                        }))
                );
    }

    @NotNull
    public <T> Mono<TransferRequestRefreshResult<Result<T>>> toRefreshResult(
            Result<T> innerResult,
            TransferRequestCreationResult tr,
            TransferRequestModel oldRequest) {
        return Mono.just(new TransferRequestRefreshResult<>(innerResult, tr.getRequest(),
                tr.getNotifiedUsers(), !tr.getRequest().getResponsible().equals(oldRequest.getResponsible())));
    }

    @NotNull
    public <T> Mono<TransferRequestRefreshResult<Result<T>>> toCancelRefreshResult(
            Result<T> innerResult,
            TransferRequestModel tr) {
        return Mono.just(new TransferRequestRefreshResult<>(innerResult, tr, Set.of(), false));
    }


    @NotNull
    public <T> Mono<TransferRequestRefreshResult<Result<T>>> refreshRequest(
            YdbTxSession txSession,
            Result<T> failedResult,
            TransferRequestModel request,
            ResponsibleAndNotified responsible,
            YaUserDetails currentUser) {
        return refreshTransferRequestService.refreshTransferRequest(txSession, request, responsible)
                .map(refreshO ->
                        new TransferRequestRefreshResult<>(failedResult,
                                refreshO.map(RefreshTransferRequest::getRefreshedTransferRequest).orElse(request),
                                refreshO.map(RefreshTransferRequest::getNotifiedUsers)
                                        .map(users -> TransferRequestResponsibleAndNotifyService
                                                .excludeAlreadyNotifiedUsers(request, users, currentUser))
                                        .orElse(Set.of()),
                                refreshO.isPresent()));
    }

    public Mono<Result<ExpandedTransferRequests<Page<TransferRequestHistoryModel>>>> history(
            String transferRequestId,
            PageRequest pageRequest,
            YaUserDetails currentUser,
            Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(ra -> ra.andThen(vb ->
                validationService.validateId(transferRequestId, locale)).andThenMono(vc ->
                        validationService.validateHistoryPageRequest(pageRequest, locale).andThenMono(validatedPage ->
                                tableClient.usingSessionMonoRetryable(session -> storeService
                                        .getById(immediateTx(session), transferRequestId)
                                        .map(requestO -> validationService
                                                .validateExists(requestO.orElse(null), locale))
                                        .flatMap(requestR -> requestR.andThenMono(request -> securityService
                                                .checkReadPermissions(request, currentUser, locale)))
                                        .flatMap(requestR -> requestR.applyMono(request -> storeService
                                                        .searchTransferRequestsHistory(immediateTx(session),
                                                                request, validatedPage)
                                                        .flatMap(events -> answerService.expandHistory(session,
                                                                validatedPage, events))))))));
    }

    private YdbTxSession immediateTx(YdbSession session) {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY);
    }

    public Mono<TransferRequestModel> cancelRequest(YdbTxSession txSession, TransferRequestModel request,
                                                     ResponsibleAndNotified responsibleAndNotified,
                                                     YaUserDetails currentUser, String unique) {
        Instant now = Instant.now();
        TransferRequestModel.Builder builder = TransferRequestModel.builder(request);
        builder.version(request.getVersion() + 1L);
        builder.updatedAt(now);
        builder.updatedBy(currentUser.getUser().orElseThrow().getId());
        builder.nextHistoryOrder(request.getNextHistoryOrder() + 1L);
        builder.status(TransferRequestStatus.CANCELLED);
        final TransferResponsible updatedResponsible = responsibleAndNotified.getResponsible();
        builder.responsible(updatedResponsible);
        TransferRequestModel updatedTransferRequest = builder.build();
        TransferRequestHistoryModel.Builder historyBuilder = TransferRequestHistoryModel.builder();
        historyBuilder.id(UUID.randomUUID().toString());
        historyBuilder.tenantId(Tenants.DEFAULT_TENANT_ID);
        historyBuilder.transferRequestId(request.getId());
        historyBuilder.type(TransferRequestEventType.CANCELLED);
        historyBuilder.timestamp(now);
        historyBuilder.authorId(currentUser.getUser().get().getId());
        historyBuilder.oldFields(TransferRequestHistoryFields.builder()
                .version(request.getVersion())
                .status(request.getStatus())
                .build());
        historyBuilder.newFields(TransferRequestHistoryFields.builder()
                .version(request.getVersion() + 1L)
                .status(TransferRequestStatus.CANCELLED)
                .build());
        historyBuilder.order(request.getNextHistoryOrder());
        if (!request.getResponsible().equals(updatedResponsible)) {
            historyBuilder.newResponsible(updatedResponsible);
            historyBuilder.oldResponsible(request.getResponsible());
        }
        TransferRequestHistoryModel newHistory = historyBuilder.build();
        final List<TransferRequestByResponsibleModel> updatedByResponsibleIndices = List.copyOf(
                TransferRequestIndicesService.calculateResponsibleIndices(updatedTransferRequest, updatedResponsible));
        return storeService.loadByResponsibleIndex(txSession, request).flatMap(byResponsibleIndices ->
                storeService.loadByFolderIndex(txSession, request).flatMap(byFolderIndices ->
                storeService.loadByServiceIndex(txSession, request).flatMap(byServiceIndices -> {
                    List<TransferRequestByResponsibleModel> newByResponsibleIndices = updateByResponsibleIndexStatus(
                            updatedByResponsibleIndices, TransferRequestStatus.CANCELLED);
                    List<TransferRequestByFolderModel> newByFolderIndices = updateByFolderIndexStatus(
                            byFolderIndices, TransferRequestStatus.CANCELLED);
                    List<TransferRequestByServiceModel> newByServiceIndices = updateByServiceIndexStatus(
                            byServiceIndices, TransferRequestStatus.CANCELLED);
                    return storeService.saveCancellation(txSession, updatedTransferRequest, newHistory,
                            byResponsibleIndices, byFolderIndices, byServiceIndices, newByResponsibleIndices,
                            newByFolderIndices, newByServiceIndices, unique, currentUser);
                })));
    }

    private List<TransferRequestByResponsibleModel> updateByResponsibleIndexStatus(
            List<TransferRequestByResponsibleModel> indices, TransferRequestStatus status) {
        return indices.stream().map(v -> TransferRequestByResponsibleModel.builder(v).status(status).build())
                .collect(Collectors.toList());
    }

    private List<TransferRequestByFolderModel> updateByFolderIndexStatus(
            List<TransferRequestByFolderModel> indices, TransferRequestStatus status) {
        return indices.stream().map(v -> TransferRequestByFolderModel.builder(v).status(status).build())
                .collect(Collectors.toList());
    }

    private List<TransferRequestByServiceModel> updateByServiceIndexStatus(
            List<TransferRequestByServiceModel> indices, TransferRequestStatus status) {
        return indices.stream().map(v -> TransferRequestByServiceModel.builder(v).status(status).build())
                .collect(Collectors.toList());
    }

    public ValidatedCreateTransferRequest validateCreateApplicable(ValidatedCreateTransferRequest transferRequest) {
        return switch (transferRequest.getTransferRequest().getType()) {
            case QUOTA_TRANSFER, RESERVE_TRANSFER -> quotaService.validateCreateApplicable(transferRequest);
            case PROVISION_TRANSFER -> provisionService.validateCreateApplicable(transferRequest);
            case ACCOUNT_TRANSFER, FOLDER_TRANSFER -> transferRequest;
        };
    }

    public ValidatedPutTransferRequest validatePutApplicable(ValidatedPutTransferRequest transferRequest) {
        return switch (transferRequest.getTransferRequest().orElseThrow().getType()) {
            case QUOTA_TRANSFER, RESERVE_TRANSFER -> quotaService.validatePutApplicable(transferRequest);
            case PROVISION_TRANSFER -> provisionService.validatePutApplicable(transferRequest);
            case ACCOUNT_TRANSFER, FOLDER_TRANSFER -> transferRequest;
        };
    }

    public Mono<Result<ValidatedCreateTransferRequest>> validateCreate(
            YdbTxSession txSession,
            FrontCreateTransferRequestDto transferRequest,
            TransferRequestProviderData transferRequestProviderData,
            YaUserDetails currentUser,
            Locale locale,
            boolean publicApi,
            boolean delayValidation
    ) {
        if (currentUser.getUser().isEmpty()) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.forbidden(messages
                    .getMessage("errors.access.denied", null, locale)))
                    .build()));
        }
        Result<TransferRequestType> typeR = validationService
                .validateTransferRequestType(transferRequest::getRequestType, locale);
        return typeR.andThenMono(type -> switch (type) {
            case FOLDER_TRANSFER -> validateFolderTransfer(txSession, transferRequest, currentUser, locale);
            case QUOTA_TRANSFER -> quotaService.validateQuotaTransfer(txSession, transferRequest, currentUser,
                    locale, publicApi, delayValidation);
            case ACCOUNT_TRANSFER -> validateAccountTransfer(txSession, transferRequest, currentUser, locale);
            case PROVISION_TRANSFER -> provisionService.validateProvisionTransferMono(txSession, transferRequest,
                    transferRequestProviderData, currentUser, locale, publicApi, delayValidation);
            case RESERVE_TRANSFER -> reserveService.validateReserveTransfer(txSession, transferRequest,
                    currentUser, locale, publicApi, delayValidation);
        });
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Result<ValidatedPutTransferRequest>> validatePut(YdbTxSession txSession,
                                                                  FrontPutTransferRequestDto putTransferRequest,
                                                                  TransferRequestModel transferRequest,
                                                                  ResponsibleAndNotified responsibleAndNotified,
                                                                  Long version,
                                                                  YaUserDetails currentUser,
                                                                  Locale locale,
                                                                  boolean publicApi,
                                                                  boolean delayValidation
    ) {
        Result<Void> versionR = validationService.validateVersion(transferRequest, version, locale);
        if (versionR.isFailure()) {
            return Mono.just(versionR.cast(ValidatedPutTransferRequest.class));
        }
        if (!canUserPutRequest(transferRequest, responsibleAndNotified, currentUser)) {
            return Mono.just(Result.failure(ErrorCollection.builder()
                    .addError(forbidden(messages.getMessage("errors.access.denied", null, locale)))
                    .build()));
        }
        if (!TransferRequestStatus.PENDING.equals(transferRequest.getStatus())) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                    .getMessage("errors.only.pending.transfer.request.may.be.updated", null, locale))).build()));
        }
        return switch (transferRequest.getType()) {
            case FOLDER_TRANSFER -> validatePutFolderTransfer(txSession, putTransferRequest, transferRequest,
                    responsibleAndNotified, currentUser, locale);
            case QUOTA_TRANSFER -> quotaService.validatePutQuotaTransfer(txSession, putTransferRequest, transferRequest,
                    responsibleAndNotified, currentUser, locale, publicApi, delayValidation);
            case ACCOUNT_TRANSFER -> validatePutAccountTransfer(txSession, putTransferRequest, transferRequest,
                    responsibleAndNotified, currentUser, locale);
            case PROVISION_TRANSFER -> provisionService.validatePutProvisionTransferMono(txSession, putTransferRequest,
                    transferRequest, currentUser, locale, publicApi, delayValidation);
            case RESERVE_TRANSFER -> reserveService.validatePutQuotaTransfer(txSession, putTransferRequest,
                    transferRequest, responsibleAndNotified, currentUser, locale, publicApi, delayValidation);
        };
    }

    public Mono<Result<ValidatedVoteTransferRequest>> validateVote(YdbTxSession txSession,
                                                                    FrontTransferRequestVotingDto voteParameters,
                                                                    TransferRequestModel transferRequest,
                                                                    ResponsibleAndNotified reloadedResponsible,
                                                                    Long version,
                                                                    YaUserDetails currentUser,
                                                                    Locale locale) {
        Result<Void> versionR = validationService.validateVersion(transferRequest, version, locale);
        if (versionR.isFailure()) {
            return Mono.just(versionR.cast(ValidatedVoteTransferRequest.class));
        }
        if (currentUser.getUser().isEmpty()) {
            return Mono.just(Result.failure(ErrorCollection.builder()
                    .addError(forbidden(messages.getMessage("errors.access.denied", null, locale)))
                    .build()));
        }
        if (!TransferRequestStatus.PENDING.equals(transferRequest.getStatus())) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                    .getMessage("errors.only.pending.transfer.request.may.be.voted.for", null, locale))).build()));
        }
        Result<VoteType> voteTypeR = validationService.validateVoteParameters(voteParameters, locale);
        return voteTypeR.andThenMono(voteType -> switch (transferRequest.getType()) {
            case FOLDER_TRANSFER -> validateVoteFolderTransfer(txSession, voteParameters, transferRequest,
                    reloadedResponsible, currentUser, locale);
            case QUOTA_TRANSFER, RESERVE_TRANSFER -> quotaService.validateVoteQuotaTransfer(txSession, voteType,
                    transferRequest, reloadedResponsible, currentUser, locale);
            case ACCOUNT_TRANSFER -> validateVoteAccountTransfer(txSession, voteParameters, transferRequest,
                    reloadedResponsible, currentUser, locale);
            case PROVISION_TRANSFER -> provisionService.validateVoteProvisionTransferMono(txSession, voteType,
                    transferRequest, reloadedResponsible, currentUser, locale);
        });
    }

    private Mono<Result<ValidatedCreateTransferRequest>> validateFolderTransfer(
            YdbTxSession txSession,
            FrontCreateTransferRequestDto transferRequest,
            YaUserDetails currentUser,
            Locale locale) {
        // TODO Support folders transfer
        return Mono.just(Result.failure(ErrorCollection.builder().addError("requestType", TypedError.invalid(messages
                .getMessage("errors.transfer.request.type.not.supported", null, locale))).build()));
    }

    private Mono<Result<ValidatedCreateTransferRequest>> validateAccountTransfer(
            YdbTxSession txSession,
            FrontCreateTransferRequestDto transferRequest,
            YaUserDetails currentUser,
            Locale locale) {
        // TODO Support account transfer
        return Mono.just(Result.failure(ErrorCollection.builder().addError("requestType", TypedError.invalid(messages
                .getMessage("errors.transfer.request.type.not.supported", null, locale))).build()));
    }

    private Mono<Result<ValidatedPutTransferRequest>> validatePutFolderTransfer(
            YdbTxSession txSession, FrontPutTransferRequestDto putTransferRequest,
            TransferRequestModel transferRequest, ResponsibleAndNotified reloadedResponsible,
            YaUserDetails currentUser, Locale locale) {
        // TODO Support folders transfer
        return Mono.just(Result.failure(ErrorCollection.builder().addError("requestType", TypedError.invalid(messages
                .getMessage("errors.transfer.request.type.not.supported", null, locale))).build()));
    }

    private Mono<Result<ValidatedPutTransferRequest>> validatePutAccountTransfer(
            YdbTxSession txSession, FrontPutTransferRequestDto putTransferRequest,
            TransferRequestModel transferRequest, ResponsibleAndNotified reloadedResponsible,
            YaUserDetails currentUser, Locale locale) {
        // TODO Support account transfer
        return Mono.just(Result.failure(ErrorCollection.builder().addError("requestType", TypedError.invalid(messages
                .getMessage("errors.transfer.request.type.not.supported", null, locale))).build()));
    }

    private Mono<Result<ValidatedVoteTransferRequest>> validateVoteFolderTransfer(
            YdbTxSession txSession, FrontTransferRequestVotingDto voteParameters,
            TransferRequestModel transferRequest, ResponsibleAndNotified reloadedResponsible,
            YaUserDetails currentUser, Locale locale) {
        // TODO Support folders transfer
        return Mono.just(Result.failure(ErrorCollection.builder().addError("requestType", TypedError.invalid(messages
                .getMessage("errors.transfer.request.type.not.supported", null, locale))).build()));
    }

    private Mono<Result<ValidatedVoteTransferRequest>> validateVoteAccountTransfer(
            YdbTxSession txSession, FrontTransferRequestVotingDto voteParameters,
            TransferRequestModel transferRequest, ResponsibleAndNotified reloadedResponsible,
            YaUserDetails currentUser, Locale locale) {
        // TODO Support account transfer
        return Mono.just(Result.failure(ErrorCollection.builder().addError("requestType", TypedError.invalid(messages
                .getMessage("errors.transfer.request.type.not.supported", null, locale))).build()));
    }

    public Mono<TransferRequestCreationResult> saveCreate(YdbTxSession txSession,
                                                           ValidatedCreateTransferRequest createRequest,
                                                           YaUserDetails currentUser,
                                                           String unique) {
        return preprocessCreate(txSession, createRequest, currentUser)
                .flatMap(updatedCreateRequest -> storeService.saveCreate(txSession, updatedCreateRequest, unique,
                        currentUser)
                        .then(Mono.defer(() -> applyCreate(txSession, updatedCreateRequest, currentUser))));
    }

    public Mono<TransferRequestCreationResult> savePut(YdbTxSession txSession,
                                                        ValidatedPutTransferRequest putRequest,
                                                        YaUserDetails currentUser,
                                                        String unique) {
        if (putRequest.getTransferRequest().isPresent() && putRequest.getHistory().isPresent()) {
            return preprocessPut(txSession, putRequest, currentUser)
                    .flatMap(updatedPutRequest -> storeService.savePut(txSession, updatedPutRequest, unique,
                                    currentUser)
                            .then(Mono.defer(() -> applyPut(txSession, updatedPutRequest, currentUser))));
        } else {
            return applyPut(txSession, putRequest, currentUser);
        }
    }

    public Mono<TransferRequestCreationResult> saveVote(YdbTxSession txSession,
                                                         ValidatedVoteTransferRequest voteRequest,
                                                         YaUserDetails currentUser,
                                                         String unique) {
        return preprocessVote(txSession, voteRequest, currentUser)
                .flatMap(updatedVoteRequest -> storeService.saveVote(txSession, updatedVoteRequest, unique,
                                currentUser)
                        .then(Mono.defer(() -> applyVote(txSession, updatedVoteRequest, currentUser))));
    }

    private Mono<ValidatedCreateTransferRequest> preprocessCreate(YdbTxSession txSession,
                                                                  ValidatedCreateTransferRequest createRequest,
                                                                  YaUserDetails currentUser) {
        TransferRequestModel transferRequest = createRequest.getTransferRequest();
        switch (transferRequest.getType()) {
            case PROVISION_TRANSFER:
                if (createRequest.isApply()) {
                    MoveProvisionApplicationContext applicationContext = new MoveProvisionApplicationContext(
                            createRequest.getProviders(), createRequest.getResources(), createRequest.getAccounts(),
                            createRequest.getFolders(), createRequest.getQuotas(), createRequest.getAccountsQuotas());
                    return provisionService.applyProvisionRequestMono(txSession, transferRequest,
                                    createRequest.getHistory(), applicationContext, currentUser)
                            .map(applyResult -> createRequest.copyBuilder()
                                    .transferRequest(applyResult.getUpdatedTransferRequest())
                                    .history(applyResult.getUpdatedTransferHistory())
                                    .build());
                } else {
                    return Mono.just(createRequest);
                }
            case QUOTA_TRANSFER:
            case FOLDER_TRANSFER:
            case ACCOUNT_TRANSFER:
            case RESERVE_TRANSFER:
                return Mono.just(createRequest);
            default:
                throw new IllegalArgumentException("Unexpected transfer request type: " + transferRequest.getType());
        }
    }

    private Mono<ValidatedPutTransferRequest> preprocessPut(YdbTxSession txSession,
                                                            ValidatedPutTransferRequest putTransfer,
                                                            YaUserDetails currentUser) {
        TransferRequestModel transferRequest = putTransfer.getTransferRequest().orElseThrow();
        switch (transferRequest.getType()) {
            case PROVISION_TRANSFER:
                if (putTransfer.isApply()) {
                    MoveProvisionApplicationContext applicationContext = new MoveProvisionApplicationContext(
                            putTransfer.getProviders(), putTransfer.getResources(), putTransfer.getAccounts(),
                            putTransfer.getFolders(), putTransfer.getQuotas(), putTransfer.getAccountsQuotas());
                    return provisionService.applyProvisionRequestMono(txSession, transferRequest,
                                    putTransfer.getHistory().orElseThrow(), applicationContext, currentUser)
                            .map(applyResult -> putTransfer.copyBuilder()
                                    .transferRequest(applyResult.getUpdatedTransferRequest())
                                    .history(applyResult.getUpdatedTransferHistory())
                                    .build());
                } else {
                    return Mono.just(putTransfer);
                }
            case QUOTA_TRANSFER:
            case FOLDER_TRANSFER:
            case ACCOUNT_TRANSFER:
            case RESERVE_TRANSFER:
                return Mono.just(putTransfer);
            default:
                throw new IllegalArgumentException("Unexpected transfer request type: " + transferRequest.getType());
        }
    }

    private Mono<ValidatedVoteTransferRequest> preprocessVote(YdbTxSession txSession,
                                                              ValidatedVoteTransferRequest voteRequest,
                                                              YaUserDetails currentUser) {
        TransferRequestModel transferRequest = voteRequest.getTransferRequest();
        switch (transferRequest.getType()) {
            case PROVISION_TRANSFER:
                if (voteRequest.isApply()) {
                    MoveProvisionApplicationContext applicationContext = new MoveProvisionApplicationContext(
                            voteRequest.getProviders(), voteRequest.getResources(), voteRequest.getAccounts(),
                            voteRequest.getFolders(), voteRequest.getQuotas(), voteRequest.getAccountsQuotas());
                    return provisionService.applyProvisionRequestMono(txSession, transferRequest,
                                    voteRequest.getHistory(), applicationContext, currentUser)
                            .map(applyResult -> voteRequest.copyBuilder()
                                    .transferRequest(applyResult.getUpdatedTransferRequest())
                                    .history(applyResult.getUpdatedTransferHistory())
                                    .build());
                } else {
                    return Mono.just(voteRequest);
                }
            case QUOTA_TRANSFER:
            case FOLDER_TRANSFER:
            case ACCOUNT_TRANSFER:
            case RESERVE_TRANSFER:
                return Mono.just(voteRequest);
            default:
                throw new IllegalArgumentException("Unexpected transfer request type: " + transferRequest.getType());
        }
    }

    private Mono<TransferRequestCreationResult> applyCreate(YdbTxSession txSession,
                                                            ValidatedCreateTransferRequest createRequest,
                                                            YaUserDetails currentUser) {
        TransferRequestCreationResult result = TransferRequestCreationResult.builder()
                .request(createRequest.getTransferRequest())
                .folders(createRequest.getFolders())
                .services(createRequest.getServices())
                .resources(createRequest.getResources())
                .unitsEnsembles(createRequest.getUnitsEnsembles())
                .providers(createRequest.getProviders())
                .notifiedUsers(createRequest.getNotifiedUsers())
                .build();
        if (createRequest.isApply()) {
            return applyTransferRequestCreate(txSession, createRequest, currentUser).thenReturn(result);
        }
        return Mono.just(result);
    }

    private Mono<TransferRequestCreationResult> applyPut(YdbTxSession txSession,
                                                         ValidatedPutTransferRequest putRequest,
                                                         YaUserDetails currentUser) {
        TransferRequestCreationResult result = TransferRequestCreationResult.builder()
                .request(putRequest.getTransferRequest()
                        .orElseGet(() -> putRequest.getOldTransferRequest().orElseThrow()))
                .folders(putRequest.getFolders())
                .services(putRequest.getServices())
                .resources(putRequest.getResources())
                .unitsEnsembles(putRequest.getUnitsEnsembles())
                .providers(putRequest.getProviders())
                .notifiedUsers(putRequest.getNewNotifiedUsers())
                .build();
        if (putRequest.isApply()) {
            return applyTransferRequestPut(txSession, putRequest, currentUser).thenReturn(result);
        }
        return Mono.just(result);
    }

    private Mono<TransferRequestCreationResult> applyVote(YdbTxSession txSession,
                                                          ValidatedVoteTransferRequest voteRequest,
                                                          YaUserDetails currentUser) {
        TransferRequestCreationResult result = TransferRequestCreationResult.builder()
                .request(voteRequest.getTransferRequest())
                .folders(voteRequest.getFolders())
                .services(voteRequest.getServices())
                .resources(voteRequest.getResources())
                .unitsEnsembles(voteRequest.getUnitsEnsembles())
                .providers(voteRequest.getProviders())
                .notifiedUsers(voteRequest.getNotifiedUsers())
                .build();
        if (voteRequest.isApply()) {
            return applyTransferRequestVote(txSession, voteRequest, currentUser).thenReturn(result);
        }
        return Mono.just(result);
    }

    private Mono<Void> applyTransferRequestCreate(YdbTxSession txSession, ValidatedCreateTransferRequest createRequest,
                                                  YaUserDetails currentUser) {
        return switch (createRequest.getTransferRequest().getType()) {
            case FOLDER_TRANSFER ->
                    // TODO Support folders transfer
                    Mono.empty();
            case QUOTA_TRANSFER, RESERVE_TRANSFER -> quotaService.applyQuotaTransferRequest(txSession,
                    createRequest.getQuotas(), createRequest.getTransferRequest(), currentUser.getUser().orElseThrow(),
                    createRequest.getPreGeneratedFolderOpLogIdsByFolderId(), createRequest.getResources(),
                    createRequest.getFolders(), createRequest.getNow());
            case ACCOUNT_TRANSFER ->
                    // TODO Support account transfer
                    Mono.empty();
            case PROVISION_TRANSFER -> Mono.empty();
        };
    }

    private Mono<Void> applyTransferRequestPut(YdbTxSession txSession, ValidatedPutTransferRequest putRequest,
                                               YaUserDetails currentUser) {
        TransferRequestModel transferRequest = putRequest.getTransferRequest().orElseThrow();
        return switch (transferRequest.getType()) {
            case FOLDER_TRANSFER ->
                    // TODO Support folders transfer
                    Mono.empty();
            case QUOTA_TRANSFER, RESERVE_TRANSFER -> quotaService.applyQuotaTransferRequest(txSession,
                    putRequest.getQuotas(), transferRequest, currentUser.getUser().orElseThrow(),
                    putRequest.getPreGeneratedFolderOpLogIdsByFolderId(), putRequest.getResources(),
                    putRequest.getFolders(), putRequest.getNow());
            case ACCOUNT_TRANSFER ->
                    // TODO Support account transfer
                    Mono.empty();
            case PROVISION_TRANSFER -> Mono.empty();
        };
    }

    private Mono<Void> applyTransferRequestVote(YdbTxSession txSession, ValidatedVoteTransferRequest voteRequest,
                                                YaUserDetails currentUser) {
        TransferRequestModel transferRequest = voteRequest.getTransferRequest();
        return switch (transferRequest.getType()) {
            case FOLDER_TRANSFER ->
                    // TODO Support folders transfer
                    Mono.empty();
            case QUOTA_TRANSFER, RESERVE_TRANSFER -> quotaService.applyQuotaTransferRequest(txSession,
                    voteRequest.getQuotas(), transferRequest, currentUser.getUser().orElseThrow(),
                    voteRequest.getPreGeneratedFolderOpLogIdsByFolderId(), voteRequest.getResources(),
                    voteRequest.getFolders(), voteRequest.getNow());
            case ACCOUNT_TRANSFER ->
                    // TODO Support account transfer
                    Mono.empty();
            case PROVISION_TRANSFER -> Mono.empty();
        };
    }

    public Mono<TransferRequestCreationResult> deferredCreate(TransferRequestCreationResult creationResult,
                                                              Locale locale) {
        TransferRequestType type = creationResult.getRequest().getType();
        return switch (type) {
            case FOLDER_TRANSFER, QUOTA_TRANSFER, RESERVE_TRANSFER, ACCOUNT_TRANSFER -> Mono.just(creationResult);
            case PROVISION_TRANSFER -> provisionService.startOperationIfExistsMono(creationResult.getRequest())
                    .thenReturn(creationResult);
        };
    }

    public Mono<TransferRequestCreationResult> deferredPut(TransferRequestCreationResult creationResult,
                                                           Locale locale) {
        TransferRequestType type = creationResult.getRequest().getType();
        return switch (type) {
            case FOLDER_TRANSFER, QUOTA_TRANSFER, RESERVE_TRANSFER, ACCOUNT_TRANSFER -> Mono.just(creationResult);
            case PROVISION_TRANSFER -> provisionService.startOperationIfExistsMono(creationResult.getRequest())
                    .thenReturn(creationResult);
        };
    }

    public Mono<TransferRequestCreationResult> deferredVote(TransferRequestCreationResult voteResult,
                                                            Locale locale) {
        TransferRequestType type = voteResult.getRequest().getType();
        return switch (type) {
            case FOLDER_TRANSFER, QUOTA_TRANSFER, RESERVE_TRANSFER, ACCOUNT_TRANSFER -> Mono.just(voteResult);
            case PROVISION_TRANSFER -> provisionService.startOperationIfExistsMono(voteResult.getRequest())
                    .thenReturn(voteResult);
        };
    }
}
