package ru.yandex.intranet.d.services.transfer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.yandex.ydb.table.transaction.TransactionMode;
import it.unimi.dsi.fastutil.longs.LongSet;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.dao.transfers.PendingTransferRequestsDao;
import ru.yandex.intranet.d.dao.transfers.TransferRequestByFolderDao;
import ru.yandex.intranet.d.dao.transfers.TransferRequestByResponsibleDao;
import ru.yandex.intranet.d.dao.transfers.TransferRequestByServiceDao;
import ru.yandex.intranet.d.dao.transfers.TransferRequestHistoryDao;
import ru.yandex.intranet.d.dao.transfers.TransferRequestsDao;
import ru.yandex.intranet.d.dao.users.AbcServiceMemberDao;
import ru.yandex.intranet.d.dao.users.UsersDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader;
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader;
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader;
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader;
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.transfers.PendingTransferRequestsModel;
import ru.yandex.intranet.d.model.transfers.ReserveResponsibleModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByFolderModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByResponsibleModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByServiceModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestStatus;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.AbcServiceMemberModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.transfer.model.RefreshTransferRequest;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestIndices;
import ru.yandex.intranet.d.services.transfer.model.ValidatedCreateTransferRequest;
import ru.yandex.intranet.d.services.transfer.model.ValidatedPutTransferRequest;
import ru.yandex.intranet.d.services.transfer.model.ValidatedTransferRequestContinuationToken;
import ru.yandex.intranet.d.services.transfer.model.ValidatedTransferRequestHistoryContinuationToken;
import ru.yandex.intranet.d.services.transfer.model.ValidatedTransferRequestsSearchParameters;
import ru.yandex.intranet.d.services.transfer.model.ValidatedVoteTransferRequest;
import ru.yandex.intranet.d.services.uniques.RequestUniqueService;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Transfer request store service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class TransferRequestStoreService {

    private final TransferRequestsDao transferRequestsDao;
    private final AccountsDao accountsDao;
    private final UsersDao usersDao;
    private final FolderDao folderDao;
    private final ServicesDao servicesDao;
    private final TransferRequestHistoryDao transferRequestHistoryDao;
    private final TransferRequestByResponsibleDao transferRequestByResponsibleDao;
    private final TransferRequestByFolderDao transferRequestByFolderDao;
    private final TransferRequestByServiceDao transferRequestByServiceDao;
    private final QuotasDao quotasDao;
    private final AbcServiceMemberDao abcServiceMemberDao;
    private final FolderOperationLogDao folderOperationLogDao;
    private final ResourcesLoader resourcesLoader;
    private final UnitsEnsemblesLoader unitsEnsemblesLoader;
    private final ProvidersLoader providersLoader;
    private final PendingTransferRequestsDao pendingTransferRequestsDao;
    private final RequestUniqueService requestUniqueService;
    private final AccountsQuotasDao accountsQuotasDao;
    private final YdbTableClient ydbTableClient;
    private final AccountsSpacesDao accountsSpacesDao;
    private final ResourceSegmentationsLoader resourceSegmentationsLoader;
    private final ResourceSegmentsLoader resourceSegmentsLoader;
    private final ResourceTypesLoader resourceTypesLoader;

    @SuppressWarnings("ParameterNumber")
    public TransferRequestStoreService(TransferRequestsDao transferRequestsDao,
                                       AccountsDao accountsDao,
                                       UsersDao usersDao,
                                       FolderDao folderDao,
                                       ServicesDao servicesDao,
                                       TransferRequestHistoryDao transferRequestHistoryDao,
                                       TransferRequestByResponsibleDao transferRequestByResponsibleDao,
                                       TransferRequestByFolderDao transferRequestByFolderDao,
                                       TransferRequestByServiceDao transferRequestByServiceDao,
                                       QuotasDao quotasDao,
                                       AbcServiceMemberDao abcServiceMemberDao,
                                       FolderOperationLogDao folderOperationLogDao,
                                       ResourcesLoader resourcesLoader,
                                       UnitsEnsemblesLoader unitsEnsemblesLoader,
                                       ProvidersLoader providersLoader,
                                       PendingTransferRequestsDao pendingTransferRequestsDao,
                                       RequestUniqueService requestUniqueService,
                                       AccountsQuotasDao accountsQuotasDao,
                                       YdbTableClient ydbTableClient,
                                       AccountsSpacesDao accountsSpacesDao,
                                       ResourceSegmentationsLoader resourceSegmentationsLoader,
                                       ResourceSegmentsLoader resourceSegmentsLoader,
                                       ResourceTypesLoader resourceTypesLoader) {
        this.transferRequestsDao = transferRequestsDao;
        this.accountsDao = accountsDao;
        this.usersDao = usersDao;
        this.folderDao = folderDao;
        this.servicesDao = servicesDao;
        this.transferRequestHistoryDao = transferRequestHistoryDao;
        this.transferRequestByResponsibleDao = transferRequestByResponsibleDao;
        this.transferRequestByFolderDao = transferRequestByFolderDao;
        this.transferRequestByServiceDao = transferRequestByServiceDao;
        this.quotasDao = quotasDao;
        this.abcServiceMemberDao = abcServiceMemberDao;
        this.folderOperationLogDao = folderOperationLogDao;
        this.resourcesLoader = resourcesLoader;
        this.unitsEnsemblesLoader = unitsEnsemblesLoader;
        this.providersLoader = providersLoader;
        this.pendingTransferRequestsDao = pendingTransferRequestsDao;
        this.requestUniqueService = requestUniqueService;
        this.accountsQuotasDao = accountsQuotasDao;
        this.ydbTableClient = ydbTableClient;
        this.accountsSpacesDao = accountsSpacesDao;
        this.resourceSegmentationsLoader = resourceSegmentationsLoader;
        this.resourceSegmentsLoader = resourceSegmentsLoader;
        this.resourceTypesLoader = resourceTypesLoader;
    }

    public Mono<Optional<TransferRequestModel>> getById(YdbTxSession session, String id) {
        return transferRequestsDao.getById(session, id, Tenants.DEFAULT_TENANT_ID);
    }

    public Mono<List<TransferRequestModel>> searchTransferRequests(
            YdbTxSession session,
            ValidatedTransferRequestsSearchParameters parameters,
            PageRequest.Validated<ValidatedTransferRequestContinuationToken> page,
            YaUserDetails currentUser) {
        if (parameters.getFilterByCurrentUser().isPresent() && parameters.getFilterByCurrentUser().get()) {
            if (page.getContinuationToken().isEmpty()) {
                return transferRequestsDao.findByResponsibleFirstPage(session,
                        currentUser.getUser().get().getId(), parameters.getFilterByStatus(), Tenants.DEFAULT_TENANT_ID,
                        page.getLimit() + 1);
            } else {
                return transferRequestsDao.findByResponsibleNextPage(session,
                        currentUser.getUser().get().getId(), parameters.getFilterByStatus(), Tenants.DEFAULT_TENANT_ID,
                        page.getContinuationToken().get().getCreatedAt(), page.getContinuationToken().get().getId(),
                        page.getLimit() + 1);
            }
        } else if (parameters.getFilterByFolder().isPresent()) {
            if (page.getContinuationToken().isEmpty()) {
                return transferRequestsDao.findByFolderFirstPage(session,
                        parameters.getFilterByFolder().get().getId(), parameters.getFilterByStatus(),
                        Tenants.DEFAULT_TENANT_ID, page.getLimit() + 1);
            } else {
                return transferRequestsDao.findByFolderNextPage(session,
                        parameters.getFilterByFolder().get().getId(), parameters.getFilterByStatus(),
                        Tenants.DEFAULT_TENANT_ID, page.getContinuationToken().get().getCreatedAt(),
                        page.getContinuationToken().get().getId(), page.getLimit() + 1);
            }
        } else {
            if (page.getContinuationToken().isEmpty()) {
                return transferRequestsDao.findByServiceFirstPage(session,
                        parameters.getFilterByService().get().getId(), parameters.getFilterByStatus(),
                        Tenants.DEFAULT_TENANT_ID, page.getLimit() + 1);
            } else {
                return transferRequestsDao.findByServiceNextPage(session,
                        parameters.getFilterByService().get().getId(), parameters.getFilterByStatus(),
                        Tenants.DEFAULT_TENANT_ID, page.getContinuationToken().get().getCreatedAt(),
                        page.getContinuationToken().get().getId(), page.getLimit() + 1);
            }
        }
    }

    public Mono<List<TransferRequestHistoryModel>> searchTransferRequestsHistory(
            YdbTxSession session,
            TransferRequestModel transferRequest,
            PageRequest.Validated<ValidatedTransferRequestHistoryContinuationToken> page) {
        if (page.getContinuationToken().isEmpty()) {
            return transferRequestHistoryDao.findFirstPage(session, transferRequest.getId(),
                    Tenants.DEFAULT_TENANT_ID, page.getLimit());
        } else {
            return transferRequestHistoryDao.findNextPage(session, transferRequest.getId(),
                    Tenants.DEFAULT_TENANT_ID, page.getContinuationToken().get().getTimestamp(),
                    page.getContinuationToken().get().getId(), page.getLimit());
        }
    }

    public Mono<Optional<FolderModel>> getFolder(YdbTxSession session, String id) {
        return folderDao.getById(session, id, Tenants.DEFAULT_TENANT_ID);
    }

    public Mono<List<AccountSpaceModel>> loadAccountSpaces(YdbSession session, List<String> ids) {
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> accountsSpacesDao.getByIds(immediateTx(session), ids, Tenants.DEFAULT_TENANT_ID))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<ResourceSegmentationModel>> loadResourceSegmentations(YdbTxSession session, List<String> ids) {
        List<Tuple2<String, TenantId>> idsWithTenant = ids.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).toList();
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceSegmentationsLoader.getResourceSegmentationsByIds(session, idsWithTenant))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<ResourceSegmentationModel>> loadResourceSegmentations(YdbSession session, List<String> ids) {
        List<Tuple2<String, TenantId>> idsWithTenant = ids.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).toList();
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceSegmentationsLoader.getResourceSegmentationsByIds(
                        immediateTx(session), idsWithTenant))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<ResourceSegmentModel>> loadResourceSegments(YdbTxSession session, List<String> ids) {
        List<Tuple2<String, TenantId>> idsWithTenant = ids.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).toList();
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceSegmentsLoader.getResourceSegmentsByIds(session, idsWithTenant))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<ResourceSegmentModel>> loadResourceSegments(YdbSession session, List<String> ids) {
        List<Tuple2<String, TenantId>> idsWithTenant = ids.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).toList();
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceSegmentsLoader.getResourceSegmentsByIds(immediateTx(session), idsWithTenant))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<ResourceTypeModel>> loadResourceTypes(YdbTxSession session, List<String> ids) {
        List<Tuple2<String, TenantId>> idsWithTenant = ids.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).toList();
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceTypesLoader.getResourceTypesByIds(session, idsWithTenant))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }


    public Mono<List<ResourceTypeModel>> loadResourceTypes(YdbSession session, List<String> ids) {
        List<Tuple2<String, TenantId>> idsWithTenant = ids.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).toList();
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourceTypesLoader.getResourceTypesByIds(immediateTx(session), idsWithTenant))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<Optional<ServiceMinimalModel>> getService(YdbTxSession session, long id) {
        return servicesDao.getByIdMinimal(session, id).map(WithTxId::get);
    }

    public Mono<List<AccountModel>> loadAccounts(YdbSession session, Set<String> accountIds) {
        if (accountIds.isEmpty()) {
            return Mono.just(List.of());
        }
        return Flux.fromIterable(Lists.partition(new ArrayList<>(accountIds), 500))
                .concatMap(v -> accountsDao.getAllByIdsWithDeleted(immediateTx(session), v, Tenants.DEFAULT_TENANT_ID))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<AccountModel>> loadAccounts(YdbTxSession txSession, Set<String> accountIds) {
        return loadAccounts(txSession, accountIds, false);
    }
    public Mono<List<AccountModel>> loadAccounts(YdbTxSession txSession, Set<String> accountIds, boolean background) {
        if (accountIds.isEmpty()) {
            return Mono.just(List.of());
        }
        if (background) {
            return ydbTableClient.usingSessionMonoRetryable(s ->
                    Flux.fromIterable(Lists.partition(new ArrayList<>(accountIds), 500))
                            .concatMap(v -> accountsDao.getAllByIdsWithDeleted(immediateTx(s), v,
                                    Tenants.DEFAULT_TENANT_ID))
                            .collectList().map(l -> l.stream().flatMap(Collection::stream)
                                    .collect(Collectors.toList())));
        } else {
            return Flux.fromIterable(Lists.partition(new ArrayList<>(accountIds), 500))
                    .concatMap(v -> accountsDao.getAllByIdsWithDeleted(txSession, v, Tenants.DEFAULT_TENANT_ID))
                    .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
        }
    }

    public Mono<List<UserModel>> loadUsers(YdbSession session, Set<String> userIds) {
        if (userIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Tuple2<String, TenantId>> userIdsToLoad = userIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(userIdsToLoad, 500))
                .concatMap(v -> usersDao.getByIds(immediateTx(session), v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<UserModel>> loadUsersByStaffIds(YdbTxSession session, Set<Long> staffIds, boolean background) {
        if (staffIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Tuple2<Long, TenantId>> staffIdsToLoad = staffIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        if (background) {
            return ydbTableClient.usingSessionMonoRetryable(s ->
                    Flux.fromIterable(Lists.partition(staffIdsToLoad, 500))
                    .concatMap(v -> usersDao.getByStaffIds(immediateTx(s), v))
                    .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList())));
        } else {
            return Flux.fromIterable(Lists.partition(staffIdsToLoad, 500))
                    .concatMap(v -> usersDao.getByStaffIds(session, v))
                    .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
        }
    }

    public Mono<List<FolderModel>> loadFolders(YdbSession session, Set<String> folderIds) {
        if (folderIds.isEmpty()) {
            return Mono.just(List.of());
        }
        return Flux.fromIterable(Lists.partition(new ArrayList<>(folderIds), 500))
                .concatMap(v -> folderDao.getByIds(immediateTx(session), v, Tenants.DEFAULT_TENANT_ID))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<FolderModel>> loadFolders(YdbTxSession session, Set<String> folderIds) {
        if (folderIds.isEmpty()) {
            return Mono.just(List.of());
        }
        return Flux.fromIterable(Lists.partition(new ArrayList<>(folderIds), 500))
                .concatMap(v -> folderDao.getByIds(session, v, Tenants.DEFAULT_TENANT_ID))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<Optional<FolderModel>> loadFolder(YdbTxSession session, String folderId) {
        return folderDao.getById(session, folderId, Tenants.DEFAULT_TENANT_ID);
    }

    public Mono<List<ServiceMinimalModel>> loadServices(YdbSession session, Set<Long> serviceIds) {
        if (serviceIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Long> serviceIdsToLoad = new ArrayList<>(serviceIds);
        return Flux.fromIterable(Lists.partition(serviceIdsToLoad, 500))
                .concatMap(v -> servicesDao.getByIdsMinimal(immediateTx(session), v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<ServiceMinimalModel>> loadServices(YdbTxSession session, Set<Long> serviceIds) {
        if (serviceIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Long> serviceIdsToLoad = new ArrayList<>(serviceIds);
        return Flux.fromIterable(Lists.partition(serviceIdsToLoad, 500))
                .concatMap(v -> servicesDao.getByIdsMinimal(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<Map<Long, LongSet>> loadServicesParents(YdbTxSession txSession, List<Long> serviceIds,
                                                        boolean background) {
        if (serviceIds.isEmpty()) {
            return Mono.just(Map.of());
        }
        if (background) {
            return ydbTableClient.usingSessionMonoRetryable(s ->
                Flux.fromIterable(Lists.partition(serviceIds, 500))
                        .concatMap(v -> servicesDao.getAllParentsForServices(immediateTx(s), v,
                                Tenants.DEFAULT_TENANT_ID))
                        .collectList().map(l -> {
                            Map<Long, LongSet> result = new HashMap<>();
                            l.forEach(result::putAll);
                            return result;
                        }));
        } else {
            return Flux.fromIterable(Lists.partition(serviceIds, 500))
                    .concatMap(v -> servicesDao.getAllParentsForServices(txSession, v, Tenants.DEFAULT_TENANT_ID))
                    .collectList().map(l -> {
                        Map<Long, LongSet> result = new HashMap<>();
                        l.forEach(result::putAll);
                        return result;
                    });
        }
    }

    public Mono<LongSet> loadServiceParents(YdbTxSession txSession, long serviceId, boolean background) {
        if (background) {
            return ydbTableClient.usingSessionMonoRetryable(s ->
                    servicesDao.getAllParents(immediateTx(s), serviceId, Tenants.DEFAULT_TENANT_ID));
        } else {
            return servicesDao.getAllParents(txSession, serviceId, Tenants.DEFAULT_TENANT_ID);
        }
    }

    public Mono<List<QuotaModel>> loadQuotas(YdbTxSession session, Set<String> folderIds) {
        if (folderIds.isEmpty()) {
            return Mono.just(List.of());
        }
        return Flux.fromIterable(Lists.partition(new ArrayList<>(folderIds), 500))
                .concatMap(v -> quotasDao.getByFolders(session, v, Tenants.DEFAULT_TENANT_ID))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<AccountsQuotasModel>> loadAccountsQuotas(YdbTxSession session, Set<String> accountIds) {
        if (accountIds.isEmpty()) {
            return Mono.just(List.of());
        }
        return Flux.fromIterable(Lists.partition(new ArrayList<>(accountIds), 500))
                .concatMap(v -> accountsQuotasDao.getAllByAccountIds(session, Tenants.DEFAULT_TENANT_ID,
                        new HashSet<>(v)))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<AbcServiceMemberModel>> loadServiceRoles(YdbTxSession txSession, Set<Long> serviceIds,
                                                              Set<Long> roleIds, boolean background) {
        if (serviceIds.isEmpty() || roleIds.isEmpty()) {
            return Mono.just(List.of());
        }
        if (background) {
            return ydbTableClient.usingSessionMonoRetryable(s -> abcServiceMemberDao.getByServicesAndRoles(
                    immediateTx(s), serviceIds, roleIds));
        } else {
            return abcServiceMemberDao.getByServicesAndRoles(txSession, serviceIds, roleIds);
        }
    }

    public Mono<List<TransferRequestByResponsibleModel>> loadByResponsibleIndex(YdbTxSession txSession,
                                                                                TransferRequestModel request) {
        Set<String> responsibleIds = new HashSet<>();
        request.getResponsible().getResponsible().forEach(r -> r.getResponsible()
                .forEach(v -> responsibleIds.addAll(v.getResponsibleIds())));
        request.getResponsible().getReserveResponsibleModel().map(ReserveResponsibleModel::getResponsibleIds)
                .ifPresent(responsibleIds::addAll);
        List<TransferRequestByResponsibleModel.Identity> ids = responsibleIds.stream()
                .map(v -> new TransferRequestByResponsibleModel.Identity(v, request.getStatus(),
                        request.getCreatedAt(), request.getId())).collect(Collectors.toList());
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> transferRequestByResponsibleDao.getByIds(txSession, v, Tenants.DEFAULT_TENANT_ID))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<TransferRequestByFolderModel>> loadByFolderIndex(YdbTxSession txSession,
                                                                      TransferRequestModel request) {
        Set<String> folderIds = new HashSet<>();
        request.getParameters().getFolderTransfers().forEach(t -> folderIds.add(t.getFolderId()));
        request.getParameters().getQuotaTransfers().forEach(t -> folderIds.add(t.getDestinationFolderId()));
        request.getParameters().getAccountTransfers().forEach(t -> {
            folderIds.add(t.getSourceFolderId());
            folderIds.add(t.getDestinationFolderId());
        });
        request.getParameters().getProvisionTransfers().forEach(t -> {
            folderIds.add(t.getSourceFolderId());
            folderIds.add(t.getDestinationFolderId());
        });
        List<TransferRequestByFolderModel.Identity> ids = folderIds.stream()
                .map(v -> new TransferRequestByFolderModel.Identity(v, request.getStatus(), request.getCreatedAt(),
                        request.getId())).collect(Collectors.toList());
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> transferRequestByFolderDao.getByIds(txSession, v, Tenants.DEFAULT_TENANT_ID))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<TransferRequestByServiceModel>> loadByServiceIndex(YdbTxSession txSession,
                                                                        TransferRequestModel request) {
        Set<Long> serviceIds = new HashSet<>();
        request.getParameters().getFolderTransfers().forEach(t -> {
            serviceIds.add(t.getSourceServiceId());
            serviceIds.add(t.getDestinationServiceId());
        });
        request.getParameters().getQuotaTransfers().forEach(t -> serviceIds.add(t.getDestinationServiceId()));
        request.getParameters().getAccountTransfers().forEach(t -> {
            serviceIds.add(t.getSourceServiceId());
            serviceIds.add(t.getDestinationServiceId());
        });
        request.getParameters().getProvisionTransfers().forEach(t -> {
            serviceIds.add(t.getSourceServiceId());
            serviceIds.add(t.getDestinationServiceId());
        });
        List<TransferRequestByServiceModel.Identity> ids = serviceIds.stream()
                .map(v -> new TransferRequestByServiceModel.Identity(v, request.getStatus(), request.getCreatedAt(),
                        request.getId())).collect(Collectors.toList());
        if (ids.isEmpty()) {
            return Mono.just(List.of());
        }
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> transferRequestByServiceDao.getByIds(txSession, v, Tenants.DEFAULT_TENANT_ID))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<ResourceModel>> loadResources(YdbSession session, Set<String> resourceIds) {
        if (resourceIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Tuple2<String, TenantId>> resourceIdsToLoad = resourceIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(resourceIdsToLoad, 500))
                .concatMap(v -> resourcesLoader.getResourcesByIds(immediateTx(session), v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<ResourceModel>> loadResources(YdbTxSession session, Set<String> resourceIds) {
        if (resourceIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Tuple2<String, TenantId>> resourceIdsToLoad = resourceIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(resourceIdsToLoad, 500))
                .concatMap(v -> resourcesLoader.getResourcesByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<UnitsEnsembleModel>> loadUnitsEnsembles(YdbSession session, Set<String> unitsEnsembleIds) {
        if (unitsEnsembleIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Tuple2<String, TenantId>> unitsEnsembleIdsToLoad = unitsEnsembleIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(unitsEnsembleIdsToLoad, 500))
                .concatMap(v -> unitsEnsemblesLoader.getUnitsEnsemblesByIds(immediateTx(session), v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<UnitsEnsembleModel>> loadUnitsEnsembles(YdbTxSession session, Set<String> unitsEnsembleIds) {
        if (unitsEnsembleIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Tuple2<String, TenantId>> unitsEnsembleIdsToLoad = unitsEnsembleIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(unitsEnsembleIdsToLoad, 500))
                .concatMap(v -> unitsEnsemblesLoader.getUnitsEnsemblesByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<FolderModel>> loadFolders(YdbSession session, Set<String> folderIds,
                                               List<FolderModel> knownFolders) {
        Map<String, FolderModel> knownFoldersById = knownFolders.stream()
                .collect(Collectors.toMap(FolderModel::getId, Function.identity()));
        Set<String> folderIdsToLoad = Sets.difference(folderIds, knownFoldersById.keySet());
        Set<String> alreadyLoadedFolderIds = Sets.intersection(folderIds, knownFoldersById.keySet());
        List<FolderModel> alreadyLoadedFolders = alreadyLoadedFolderIds.stream().map(knownFoldersById::get).toList();
        return loadFolders(session, folderIdsToLoad).map(loadedFolders -> {
            List<FolderModel> result = new ArrayList<>(loadedFolders);
            result.addAll(alreadyLoadedFolders);
            return result;
        });
    }

    public Mono<List<ServiceMinimalModel>> loadServices(YdbSession session, Set<Long> serviceIds,
                                                        List<ServiceMinimalModel> knownServices) {
        Map<Long, ServiceMinimalModel> knownServicesById = knownServices.stream()
                .collect(Collectors.toMap(ServiceMinimalModel::getId, Function.identity()));
        Set<Long> serviceIdsToLoad = Sets.difference(serviceIds, knownServicesById.keySet());
        Set<Long> alreadyLoadedServiceIds = Sets.intersection(serviceIds, knownServicesById.keySet());
        List<ServiceMinimalModel> alreadyLoadedServices = alreadyLoadedServiceIds.stream().map(knownServicesById::get)
                .toList();
        return loadServices(session, serviceIdsToLoad).map(loadedServices -> {
            List<ServiceMinimalModel> result = new ArrayList<>(loadedServices);
            result.addAll(alreadyLoadedServices);
            return result;
        });
    }

    public Mono<List<ResourceModel>> loadResources(YdbSession session, Set<String> resourceIds,
                                                   List<ResourceModel> knownResources) {
        Map<String, ResourceModel> knownResourcesById = knownResources.stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        Set<String> resourceIdsToLoad = Sets.difference(resourceIds, knownResourcesById.keySet());
        Set<String> alreadyLoadedResourceIds = Sets.intersection(resourceIds, knownResourcesById.keySet());
        List<ResourceModel> alreadyLoadedResources = alreadyLoadedResourceIds.stream().map(knownResourcesById::get)
                .toList();
        return loadResources(session, resourceIdsToLoad).map(loadedResources -> {
            List<ResourceModel> result = new ArrayList<>(loadedResources);
            result.addAll(alreadyLoadedResources);
            return result;
        });
    }

    public Mono<List<ProviderModel>> loadProviders(YdbSession session, Set<String> providerIds,
                                                   List<ProviderModel> knownProviders) {
        Map<String, ProviderModel> knownProvidersById = knownProviders.stream()
                .collect(Collectors.toMap(ProviderModel::getId, Function.identity()));
        Set<String> providerIdsToLoad = Sets.difference(providerIds, knownProvidersById.keySet());
        Set<String> alreadyLoadedProviderIds = Sets.intersection(providerIds, knownProvidersById.keySet());
        List<ProviderModel> alreadyLoadedProviders = alreadyLoadedProviderIds.stream().map(knownProvidersById::get)
                .toList();
        return loadProviders(session, providerIdsToLoad).map(loadedProviders -> {
            List<ProviderModel> result = new ArrayList<>(loadedProviders);
            result.addAll(alreadyLoadedProviders);
            return result;
        });
    }

    public Mono<List<UnitsEnsembleModel>> loadUnitsEnsembles(YdbSession session, Set<String> unitsEnsembleIds,
                                                             List<UnitsEnsembleModel> knownUnitsEnsembles) {
        Map<String, UnitsEnsembleModel> knownUnitsEnsemblesById = knownUnitsEnsembles.stream()
                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
        Set<String> unitsEnsembleIdsToLoad = Sets.difference(unitsEnsembleIds, knownUnitsEnsemblesById.keySet());
        Set<String> alreadyLoadedUnitsEnsembleIds = Sets
                .intersection(unitsEnsembleIds, knownUnitsEnsemblesById.keySet());
        List<UnitsEnsembleModel> alreadyLoadedUnitsEnsembles = alreadyLoadedUnitsEnsembleIds.stream()
                .map(knownUnitsEnsemblesById::get).toList();
        return loadUnitsEnsembles(session, unitsEnsembleIdsToLoad)
                .map(loadedUnitsEnsembles -> {
                    List<UnitsEnsembleModel> result = new ArrayList<>(loadedUnitsEnsembles);
                    result.addAll(alreadyLoadedUnitsEnsembles);
                    return result;
                });
    }

    public Mono<List<ProviderModel>> loadProviders(YdbSession session, Set<String> providerIds) {
        if (providerIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Tuple2<String, TenantId>> providerIdsToLoad = providerIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(providerIdsToLoad, 500))
                .concatMap(v -> providersLoader.getProvidersByIds(immediateTx(session), v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<Optional<ProviderModel>> loadProvider(YdbTxSession session, String providerId) {
        return providersLoader.getProviderById(session, providerId, Tenants.DEFAULT_TENANT_ID);
    }

    public Mono<List<ProviderModel>> loadProviders(YdbTxSession session, Set<String> providerIds) {
        return loadProviders(session, providerIds, false);
    }

    public Mono<List<ProviderModel>> loadProviders(YdbTxSession session, Set<String> providerIds, boolean background) {
        if (providerIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Tuple2<String, TenantId>> providerIdsToLoad = providerIds.stream()
                .map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        if (background) {
            return ydbTableClient.usingSessionMonoRetryable(s ->
                    Flux.fromIterable(Lists.partition(providerIdsToLoad, 500))
                            .concatMap(v -> providersLoader.getProvidersByIds(immediateTx(s), v))
                            .collectList().map(l -> l.stream().flatMap(Collection::stream)
                                    .collect(Collectors.toList())));
        } else {
            return Flux.fromIterable(Lists.partition(providerIdsToLoad, 500))
                    .concatMap(v -> providersLoader.getProvidersByIds(session, v))
                    .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
        }
    }

    public Mono<List<FolderModel>> loadReserveFolder(YdbTxSession session, Long serviceId) {
        return folderDao.getAllReservedFoldersByServiceIds(session, List.of(new WithTenant<>(Tenants.DEFAULT_TENANT_ID,
                serviceId)));
    }

    public Mono<Void> upsertFolders(YdbTxSession txSession, List<FolderModel> values) {
        if (values.isEmpty()) {
            return Mono.empty();
        }
        return Flux.fromIterable(Lists.partition(values, 500))
                .concatMap(v -> folderDao.upsertAllRetryable(txSession, v)).then();
    }

    public Mono<Void> upsertQuotas(YdbTxSession txSession, List<QuotaModel> values) {
        if (values.isEmpty()) {
            return Mono.empty();
        }
        return Flux.fromIterable(Lists.partition(values, 500))
                .concatMap(v -> quotasDao.upsertAllRetryable(txSession, v)).then();
    }

    public Mono<Void> upsertFolderOperationLog(YdbTxSession txSession, List<FolderOperationLogModel> values) {
        if (values.isEmpty()) {
            return Mono.empty();
        }
        return Flux.fromIterable(Lists.partition(values, 500))
                .concatMap(v -> folderOperationLogDao.upsertAllRetryable(txSession, v)).then();
    }

    public Mono<Void> updatePendingTransferRequest(YdbTxSession txSession,
                                                   TenantId tenantId,
                                                   String id,
                                                   TransferRequestStatus newStatus) {
        return pendingTransferRequestsDao.getById(txSession, id, tenantId).flatMap(pending -> {
            if (pending.isPresent()) {
                if (newStatus.equals(TransferRequestStatus.PENDING)) {
                    return Mono.empty();
                }
                return pendingTransferRequestsDao.deleteOneRetryable(txSession, new WithTenant<>(tenantId, id)).then();
            } else {
                if (newStatus.equals(TransferRequestStatus.PENDING)) {
                    PendingTransferRequestsModel model = PendingTransferRequestsModel.builder()
                            .tenantId(tenantId)
                            .requestId(id)
                            .build();
                    return pendingTransferRequestsDao.upsertOneRetryable(txSession, model).then();
                }
                return Mono.empty();
            }
        });
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<TransferRequestModel> saveCancellation(YdbTxSession txSession,
                                                       TransferRequestModel updatedTransferRequest,
                                                       TransferRequestHistoryModel newHistory,
                                                       List<TransferRequestByResponsibleModel> oldByResponsibleIndices,
                                                       List<TransferRequestByFolderModel> oldByFolderIndices,
                                                       List<TransferRequestByServiceModel> oldByServiceIndices,
                                                       List<TransferRequestByResponsibleModel> newByResponsibleIndices,
                                                       List<TransferRequestByFolderModel> newByFolderIndices,
                                                       List<TransferRequestByServiceModel> newByServiceIndices,
                                                       String unique,
                                                       YaUserDetails currentUser) {
        TenantId tenantId = updatedTransferRequest.getTenantId();
        String transferRequestId = updatedTransferRequest.getId();
        return deleteTransferRequestByResponsible(txSession, oldByResponsibleIndices)
                .then(Mono.defer(() -> deleteTransferRequestByFolder(txSession, oldByFolderIndices)))
                .then(Mono.defer(() -> deleteTransferRequestByService(txSession, oldByServiceIndices)))
                .then(Mono.defer(() -> updatePendingTransferRequest(txSession, tenantId, transferRequestId,
                        updatedTransferRequest.getStatus())))
                .then(Mono.defer(() -> upsertTransferRequestByResponsible(txSession, newByResponsibleIndices)))
                .then(Mono.defer(() -> upsertTransferRequestByFolder(txSession, newByFolderIndices)))
                .then(Mono.defer(() -> upsertTransferRequestByService(txSession, newByServiceIndices)))
                .then(Mono.defer(() -> transferRequestHistoryDao.upsertOneRetryable(txSession, newHistory).then()))
                .then(Mono.defer(() -> transferRequestsDao.upsertOneRetryable(txSession, updatedTransferRequest)))
                .then(Mono.defer(() -> unique != null
                        ? requestUniqueService.addCancelTransferRequestMono(txSession, unique,
                            updatedTransferRequest, updatedTransferRequest.getUpdatedAt().orElseThrow(), currentUser)
                            .thenReturn(updatedTransferRequest)
                        : Mono.just(updatedTransferRequest)));
    }

    public Mono<Void> saveCreate(YdbTxSession txSession,
                                 ValidatedCreateTransferRequest createRequest,
                                 String unique,
                                 YaUserDetails currentUser) {
        TenantId tenantId = createRequest.getTransferRequest().getTenantId();
        String transferRequestId = createRequest.getTransferRequest().getId();
        return upsertTransferRequestByResponsible(txSession, createRequest.getResponsibleIndices())
                .then(Mono.defer(() -> upsertTransferRequestByFolder(txSession, createRequest.getFolderIndices())))
                .then(Mono.defer(() -> upsertTransferRequestByService(txSession, createRequest.getServiceIndices())))
                .then(Mono.defer(() -> updatePendingTransferRequest(txSession, tenantId, transferRequestId,
                        createRequest.getTransferRequest().getStatus())))
                .then(Mono.defer(() -> transferRequestHistoryDao
                        .upsertOneRetryable(txSession, createRequest.getHistory()).then()))
                .then(Mono.defer(() -> transferRequestsDao
                        .upsertOneRetryable(txSession, createRequest.getTransferRequest()).then()))
                .then(Mono.defer(() -> unique != null
                        ? requestUniqueService.addCreateTransferRequestMono(txSession, unique,
                                createRequest.getTransferRequest(), createRequest.getNow(), currentUser)
                        : Mono.empty()));
    }

    public Mono<Void> savePut(YdbTxSession txSession,
                              ValidatedPutTransferRequest putRequest, String unique, YaUserDetails currentUser) {
        if (putRequest.getHistory().isEmpty() || putRequest.getTransferRequest().isEmpty()) {
            return Mono.empty();
        }
        TenantId tenantId = putRequest.getTransferRequest().get().getTenantId();
        String transferRequestId = putRequest.getTransferRequest().get().getId();
        return deleteTransferRequestByResponsible(txSession,
                putRequest.getIndicesDifference().getRemovedResponsible())
                .then(Mono.defer(() -> deleteTransferRequestByFolder(txSession,
                        putRequest.getIndicesDifference().getRemovedFolders())))
                .then(Mono.defer(() -> deleteTransferRequestByService(txSession,
                        putRequest.getIndicesDifference().getRemovedService())))
                .then(Mono.defer(() -> upsertTransferRequestByResponsible(txSession,
                        putRequest.getIndicesDifference().getAddedResponsible())))
                .then(Mono.defer(() -> upsertTransferRequestByFolder(txSession,
                        putRequest.getIndicesDifference().getAddedFolders())))
                .then(Mono.defer(() -> upsertTransferRequestByService(txSession,
                        putRequest.getIndicesDifference().getAddedServices())))
                .then(Mono.defer(() -> updatePendingTransferRequest(txSession, tenantId, transferRequestId,
                        putRequest.getTransferRequest().get().getStatus())))
                .then(Mono.defer(() -> transferRequestHistoryDao.upsertOneRetryable(txSession,
                        putRequest.getHistory().get()).then()))
                .then(Mono.defer(() -> transferRequestsDao.upsertOneRetryable(txSession,
                        putRequest.getTransferRequest().get()).then()))
                .then(Mono.defer(() -> unique != null
                        ? requestUniqueService.addPutTransferRequestMono(txSession, unique,
                                putRequest.getTransferRequest().get(), putRequest.getNow(), currentUser)
                        : Mono.empty()));
    }

    public Mono<Void> saveRefresh(YdbTxSession txSession,
                                  RefreshTransferRequest refreshTransferRequest) {
        return saveRefresh(txSession, refreshTransferRequest.getRefreshedTransferRequest(),
                refreshTransferRequest.getHistory(), refreshTransferRequest.getIndicesDifference());
    }

    public Mono<Void> saveRefresh(YdbTxSession txSession,
                                  TransferRequestModel transferRequestModel,
                                  TransferRequestHistoryModel transferRequestHistoryModel,
                                  TransferRequestIndices.Difference indicesChanges) {
        TenantId tenantId = transferRequestModel.getTenantId();
        String transferRequestId = transferRequestModel.getId();
        return deleteTransferRequestByResponsible(txSession, indicesChanges.getRemovedResponsible())
                .then(Mono.defer(() -> deleteTransferRequestByFolder(txSession, indicesChanges.getRemovedFolders())))
                .then(Mono.defer(() -> deleteTransferRequestByService(txSession, indicesChanges.getRemovedService())))
                .then(Mono.defer(() -> upsertTransferRequestByResponsible(txSession,
                        indicesChanges.getAddedResponsible())))
                .then(Mono.defer(() -> upsertTransferRequestByFolder(txSession, indicesChanges.getAddedFolders())))
                .then(Mono.defer(() -> upsertTransferRequestByService(txSession, indicesChanges.getAddedServices())))
                .then(Mono.defer(() -> updatePendingTransferRequest(txSession, tenantId, transferRequestId,
                        transferRequestModel.getStatus())))
                .then(Mono.defer(() -> transferRequestHistoryDao
                        .upsertOneRetryable(txSession, transferRequestHistoryModel).then()))
                .then(Mono.defer(() -> transferRequestsDao.upsertOneRetryable(txSession, transferRequestModel).then()));
    }

    public Mono<Void> saveVote(YdbTxSession txSession,
                               ValidatedVoteTransferRequest voteRequest, String unique, YaUserDetails currentUser) {
        TransferRequestModel transferRequest = voteRequest.getTransferRequest();
        TenantId tenantId = transferRequest.getTenantId();
        return deleteTransferRequestByResponsible(txSession,
                voteRequest.getIndicesDifference().getRemovedResponsible())
                .then(Mono.defer(() -> updatePendingTransferRequest(txSession, tenantId, transferRequest.getId(),
                        transferRequest.getStatus())))
                .then(Mono.defer(() -> deleteTransferRequestByFolder(txSession,
                        voteRequest.getIndicesDifference().getRemovedFolders())))
                .then(Mono.defer(() -> deleteTransferRequestByService(txSession,
                        voteRequest.getIndicesDifference().getRemovedService())))
                .then(Mono.defer(() -> upsertTransferRequestByResponsible(txSession,
                        voteRequest.getIndicesDifference().getAddedResponsible())))
                .then(Mono.defer(() -> upsertTransferRequestByFolder(txSession,
                        voteRequest.getIndicesDifference().getAddedFolders())))
                .then(Mono.defer(() -> upsertTransferRequestByService(txSession,
                        voteRequest.getIndicesDifference().getAddedServices())))
                .then(Mono.defer(() -> transferRequestHistoryDao
                        .upsertOneRetryable(txSession, voteRequest.getHistory()).then()))
                .then(Mono.defer(() -> transferRequestsDao
                        .upsertOneRetryable(txSession, transferRequest).then()))
                .then(Mono.defer(() -> unique != null
                        ? requestUniqueService.addVoteForTransferRequestMono(txSession, unique,
                        transferRequest, voteRequest.getNow(), currentUser)
                        : Mono.empty()));
    }

    private Mono<Void> deleteTransferRequestByResponsible(YdbTxSession txSession,
                                                          List<TransferRequestByResponsibleModel> values) {
        if (values.isEmpty()) {
            return Mono.empty();
        }
        List<WithTenant<TransferRequestByResponsibleModel.Identity>> ids = values.stream()
                .map(v -> new WithTenant<>(v.getTenantId(), v.getIdentity())).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> transferRequestByResponsibleDao.deleteManyRetryable(txSession, v)).then();
    }

    private Mono<Void> deleteTransferRequestByFolder(YdbTxSession txSession,
                                                     List<TransferRequestByFolderModel> values) {
        if (values.isEmpty()) {
            return Mono.empty();
        }
        List<WithTenant<TransferRequestByFolderModel.Identity>> ids = values.stream()
                .map(v -> new WithTenant<>(v.getTenantId(), v.getIdentity())).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> transferRequestByFolderDao.deleteManyRetryable(txSession, v)).then();
    }

    private Mono<Void> deleteTransferRequestByService(YdbTxSession txSession,
                                                      List<TransferRequestByServiceModel> values) {
        if (values.isEmpty()) {
            return Mono.empty();
        }
        List<WithTenant<TransferRequestByServiceModel.Identity>> ids = values.stream()
                .map(v -> new WithTenant<>(v.getTenantId(), v.getIdentity())).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> transferRequestByServiceDao.deleteManyRetryable(txSession, v)).then();
    }

    private Mono<Void> upsertTransferRequestByResponsible(YdbTxSession txSession,
                                                          List<TransferRequestByResponsibleModel> values) {
        if (values.isEmpty()) {
            return Mono.empty();
        }
        return Flux.fromIterable(Lists.partition(values, 500))
                .concatMap(v -> transferRequestByResponsibleDao.upsertAllRetryable(txSession, v)).then();
    }

    private Mono<Void> upsertTransferRequestByFolder(YdbTxSession txSession,
                                                     List<TransferRequestByFolderModel> values) {
        if (values.isEmpty()) {
            return Mono.empty();
        }
        return Flux.fromIterable(Lists.partition(values, 500))
                .concatMap(v -> transferRequestByFolderDao.upsertAllRetryable(txSession, v)).then();
    }

    private Mono<Void> upsertTransferRequestByService(YdbTxSession txSession,
                                                      List<TransferRequestByServiceModel> values) {
        if (values.isEmpty()) {
            return Mono.empty();
        }
        return Flux.fromIterable(Lists.partition(values, 500))
                .concatMap(v -> transferRequestByServiceDao.upsertAllRetryable(txSession, v)).then();
    }

    private YdbTxSession immediateTx(YdbSession session) {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY);
    }

}
