package ru.yandex.intranet.d.services.transfer;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.ObjectReader;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderType;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.services.ServiceReadOnlyState;
import ru.yandex.intranet.d.model.services.ServiceState;
import ru.yandex.intranet.d.model.transfers.ResourceQuotaTransfer;
import ru.yandex.intranet.d.model.transfers.ServiceResponsible;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestStatus;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.model.transfers.TransferResponsible;
import ru.yandex.intranet.d.model.transfers.VoteType;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedProvision;
import ru.yandex.intranet.d.services.quotas.QuotasHelper;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.transfer.model.CreateTransferRequestConfirmationData;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedLoanParameters;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedProvisionTransfer;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedQuotaResourceTransfer;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedQuotaTransfer;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedQuotaTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedReserveTransfer;
import ru.yandex.intranet.d.services.transfer.model.PreValidatedTransferRequestParameters;
import ru.yandex.intranet.d.services.transfer.model.PutTransferRequestConfirmationData;
import ru.yandex.intranet.d.services.transfer.model.QuotaTransferPreApplicationData;
import ru.yandex.intranet.d.services.transfer.model.ResponsibleAndNotified;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestContinuationToken;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestHistoryContinuationToken;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestsSearchParameters;
import ru.yandex.intranet.d.services.transfer.model.ValidatedQuotaResourceTransfer;
import ru.yandex.intranet.d.services.transfer.model.ValidatedQuotaTransfer;
import ru.yandex.intranet.d.services.transfer.model.ValidatedQuotaTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.ValidatedReserveTransfer;
import ru.yandex.intranet.d.services.transfer.model.ValidatedReserveTransferParameters;
import ru.yandex.intranet.d.services.transfer.model.ValidatedTransferRequestContinuationToken;
import ru.yandex.intranet.d.services.transfer.model.ValidatedTransferRequestHistoryContinuationToken;
import ru.yandex.intranet.d.services.transfer.model.ValidatedTransferRequestsSearchParameters;
import ru.yandex.intranet.d.services.validators.AbcServiceValidator;
import ru.yandex.intranet.d.util.DisplayUtil;
import ru.yandex.intranet.d.util.FrontStringUtil;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.ProviderUtil;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TransferRequestErrorMessages;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.AmountDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestStatusDto;
import ru.yandex.intranet.d.web.model.transfers.TransferRequestTypeDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateProvisionTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateQuotaResourceTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateQuotaTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateReserveTransferDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontCreateTransferRequestParametersDto;
import ru.yandex.intranet.d.web.model.transfers.front.FrontTransferRequestVotingDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Transfer request validation service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class TransferRequestValidationService {
    private static final int MAX_DESCRIPTION_LENGTH = 8192;
    private static final Pattern KEY_PATTERN = Pattern.compile("^([A-Za-z]{2,512})-(\\d{1,64})$");
    private static final int MAX_FOLDER_PER_REQUEST = 100;
    private static final Set<FolderType> INVALID_FOLDER_TYPES = EnumSet.of(
            FolderType.PROVIDER_RESERVE
    );

    private final MessageSource messages;
    private final TransferRequestSecurityService securityService;
    private final TransferRequestStoreService storeService;
    private final SecurityManagerService securityManagerService;
    private final ObjectReader transferRequestContinuationTokenReader;
    private final ObjectReader transferRequestHistoryContinuationTokenReader;
    private final TransferRequestErrorMessages transferRequestErrorMessages;

    public TransferRequestValidationService(
            @Qualifier("messageSource") MessageSource messages,
            TransferRequestSecurityService securityService,
            TransferRequestStoreService storeService,
            SecurityManagerService securityManagerService,
            @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper,
            @Qualifier("transferRequestErrorMessages") TransferRequestErrorMessages transferRequestErrorMessages) {
        this.messages = messages;
        this.securityService = securityService;
        this.storeService = storeService;
        this.securityManagerService = securityManagerService;
        this.transferRequestContinuationTokenReader = objectMapper.getObjectMapper()
                .readerFor(TransferRequestContinuationToken.class);
        this.transferRequestHistoryContinuationTokenReader = objectMapper.getObjectMapper()
                .readerFor(TransferRequestHistoryContinuationToken.class);
        this.transferRequestErrorMessages = transferRequestErrorMessages;
    }

    @SuppressWarnings("ParameterNumber")
    public void validateCreateFields(Supplier<Optional<String>> descriptionGetter,
                                     Supplier<Optional<FrontCreateTransferRequestParametersDto>> parametersGetter,
                                     PreValidatedTransferRequestParameters.Builder builder,
                                     ErrorCollection.Builder errors,
                                     Locale locale) {
        validateOptionalText(descriptionGetter, builder::description, errors, "description",
                MAX_DESCRIPTION_LENGTH, locale);
        validateParameters(parametersGetter, errors, locale);
    }

    public void validateParameters(Supplier<Optional<FrontCreateTransferRequestParametersDto>> parametersGetter,
                                    ErrorCollection.Builder errors, Locale locale) {
        if (parametersGetter.get().isEmpty()) {
            errors.addError("parameters", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        FrontCreateTransferRequestParametersDto frontCreateTransferRequestParametersDto = parametersGetter.get().get();
        long fieldCount = Stream.of(frontCreateTransferRequestParametersDto.getQuotaTransfers(),
                        frontCreateTransferRequestParametersDto.getReserveTransfer(),
                        frontCreateTransferRequestParametersDto.getProvisionTransfers())
                .filter(Optional::isPresent)
                .map(Optional::get)
                .filter(o -> !(o instanceof Collection) || !((Collection<?>) o).isEmpty())
                .count();
        if (fieldCount != 1) {
            errors.addError("parameters", TypedError.invalid(messages
                    .getMessage("errors.transfers.invalid.parameters", null, locale)));
        }
    }

    public void preValidateQuotaTransferParameters(FrontCreateTransferRequestParametersDto parameters,
                                                   PreValidatedQuotaTransferParameters.Builder builder,
                                                   ErrorCollection.Builder errors,
                                                   Locale locale, boolean publicApi) {
        if (parameters.getQuotaTransfers().isEmpty() || parameters.getQuotaTransfers().get().isEmpty()) {
            errors.addError("parameters.quotaTransfers", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        for (int i = 0; i < parameters.getQuotaTransfers().get().size(); i++) {
            FrontCreateQuotaTransferDto quotaTransfer = parameters.getQuotaTransfers().get().get(i);
            if (quotaTransfer == null) {
                errors.addError("parameters.quotaTransfers." + i, TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                continue;
            }
            PreValidatedQuotaTransfer.Builder quotaTransferBuilder = PreValidatedQuotaTransfer.builder();
            quotaTransferBuilder.index(i);
            ErrorCollection.Builder quotaTransferErrors = ErrorCollection.builder();
            if (!publicApi) {
                validateServiceId(quotaTransfer::getDestinationServiceId, quotaTransferBuilder::destinationServiceId,
                        quotaTransferErrors, "parameters.quotaTransfers." + i + ".destinationServiceId", locale);
            }
            validateFolderId(quotaTransfer::getDestinationFolderId, quotaTransferBuilder::destinationFolderId,
                    quotaTransferErrors, "parameters.quotaTransfers." + i
                            + (publicApi ? ".folderId" : ".destinationFolderId"), locale);
            validateResourceTransfer(quotaTransfer::getResourceTransfers, quotaTransferBuilder::addTransfer,
                    quotaTransferErrors, "parameters.quotaTransfers." + i + ".resourceTransfers", locale,
                    publicApi, false);
            if (quotaTransferErrors.hasAnyErrors()) {
                errors.add(quotaTransferErrors);
            } else {
                builder.addTransfer(quotaTransferBuilder.build());
            }
        }
        List<String> folderIds = new ArrayList<>();
        builder.getTransfers().forEach(quotaTransfer -> folderIds.add(quotaTransfer.getDestinationFolderId()));
        Set<String> uniqueFolderIds = new HashSet<>(folderIds);
        if (folderIds.size() > uniqueFolderIds.size()) {
            errors.addError("parameters.quotaTransfers", TypedError.invalid(messages
                    .getMessage("errors.duplicate.folder.ids.are.not.allowed", null, locale)));
        }
        if (builder.getTransfers().size() > MAX_FOLDER_PER_REQUEST) {
            errors.addError("parameters.quotaTransfers", TypedError.invalid(messages
                    .getMessage("errors.too.many.folders.per.transfer.request", null, locale)));
        }
    }

    public void preValidateReserveTransferParameters(FrontCreateTransferRequestParametersDto parameters,
                                                     PreValidatedReserveTransfer.Builder builder,
                                                     ErrorCollection.Builder errors,
                                                     Locale locale, boolean publicApi) {
        if (parameters.getReserveTransfer().isEmpty()) {
            errors.addError("parameters.reserveTransfer", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        FrontCreateReserveTransferDto reserveTransferDto = parameters.getReserveTransfer().get();
        ErrorCollection.Builder reserveTransferErrors = ErrorCollection.builder();
        if (!publicApi) {
            validateServiceId(reserveTransferDto::getDestinationServiceId, builder::destinationServiceId,
                    reserveTransferErrors, "parameters.reserveTransfer.destinationServiceId", locale);
        }
        validateFolderId(reserveTransferDto::getDestinationFolderId, builder::destinationFolderId,
                reserveTransferErrors, "parameters.reserveTransfer"
                        + (publicApi ? ".folderId" : ".destinationFolderId"), locale);
        validateProviderId(reserveTransferDto::getProviderId, builder::providerId,
                reserveTransferErrors, "parameters.reserveTransfer.providerId", locale);
        validateResourceTransfer(reserveTransferDto::getResourceTransfers, builder::addTransfer, reserveTransferErrors,
                "parameters.reserveTransfer.resourceTransfers", locale, publicApi, false);

        builder.getTransfers().stream()
                .filter(t -> t.getDelta().compareTo(BigDecimal.ZERO) <= 0)
                .forEach(t -> reserveTransferErrors.addError(
                        "parameters.reserveTransfer.resourceTransfers." + t.getIndex() + ".delta",
                        TypedError.invalid(messages.getMessage("errors.number.must.be.positive", null, locale))
                ));

        if (reserveTransferErrors.hasAnyErrors()) {
            errors.add(reserveTransferErrors);
        }
    }

    public List<PreValidatedProvisionTransfer.Builder> preValidateProvisionTransferParameters(
            FrontCreateTransferRequestParametersDto parameters,
            @Nullable PreValidatedLoanParameters loanParameters,
            ErrorCollection.Builder errors,
            Locale locale,
            boolean publicApi
    ) {
        String field = "parameters.provisionTransfers";
        if (parameters.getProvisionTransfers().isEmpty()) {
            errors.addError(field, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return List.of();
        }
        boolean provideOverCommitReserve = loanParameters != null
                && Boolean.TRUE.equals(loanParameters.getProvideOverCommitReserve());
        boolean possibleOverCommitPayOff = loanParameters != null && loanParameters.getPayOffParameters() != null;
        List<FrontCreateProvisionTransferDto> provisionTransfersDto = parameters.getProvisionTransfers().get();
        List<PreValidatedProvisionTransfer.Builder> result = new ArrayList<>();
        for (int i = 0; i < provisionTransfersDto.size(); i++) {
            FrontCreateProvisionTransferDto provisionTransferDto = provisionTransfersDto.get(i);
            PreValidatedProvisionTransfer.Builder builder = PreValidatedProvisionTransfer.builder();
            String indexedField = field + "." + i;
            if (!publicApi) {
                validateServiceId(provisionTransferDto.getSourceServiceId(),
                        builder::sourceServiceId, errors,
                        indexedField + ".sourceServiceId", locale);
                validateServiceId(provisionTransferDto.getDestinationServiceId(),
                        builder::destinationServiceId, errors,
                        indexedField + ".destinationServiceId", locale);
            }
            validateFolderId(provisionTransferDto.getSourceFolderId(),
                    builder::sourceFolderId, errors,
                    indexedField + ".sourceFolderId", locale);
            validateFolderId(provisionTransferDto.getDestinationFolderId(),
                    builder::destinationFolderId, errors,
                    indexedField + ".destinationFolderId", locale);
            validateAccountId(provisionTransferDto.getSourceAccountId(),
                    builder::sourceAccountId, errors,
                    indexedField + ".sourceAccountId", locale);
            validateAccountId(provisionTransferDto.getDestinationAccountId(),
                    builder::destinationAccountId, errors,
                    indexedField + ".destinationAccountId", locale);

            validateResourceTransfer(provisionTransferDto.getSourceAccountTransfers(),
                    builder::addSourceAccountTransfer, errors,
                    indexedField + ".sourceAccountTransfers", locale, publicApi,
                    provideOverCommitReserve);
            validateResourceTransfer(provisionTransferDto.getDestinationAccountTransfers(),
                    builder::addDestinationAccountTransfer, errors,
                    indexedField + ".destinationAccountTransfers", locale, publicApi,
                    possibleOverCommitPayOff);
            result.add(builder);
        }
        return result;
    }

    public void validateResourceDuplicates(List<PreValidatedQuotaResourceTransfer> transfers, String fieldKey,
                                           ErrorCollection.Builder errors, Locale locale) {
        Set<String> resourcesIds = transfers.stream()
                .map(PreValidatedQuotaResourceTransfer::getResourceId)
                .collect(Collectors.toSet());

        if (transfers.size() > resourcesIds.size()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.resource.id.duplicate", null, locale)));
        }
    }

    private void validateResourceTransfer(Supplier<Optional<List<FrontCreateQuotaResourceTransferDto>>> getter,
                                          Consumer<PreValidatedQuotaResourceTransfer> setter,
                                          ErrorCollection.Builder errorBuilder,
                                          String fieldKey,
                                          Locale locale,
                                          boolean publicApi,
                                          boolean allowEmptyField) {
        validateResourceTransfer(getter.get().orElse(null), setter, errorBuilder, fieldKey, locale,
                publicApi, allowEmptyField);
    }

    private void validateResourceTransfer(@Nullable List<FrontCreateQuotaResourceTransferDto> values,
                                          Consumer<PreValidatedQuotaResourceTransfer> setter,
                                          ErrorCollection.Builder errorBuilder,
                                          String fieldKey,
                                          Locale locale,
                                          boolean publicApi,
                                          boolean allowEmptyField) {
        if (values != null && !values.isEmpty()) {
            List<PreValidatedQuotaResourceTransfer> transfers = new ArrayList<>(values.size());
            for (int j = 0; j < values.size(); j++) {
                FrontCreateQuotaResourceTransferDto resourceTransfer = values.get(j);
                if (resourceTransfer == null) {
                    errorBuilder.addError(fieldKey + '.' + j,
                            TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
                    continue;
                }
                PreValidatedQuotaResourceTransfer.Builder resourceTransferBuilder
                        = PreValidatedQuotaResourceTransfer.builder();
                resourceTransferBuilder.index(j);
                ErrorCollection.Builder resourceTransferErrors = ErrorCollection.builder();
                validateResourceId(resourceTransfer::getResourceId, resourceTransferBuilder::resourceId,
                        resourceTransferErrors, fieldKey + '.' + j + ".resourceId", locale);
                if (publicApi) {
                    validateUnitKey(resourceTransfer::getDeltaUnitId, resourceTransferBuilder::deltaUnitId,
                            resourceTransferErrors, fieldKey + '.' + j + ".deltaUnitKey", locale);
                } else {
                    validateUnitId(resourceTransfer::getDeltaUnitId, resourceTransferBuilder::deltaUnitId,
                            resourceTransferErrors, fieldKey + '.' + j + ".deltaUnitId", locale);
                }
                validateDelta(resourceTransfer::getDelta, resourceTransferBuilder::delta,
                        resourceTransferErrors, fieldKey + '.' + j + ".delta", locale);
                if (resourceTransferErrors.hasAnyErrors()) {
                    errorBuilder.add(resourceTransferErrors);
                } else {
                    PreValidatedQuotaResourceTransfer transfer = resourceTransferBuilder.build();
                    setter.accept(transfer);
                    transfers.add(transfer);
                }
            }
            validateResourceDuplicates(transfers, fieldKey, errorBuilder, locale);
        } else if (!allowEmptyField) {
            errorBuilder.addError(fieldKey,
                    TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)));
        }
    }

    public Result<TransferRequestType> validateTransferRequestType(
            Supplier<Optional<TransferRequestTypeDto>> supplier, Locale locale) {
        Optional<TransferRequestTypeDto> typeO = supplier.get();
        if (typeO.isEmpty()) {
            return Result.failure(ErrorCollection.builder().addError("requestType", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale))).build());
        }
        TransferRequestTypeDto type = typeO.get();
        return switch (type) {
            case FOLDER_TRANSFER -> Result.success(TransferRequestType.FOLDER_TRANSFER);
            case QUOTA_TRANSFER -> Result.success(TransferRequestType.QUOTA_TRANSFER);
            case ACCOUNT_TRANSFER -> Result.success(TransferRequestType.ACCOUNT_TRANSFER);
            case PROVISION_TRANSFER -> Result.success(TransferRequestType.PROVISION_TRANSFER);
            case RESERVE_TRANSFER -> Result.success(TransferRequestType.RESERVE_TRANSFER);
            default -> Result.failure(ErrorCollection.builder().addError("requestType", TypedError.invalid(messages
                    .getMessage("errors.invalid.transfer.request.type", null, locale))).build());
        };
    }

    public Result<TransferRequestModel> validateExists(TransferRequestModel transferRequest, Locale locale) {
        if (transferRequest == null) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.transfer.request.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(transferRequest);
    }

    public Result<Void> validateId(String transferRequestId, Locale locale) {
        if (!Uuids.isValidUuid(transferRequestId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.transfer.request.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    public Result<PageRequest.Validated<ValidatedTransferRequestContinuationToken>> validatePageRequest(
            PageRequest pageRequest, Locale locale) {
        return pageRequest.<TransferRequestContinuationToken>validate(transferRequestContinuationTokenReader,
                messages, locale).andThen(validatedPage -> {
            ErrorCollection.Builder errors = ErrorCollection.builder();
            if (validatedPage.getContinuationToken().isEmpty()) {
                return Result.success(new PageRequest.Validated<>(null, validatedPage.getLimit()));
            }
            ValidatedTransferRequestContinuationToken.Builder builder
                    = ValidatedTransferRequestContinuationToken.builder();
            if (validatedPage.getContinuationToken().get().getId().isEmpty()) {
                errors.addError(TypedError.badRequest(messages
                        .getMessage("errors.invalid.continuation.token", null, locale)));
            } else if (!Uuids.isValidUuid(validatedPage.getContinuationToken().get().getId().get())) {
                errors.addError(TypedError.badRequest(messages
                        .getMessage("errors.invalid.continuation.token", null, locale)));
            } else {
                builder.id(validatedPage.getContinuationToken().get().getId().get());
            }
            if (validatedPage.getContinuationToken().get().getTimestamp().isEmpty()) {
                errors.addError(TypedError.badRequest(messages
                        .getMessage("errors.invalid.continuation.token", null, locale)));
            } else {
                try {
                    builder.createdAt(Instant.ofEpochMilli(validatedPage
                            .getContinuationToken().get().getTimestamp().get()));
                } catch (Exception e) {
                    errors.addError(TypedError.badRequest(messages
                            .getMessage("errors.invalid.continuation.token", null, locale)));
                }
            }
            if (errors.hasAnyErrors()) {
                return Result.failure(errors.build());
            }
            return Result.success(new PageRequest.Validated<>(builder.build(), validatedPage.getLimit()));
        });
    }

    public Result<PageRequest.Validated<ValidatedTransferRequestHistoryContinuationToken>> validateHistoryPageRequest(
            PageRequest pageRequest, Locale locale) {
        return pageRequest.<TransferRequestHistoryContinuationToken>validate(
                transferRequestHistoryContinuationTokenReader, messages, locale).andThen(validatedPage -> {
            ErrorCollection.Builder errors = ErrorCollection.builder();
            if (validatedPage.getContinuationToken().isEmpty()) {
                return Result.success(new PageRequest.Validated<>(null, validatedPage.getLimit()));
            }
            ValidatedTransferRequestHistoryContinuationToken.Builder builder
                    = ValidatedTransferRequestHistoryContinuationToken.builder();
            if (validatedPage.getContinuationToken().get().getId().isEmpty()) {
                errors.addError(TypedError.badRequest(messages
                        .getMessage("errors.invalid.continuation.token", null, locale)));
            } else if (!Uuids.isValidUuid(validatedPage.getContinuationToken().get().getId().get())) {
                errors.addError(TypedError.badRequest(messages
                        .getMessage("errors.invalid.continuation.token", null, locale)));
            } else {
                builder.id(validatedPage.getContinuationToken().get().getId().get());
            }
            if (validatedPage.getContinuationToken().get().getTimestamp().isEmpty()) {
                errors.addError(TypedError.badRequest(messages
                        .getMessage("errors.invalid.continuation.token", null, locale)));
            } else {
                try {
                    builder.timestamp(Instant.ofEpochMilli(validatedPage
                            .getContinuationToken().get().getTimestamp().get()));
                } catch (Exception e) {
                    errors.addError(TypedError.badRequest(messages
                            .getMessage("errors.invalid.continuation.token", null, locale)));
                }
            }
            if (errors.hasAnyErrors()) {
                return Result.failure(errors.build());
            }
            return Result.success(new PageRequest.Validated<>(builder.build(), validatedPage.getLimit()));
        });
    }

    public Result<Void> validateVersion(TransferRequestModel request, Long version, Locale locale) {
        if (version == null) {
            return Result.failure(ErrorCollection.builder().addError("version", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale))).build());
        }
        if (version != request.getVersion()) {
            return Result.failure(ErrorCollection.builder().addError("version",
                    TypedError.versionMismatch(messages
                            .getMessage("errors.version.mismatch", null, locale))).build());
        }
        return Result.success(null);
    }

    public Result<Void> validateCancellation(TransferRequestModel request, Locale locale) {
        if (!TransferRequestStatus.PENDING.equals(request.getStatus())) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                    .getMessage("errors.only.pending.transfer.request.may.be.cancelled", null, locale))).build());
        }
        return Result.success(null);
    }

    public Result<CreateTransferRequestConfirmationData> validateResponsible(
            ResponsibleAndNotified responsibleAndNotified, Supplier<Optional<Boolean>> addConfirmationGetter,
            YaUserDetails currentUser, Locale locale) {
        TransferResponsible responsible = responsibleAndNotified.getResponsible();
        Optional<Boolean> addConfirmationO = addConfirmationGetter.get();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (addConfirmationO.isEmpty()) {
            errors.addError("addConfirmation", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        }
        boolean addConfirmation = addConfirmationO.orElse(false);
        boolean apply = false;
        if (responsibleAndNotified.isProviderResponsibleAutoConfirm()) {
            // See DISPENSER-3540
            addConfirmation = true;
            apply = true;
        } else {
            if (addConfirmation) {
                boolean canConfirm = responsible.getResponsible().stream()
                        .anyMatch(foldersResponsible -> foldersResponsible.getResponsible()
                                .stream().anyMatch(serviceResponsible -> serviceResponsible.getResponsibleIds()
                                        .contains(currentUser.getUser().get().getId())));
                if (!canConfirm) {
                    addConfirmation = false;
                } else {
                    if (responsible.getReserveResponsibleModel().isEmpty()) {
                        apply = responsible.getResponsible().stream()
                                .allMatch(foldersResponsible -> foldersResponsible.getResponsible()
                                        .stream().anyMatch(serviceResponsible -> serviceResponsible.getResponsibleIds()
                                                .contains(currentUser.getUser().get().getId())));
                    }
                }
            }
            boolean noOneCanConfirm = responsible.getResponsible().stream().anyMatch(foldersResponsible -> {
                long totalResponsible = 0L;
                for (ServiceResponsible serviceResponsible : foldersResponsible.getResponsible()) {
                    totalResponsible += serviceResponsible.getResponsibleIds().size();
                }
                return totalResponsible == 0L;
            }) || (responsible.getReserveResponsibleModel().isPresent()
                    && responsible.getReserveResponsibleModel().get().getResponsibleIds().isEmpty());
            if (noOneCanConfirm) {
                errors.addError(TypedError.invalid(messages
                        .getMessage("errors.can.not.collect.quorum.for.transfer.request.confirmation", null, locale)));
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        } else {
            return Result.success(CreateTransferRequestConfirmationData.builder()
                    .apply(apply)
                    .confirm(addConfirmation)
                    .user(currentUser.getUser().orElseThrow())
                    .providerResponsibleAutoConfirm(responsibleAndNotified.isProviderResponsibleAutoConfirm())
                    .build());
        }
    }

    public Result<PutTransferRequestConfirmationData> validateResponsiblePut(
            ResponsibleAndNotified responsibleAndNotified, Supplier<Optional<Boolean>> addConfirmationGetter,
            YaUserDetails currentUser, Locale locale) {
        TransferResponsible responsible = responsibleAndNotified.getResponsible();
        Optional<Boolean> addConfirmationO = addConfirmationGetter.get();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (addConfirmationO.isEmpty()) {
            errors.addError("addConfirmation", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        }
        boolean addConfirmation = addConfirmationO.orElse(false);
        UserModel user = currentUser.getUser().get();
        boolean isProviderResponsible = responsible.getProviderResponsible().stream().anyMatch(providerResponsible ->
                providerResponsible.getResponsibleId().equals(user.getId()));
        if (addConfirmation) {
            boolean canConfirm = responsible.getResponsible().stream()
                    .anyMatch(foldersResponsible -> foldersResponsible.getResponsible()
                            .stream().anyMatch(serviceResponsible -> serviceResponsible.getResponsibleIds()
                                    .contains(user.getId()))) || isProviderResponsible;
            if (!canConfirm) {
                addConfirmation = false;
            }
        }
        boolean noOneCanConfirm = responsible.getResponsible().stream().anyMatch(foldersResponsible -> {
            long totalResponsible = 0L;
            for (ServiceResponsible serviceResponsible : foldersResponsible.getResponsible()) {
                totalResponsible += serviceResponsible.getResponsibleIds().size();
            }
            return totalResponsible == 0L;
        });
        if (noOneCanConfirm && !isProviderResponsible) {
            errors.addError(TypedError.invalid(messages
                    .getMessage("errors.can.not.collect.quorum.for.transfer.request.confirmation", null, locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        } else {
            return Result.success(PutTransferRequestConfirmationData.builder()
                    .confirm(addConfirmation)
                    .user(user)
                    .build());
        }
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Result<ValidatedQuotaTransferParameters>> validateQuotaTransferParameters(
            PreValidatedQuotaTransferParameters parameters, YaUserDetails currentUser,
            Locale locale, List<FolderModel> folders, List<ServiceMinimalModel> services,
            List<ResourceModel> resources, List<UnitsEnsembleModel> unitsEnsembles, List<ProviderModel> providers,
            boolean publicApi) {
        Map<String, FolderModel> folderById = folders.stream()
                .collect(Collectors.toMap(FolderModel::getId, Function.identity()));
        Map<Long, ServiceMinimalModel> serviceById = services.stream()
                .collect(Collectors.toMap(ServiceMinimalModel::getId, Function.identity()));
        Map<String, ResourceModel> resourceById = resources.stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        Map<String, UnitsEnsembleModel> unitsEnsembleById = unitsEnsembles.stream()
                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
        Map<String, ProviderModel> providerById = providers.stream()
                .collect(Collectors.toMap(ProviderModel::getId, Function.identity()));
        ValidatedQuotaTransferParameters.Builder builder = ValidatedQuotaTransferParameters.builder();
        builder.addFolders(folders);
        builder.addServices(services);
        builder.addResources(resources);
        builder.addUnitsEnsembles(unitsEnsembles);
        builder.addProviders(providers);
        ErrorCollection.Builder errorsBuilder = ErrorCollection.builder();
        parameters.getTransfers().forEach(quotaTransfer -> {
            ValidatedQuotaTransfer.Builder quotaTransferBuilder = ValidatedQuotaTransfer.builder();
            ErrorCollection.Builder quotaTransferErrorsBuilder = ErrorCollection.builder();
            quotaTransferBuilder.index(quotaTransfer.getIndex());
            FolderModel destinationFolder = folderById.get(quotaTransfer.getDestinationFolderId());
            String folderIdFieldKey = publicApi ? ".folderId" : ".destinationFolderId";
            if (destinationFolder == null || destinationFolder.isDeleted()) {
                quotaTransferErrorsBuilder.addError("parameters.quotaTransfers."
                        + quotaTransfer.getIndex() + folderIdFieldKey, TypedError.invalid(messages
                        .getMessage("errors.folder.not.found", null, locale)));
            } else if (!publicApi && quotaTransfer.getDestinationServiceId().isPresent()
                    && !Long.valueOf(destinationFolder.getServiceId())
                            .equals(quotaTransfer.getDestinationServiceId().get())) {
                quotaTransferErrorsBuilder.addError("parameters.quotaTransfers."
                        + quotaTransfer.getIndex() + folderIdFieldKey, TypedError.invalid(messages
                        .getMessage("errors.folder.and.service.does.not.match", null, locale)));
            } else {
                quotaTransferBuilder.destinationFolder(destinationFolder);
            }
            ServiceMinimalModel destinationService = null;
            if (!publicApi && quotaTransfer.getDestinationServiceId().isPresent()) {
                destinationService = serviceById.get(quotaTransfer.getDestinationServiceId().get());
            } else if (destinationFolder != null) {
                destinationService = serviceById.get(destinationFolder.getServiceId());
            }
            validateServiceTransfers(destinationService, quotaTransfer.getTransfers(), quotaTransferErrorsBuilder,
                    quotaTransferBuilder::destinationService, "parameters.quotaTransfers." + quotaTransfer.getIndex()
                            + (publicApi ? "" : ".destinationServiceId"), locale);

            quotaTransfer.getTransfers().forEach(resourceTransfer -> {
                validateQuotaResourceTransfer(resourceTransfer, resourceById, providerById, unitsEnsembleById,
                        quotaTransferBuilder::addTransfer, quotaTransferErrorsBuilder,
                        "parameters.quotaTransfers." + quotaTransfer.getIndex() + ".resourceTransfers", locale,
                        publicApi, false);
            });
            if (quotaTransferErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(quotaTransferErrorsBuilder);
            } else {
                builder.addTransfer(quotaTransferBuilder.build());
            }
        });
        if (errorsBuilder.hasAnyErrors()) {
            return Mono.just(Result.failure(errorsBuilder.build()));
        }
        ValidatedQuotaTransferParameters validatedParameters = builder.build();
        return securityService.checkQuotaTransferReadPermissions(validatedParameters, currentUser, locale);
    }

    @SuppressWarnings("ParameterNumber")
    public Mono<Result<ValidatedReserveTransferParameters>> validateReserveTransferParameters(
            PreValidatedReserveTransfer preValidatedReserveTransfer, YaUserDetails currentUser, Locale locale,
            Optional<FolderModel> folder, Optional<FolderModel> reserveFolder, List<ServiceMinimalModel> services,
            List<ResourceModel> resources, Optional<ProviderModel> provider, List<UnitsEnsembleModel> unitsEnsembles,
            boolean publicApi) {
        Map<String, ResourceModel> resourceById = resources.stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        Map<String, UnitsEnsembleModel> unitsEnsembleById = unitsEnsembles.stream()
                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
        Map<Long, ServiceMinimalModel> serviceById = services.stream()
                .collect(Collectors.toMap(ServiceMinimalModel::getId, Function.identity()));

        ValidatedReserveTransferParameters.Builder builder = ValidatedReserveTransferParameters.builder();
        builder.addFolders(Stream.of(folder, reserveFolder).filter(Optional::isPresent).map(Optional::get)
                .collect(Collectors.toSet()));
        builder.addServices(services);
        builder.addResources(resources);
        builder.addUnitsEnsembles(unitsEnsembles);

        ErrorCollection.Builder errorsBuilder = ErrorCollection.builder();
        ValidatedReserveTransfer.Builder validatedReserveTransfer = ValidatedReserveTransfer.builder();

        if (provider.isEmpty() || provider.get().isDeleted()) {
            errorsBuilder.addError("parameters.reserveTransfer.providerId", TypedError.invalid(messages
                    .getMessage("errors.provider.not.found", null, locale)));
        } else if (!provider.get().isManaged()) {
            errorsBuilder.addError("parameters.reserveTransfer.providerId", TypedError.invalid(messages
                    .getMessage("errors.provider.is.not.managed", null, locale)));
        } else if (reserveFolder.isEmpty()) {
            errorsBuilder.addError("parameters.reserveTransfer.providerId", TypedError.invalid(messages
                    .getMessage("errors.provider.reserved.folder.not.found", null, locale)));
        } else {
            validatedReserveTransfer.provider(provider.get());
            validatedReserveTransfer.reserveFolder(reserveFolder.get());
        }

        FolderModel destinationFolder = folder.orElse(null);
        String folderIdKey = publicApi ? ".folderId" : ".destinationFolderId";
        if (destinationFolder == null || destinationFolder.isDeleted()) {
            errorsBuilder.addError("parameters.reserveTransfer" + folderIdKey, TypedError.invalid(messages
                    .getMessage("errors.folder.not.found", null, locale)));
        } else if (!publicApi && preValidatedReserveTransfer.getDestinationServiceId().isPresent()
                && !Long.valueOf(destinationFolder.getServiceId())
                        .equals(preValidatedReserveTransfer.getDestinationServiceId().get())) {
            errorsBuilder.addError("parameters.reserveTransfer" + folderIdKey, TypedError.invalid(messages
                    .getMessage("errors.folder.and.service.does.not.match", null, locale)));
        } else if (INVALID_FOLDER_TYPES.contains(destinationFolder.getFolderType())) {
            errorsBuilder.addError("parameters.reserveTransfer" + folderIdKey, TypedError.invalid(messages
                    .getMessage("errors.folder.not.support.type", null, locale)));
        } else {
            validatedReserveTransfer.destinationFolder(destinationFolder);
        }

        ServiceMinimalModel destinationService = null;
        if (!publicApi && preValidatedReserveTransfer.getDestinationServiceId().isPresent()) {
            destinationService = serviceById.get(preValidatedReserveTransfer.getDestinationServiceId().get());
        } else if (destinationFolder != null) {
            destinationService = serviceById.get(destinationFolder.getServiceId());
        }
        if (destinationService == null) {
            errorsBuilder.addError("parameters.reserveTransfer" + (publicApi ? "" : ".destinationServiceId"),
                    TypedError.invalid(messages.getMessage("errors.service.not.found", null, locale)));
        } else {
            boolean validState = AbcServiceValidator.ALLOWED_SERVICE_STATES.contains(destinationService.getState());
            boolean validReadOnlyState = destinationService.getReadOnlyState() == null ||
                    AbcServiceValidator.ALLOWED_SERVICE_READONLY_STATES.contains(
                            destinationService.getReadOnlyState());
            if (!destinationService.isExportable()) {
                errorsBuilder.addError("parameters.reserveTransfer" + (publicApi ? "" : ".destinationServiceId"),
                        TypedError.invalid(messages.getMessage("errors.service.is.non.exportable", null,
                                locale)));
            } else if (!validState || !validReadOnlyState) {
                errorsBuilder.addError("parameters.reserveTransfer" + (publicApi ? "" : ".destinationServiceId"),
                        TypedError.invalid(messages.getMessage("errors.service.bad.status", null,
                                locale)));
            } else {
                validatedReserveTransfer.destinationService(destinationService);
            }
        }

        preValidatedReserveTransfer.getResourceTransfers().forEach(resourceTransfer -> {
            ValidatedQuotaResourceTransfer.Builder resourceTransferBuilder
                    = ValidatedQuotaResourceTransfer.builder();
            resourceTransferBuilder.index(resourceTransfer.getIndex());
            ErrorCollection.Builder resourceTransferErrorsBuilder = ErrorCollection.builder();
            ResourceModel resource = resourceById.get(resourceTransfer.getResourceId());

            if (resource == null || resource.isDeleted()) {
                resourceTransferErrorsBuilder.addError("parameters.reserveTransfer.resourceTransfers."
                                + resourceTransfer.getIndex() + ".resourceId",
                        TypedError.invalid(messages.getMessage("errors.resource.not.found", null, locale)));
            } else if (!resource.getProviderId().equals(preValidatedReserveTransfer.getProviderId())) {
                resourceTransferErrorsBuilder.addError("parameters.reserveTransfer.resourceTransfers."
                                + resourceTransfer.getIndex() + ".resourceId",
                        TypedError.invalid(messages.getMessage("errors.wrong.provider.for.resource", null, locale)));
            } else if (!resource.isManaged()) {
                resourceTransferErrorsBuilder.addError("parameters.reserveTransfer.resourceTransfers."
                                + resourceTransfer.getIndex() + ".resourceId",
                        TypedError.invalid(messages.getMessage("errors.resource.not.managed", null, locale)));
            } else {
                resourceTransferBuilder.resource(resource);
                UnitsEnsembleModel unitsEnsemble = unitsEnsembleById.get(resource.getUnitsEnsembleId());
                UnitModel unit;
                if (publicApi) {
                    unit = unitsEnsemble.unitByKey(resourceTransfer.getDeltaUnitId()).orElse(null);
                } else {
                    unit = unitsEnsemble.unitById(resourceTransfer.getDeltaUnitId()).orElse(null);
                }
                if (unit == null) {
                    resourceTransferErrorsBuilder.addError("parameters.reserveTransfer.resourceTransfers."
                                    + resourceTransfer.getIndex() + (publicApi ? ".deltaUnitKey" : ".deltaUnitId"),
                            TypedError.invalid(messages.getMessage("errors.unit.not.found", null, locale)));
                } else {
                    resourceTransferBuilder.deltaUnit(unit);
                    resourceTransferBuilder.resourceUnitsEnsemble(unitsEnsemble);
                    validateDelta(resourceTransfer.getDelta(), resource, unitsEnsemble, unit,
                            "parameters.reserveTransfer.resourceTransfers." + resourceTransfer.getIndex()
                                    + ".delta", resourceTransferErrorsBuilder, resourceTransferBuilder::delta, locale,
                            false);
                }
            }
            if (resourceTransferErrorsBuilder.hasAnyErrors()) {
                errorsBuilder.add(resourceTransferErrorsBuilder);
            } else {
                validatedReserveTransfer.addTransfer(resourceTransferBuilder.build());
            }
        });

        if (errorsBuilder.hasAnyErrors()) {
            return Mono.just(Result.failure(errorsBuilder.build()));
        }

        ValidatedReserveTransferParameters validatedParameters = builder.transfer(validatedReserveTransfer.build())
                .build();
        return securityService.checkReserveTransferReadPermissions(validatedParameters, currentUser, locale);
    }

    @SuppressWarnings("ParameterNumber")
    public void validateQuotaResourceTransfer(PreValidatedQuotaResourceTransfer resourceTransfer,
                                              Map<String, ResourceModel> resourceById,
                                              Map<String, ProviderModel> providerById,
                                              Map<String, UnitsEnsembleModel> unitsEnsembleById,
                                              Consumer<ValidatedQuotaResourceTransfer> setter,
                                              ErrorCollection.Builder errors,
                                              String field, Locale locale, boolean publicApi,
                                              boolean disallowNegativeDelta) {
        ValidatedQuotaResourceTransfer.Builder resourceTransferBuilder
                = ValidatedQuotaResourceTransfer.builder();
        resourceTransferBuilder.index(resourceTransfer.getIndex());
        ErrorCollection.Builder resourceTransferErrorsBuilder = ErrorCollection.builder();
        ResourceModel resource = resourceById.get(resourceTransfer.getResourceId());
        ProviderModel provider = resource != null ? providerById.get(resource.getProviderId()) : null;
        String extendedField = field + "." + resourceTransfer.getIndex();
        if (resource == null || resource.isDeleted() || provider.isDeleted()) {
            resourceTransferErrorsBuilder.addError(extendedField + ".resourceId",
                    TypedError.invalid(messages.getMessage("errors.resource.not.found", null, locale)));
        } else if (!resource.isManaged() || !provider.isManaged()) {
            resourceTransferErrorsBuilder.addError(extendedField + ".resourceId",
                    TypedError.invalid(messages.getMessage("errors.resource.not.managed", null, locale)));
        } else if (resource.isVirtual()) {
            resourceTransferErrorsBuilder.addError(extendedField + ".resourceId",
                    TypedError.invalid(messages.getMessage("errors.resource.is.virtual", null, locale)));
        } else {
            resourceTransferBuilder.resource(resource);
            UnitsEnsembleModel unitsEnsemble = unitsEnsembleById.get(resource.getUnitsEnsembleId());
            UnitModel unit;
            if (publicApi) {
                unit = unitsEnsemble.unitByKey(resourceTransfer.getDeltaUnitId()).orElse(null);
            } else {
                unit = unitsEnsemble.unitById(resourceTransfer.getDeltaUnitId()).orElse(null);
            }
            if (unit == null) {
                resourceTransferErrorsBuilder.addError(extendedField + (publicApi ? ".deltaUnitKey" : ".deltaUnitId"),
                        TypedError.invalid(messages.getMessage("errors.unit.not.found", null, locale)));
            } else {
                resourceTransferBuilder.deltaUnit(unit);
                resourceTransferBuilder.resourceUnitsEnsemble(unitsEnsemble);
                validateDelta(resourceTransfer.getDelta(), resource, unitsEnsemble, unit,
                        extendedField + ".delta", resourceTransferErrorsBuilder,
                        resourceTransferBuilder::delta, locale, disallowNegativeDelta);
            }
        }
        if (resourceTransferErrorsBuilder.hasAnyErrors()) {
            errors.add(resourceTransferErrorsBuilder);
        } else {
            setter.accept(resourceTransferBuilder.build());
        }
    }

    public void validateServiceTransfers(ServiceMinimalModel service,
                                         List<PreValidatedQuotaResourceTransfer> serviceTransfers,
                                         ErrorCollection.Builder errors, Consumer<ServiceMinimalModel> setter,
                                         String field, Locale locale) {
        if (service == null) {
            errors.addError(field,
                    TypedError.invalid(messages.getMessage("errors.service.not.found", null, locale)));
        } else {
            boolean isRemoveProvisions = serviceTransfers.stream()
                    .allMatch(transfer -> transfer.getDelta().compareTo(BigDecimal.ZERO) < 0);
            Set<ServiceState> allowedServiceState = isRemoveProvisions ?
                    AbcServiceValidator.ALLOWED_SERVICE_STATES_FOR_QUOTA_EXPORT :
                    AbcServiceValidator.ALLOWED_SERVICE_STATES;
            Set<ServiceReadOnlyState> allowedServiceReadOnlyState = isRemoveProvisions ?
                    AbcServiceValidator.ALLOWED_SERVICE_READONLY_STATES_FOR_QUOTA_EXPORT :
                    AbcServiceValidator.ALLOWED_SERVICE_READONLY_STATES;
            boolean validState = allowedServiceState.contains(service.getState());
            boolean validReadOnlyState = service.getReadOnlyState() == null ||
                    allowedServiceReadOnlyState.contains(service.getReadOnlyState());
            if (!isRemoveProvisions && !service.isExportable()) {
                errors.addError(field,
                        TypedError.invalid(messages.getMessage("errors.service.is.non.exportable", null, locale)));
            } else if (!validState || !validReadOnlyState) {
                errors.addError(field,
                        TypedError.invalid(messages.getMessage("errors.service.bad.status", null, locale)));
            } else {
                setter.accept(service);
            }
        }
    }

    public void validateFolderWithService(FolderModel folder, Long serviceId, Consumer<FolderModel> setter,
                                          ErrorCollection.Builder errors, String field, Locale locale,
                                          boolean publicApi) {
        if (folder == null || folder.isDeleted()) {
            errors.addError(field,
                    TypedError.invalid(messages.getMessage("errors.folder.not.found", null, locale)));
        } else if (!publicApi && !Long.valueOf(folder.getServiceId()).equals(serviceId)) {
            errors.addError(field,
                    TypedError.invalid(messages.getMessage("errors.folder.and.service.does.not.match", null,
                            locale)));
        } else {
            setter.accept(folder);
        }
    }

    public void validateAccountWithFolder(@Nullable AccountModel accountModel, String folderId,
                                          Consumer<AccountModel> setter, ErrorCollection.Builder errors, String field,
                                          Locale locale, boolean publicApi) {
        if (accountModel == null || accountModel.isDeleted()) {
            errors.addError(field,
                    TypedError.invalid(messages.getMessage("errors.account.not.found", null, locale)));
        } else if (accountModel.isFreeTier()) {
            errors.addError(field,
                    TypedError.invalid(messages.getMessage("errors.account.free.tier.quota.can.not.be.transferred",
                            null, locale)));
        } else if (!accountModel.getFolderId().equals(folderId)) {
            errors.addError(field,
                    TypedError.invalid(messages.getMessage("errors.folder.not.match.account.one",
                            new Object[]{accountModel.getFolderId()}, locale)));
        } else {
            setter.accept(accountModel);
        }
    }

    public void validateProvisionTransferAccounts(AccountModel sourceAccount, AccountModel destinationAccount,
                                                  ErrorCollection.Builder errors, String field, Locale locale) {
        if (sourceAccount.getId().equals(destinationAccount.getId())) {
            errors.addError(field, TypedError.invalid(
                    messages.getMessage("errors.transfers.cannot.move.quotas.between.same.account",
                            null, locale)));
        }
        if (!sourceAccount.getProviderId().equals(destinationAccount.getProviderId())) {
            errors.addError(field, TypedError.invalid(
                    messages.getMessage("errors.transfers.accounts.have.different.providers",
                            new Object[]{DisplayUtil.getAccountDisplayStringWithId(sourceAccount),
                                    DisplayUtil.getAccountDisplayStringWithId(destinationAccount)}, locale)));
        }
        if (!sourceAccount.getAccountsSpacesId().equals(destinationAccount.getAccountsSpacesId())) {
            errors.addError(field, TypedError.invalid(
                    messages.getMessage("errors.transfers.accounts.have.different.accounts.spaces",
                            new Object[]{DisplayUtil.getAccountDisplayStringWithId(sourceAccount),
                                    DisplayUtil.getAccountDisplayStringWithId(destinationAccount)}, locale)));
        }
    }

    public void validateAccountResource(AccountModel account, @Nullable ResourceModel resource,
                                        ErrorCollection.Builder errors, String field, Locale locale) {
        if (resource == null) {
            return;
        }
        if (!account.getProviderId().equals(resource.getProviderId())) {
            errors.addError(field + ".resourceId", TypedError.invalid(
                    messages.getMessage("errors.transfers.resource.have.different.provider.from.account",
                            null, locale)));
        }
        if (!Objects.equals(account.getAccountsSpacesId().orElse(null), resource.getAccountsSpacesId())) {
            errors.addError(field + ".resourceId", TypedError.invalid(
                    messages.getMessage("errors.transfers.resource.have.different.accounts.space.from.account",
                            null, locale)));
        }
    }

    public void validateExchangeTransfer(List<PreValidatedProvisionTransfer> provisionTransfers,
                                         Map<String, AccountModel> accountsById,
                                         ErrorCollection.Builder errors, String field, Locale locale) {
        Set<String> sourceFolderIds = new HashSet<>();
        Set<String> destinationFolderIds = new HashSet<>();
        Set<PreValidatedProvisionTransfer.Key> provisionTransferKeys = new HashSet<>();
        for (PreValidatedProvisionTransfer provisionTransfer : provisionTransfers) {
            sourceFolderIds.add(provisionTransfer.getSourceFolderId());
            destinationFolderIds.add(provisionTransfer.getDestinationFolderId());
            provisionTransferKeys.add(new PreValidatedProvisionTransfer.Key(provisionTransfer.getSourceAccountId(),
                    provisionTransfer.getDestinationAccountId(), provisionTransfer.getSourceFolderId(),
                    provisionTransfer.getDestinationFolderId()));
        }
        boolean isExchangeTransfer = provisionTransfers.size() == 2 &&
                sourceFolderIds.size() == 2 &&
                sourceFolderIds.equals(destinationFolderIds) &&
                provisionTransfers.stream()
                        .allMatch(t ->
                                (accountsById.containsKey(t.getSourceAccountId()) &&
                                        accountsById.get(t.getSourceAccountId()).getReserveType().isPresent()) ^
                                (accountsById.containsKey(t.getDestinationAccountId()) &&
                                        accountsById.get(t.getDestinationAccountId()).getReserveType().isPresent()));
        if (!isExchangeTransfer) {
            if (sourceFolderIds.size() > 1 || destinationFolderIds.size() > 1) {
                errors.addError(field, TypedError.invalid(
                        messages.getMessage("errors.transfers.more.than.two.folders.on.provision.transfer",
                                null, locale)));
            }
            if (provisionTransferKeys.size() < provisionTransfers.size()) {
                errors.addError(field, TypedError.invalid(
                        messages.getMessage("errors.transfers.duplicated.provision.transfers.are.not.allowed",
                                null, locale)
                ));
            }
        }
    }

    public void validateReserveTransfer(List<PreValidatedProvisionTransfer> provisionTransfers,
                                        Map<String, AccountModel> accountsById,
                                        ErrorCollection.Builder errors, String field, Locale locale) {
        Set<String> accountIds = new HashSet<>();
        for (PreValidatedProvisionTransfer provisionTransfer : provisionTransfers) {
            accountIds.add(provisionTransfer.getSourceAccountId());
            accountIds.add(provisionTransfer.getDestinationAccountId());
        }
        if (accountIds.stream().anyMatch(accountId -> !accountsById.containsKey(accountId))) {
            return;
        }

        boolean existReserveAccount = accountIds.stream()
                .anyMatch(accountId -> accountsById.get(accountId).getReserveType().isPresent());
        int countProviders = accountIds.stream()
                .map(accountId -> accountsById.get(accountId).getProviderId())
                .collect(Collectors.toSet())
                .size();
        if (existReserveAccount && countProviders > 1) {
            errors.addError(field, TypedError.invalid(
                    messages.getMessage("errors.transfers.more.than.one.providers.on.provision.reserve.transfer",
                            null, locale)));
        }
    }

    public void duplicateAccountResourceError(AccountModel accountModel, ErrorCollection.Builder errors, String field,
                                              Locale locale) {
        errors.addError(field + ".resourceId", TypedError.invalid(
                messages.getMessage("errors.transfers.resource.for.account.0.already.used.in.another.transfer",
                        new Object[]{DisplayUtil.getAccountDisplayStringWithId(accountModel)}, locale)));
    }

    public Mono<Result<QuotaTransferPreApplicationData>> validateQuotaTransferApplication(
            YdbTxSession txSession, ValidatedQuotaTransferParameters parameters, Locale locale,
            boolean delayValidation
    ) {
        Map<String, BigInteger> sumPerResource = new HashMap<>();
        parameters.getTransfers().forEach(quotaTransfer -> quotaTransfer.getTransfers().forEach(resourceTransfer -> {
            BigInteger currentSum = sumPerResource.getOrDefault(resourceTransfer.getResource().getId(),
                    BigInteger.ZERO);
            sumPerResource.put(resourceTransfer.getResource().getId(),
                    currentSum.add(BigInteger.valueOf(resourceTransfer.getDelta())));
        }));
        if (sumPerResource.values().stream().anyMatch(v -> v.compareTo(BigInteger.ZERO) != 0)) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                    .getMessage("errors.transfer.request.has.non.zero.quota.sum", null, locale)))
                    .build()));
        }
        Set<String> folderIds = new HashSet<>();
        parameters.getTransfers().forEach(t -> folderIds.add(t.getDestinationFolder().getId()));
        return storeService.loadQuotas(txSession, folderIds).map(quotas -> {
            Map<String, Map<String, QuotaModel>> quotasByFolderResource = new HashMap<>();
            quotas.forEach(quota -> quotasByFolderResource.computeIfAbsent(quota.getFolderId(),
                    k -> new HashMap<>()).put(quota.getResourceId(), quota));
            ErrorCollection.Builder errors = ErrorCollection.builder();
            parameters.getTransfers().forEach(quotaTransfer -> {
                quotaTransfer.getTransfers().forEach(resourceTransfer -> {
                    QuotaModel currentQuota = quotasByFolderResource
                            .getOrDefault(quotaTransfer.getDestinationFolder().getId(), Collections.emptyMap())
                            .getOrDefault(resourceTransfer.getResource().getId(), null);
                    validateFolderQuotaTransfer(resourceTransfer, currentQuota, errors,
                            "parameters.quotaTransfers." + quotaTransfer.getIndex() + ".resourceTransfers", locale,
                            delayValidation);
                });
            });
            if (errors.hasAnyErrors()) {
                return Result.failure(errors.build());
            }
            return Result.success(QuotaTransferPreApplicationData.builder().addQuotas(quotas).build());
        });
    }

    public Mono<Result<QuotaTransferPreApplicationData>> validateReserveTransferApplication(
            YdbTxSession txSession, ValidatedReserveTransferParameters parameters, Locale locale,
            boolean delayValidation
    ) {
        String reserveFolderId = parameters.getTransfer().getReserveFolder().getId();
        String destinationFolderId = parameters.getTransfer().getDestinationFolder().getId();
        Set<String> folderIds = Set.of(reserveFolderId, destinationFolderId);
        return storeService.loadQuotas(txSession, folderIds).map(quotas -> {
            ErrorCollection.Builder errors = ErrorCollection.builder();
            Map<String, Map<String, QuotaModel>> quotasByFolderResource = new HashMap<>();
            quotas.forEach(quota -> quotasByFolderResource.computeIfAbsent(quota.getFolderId(),
                    k -> new HashMap<>()).put(quota.getResourceId(), quota));
            parameters.getTransfer().getTransfers().forEach(resourceTransfer -> {
                QuotaModel currentReserveQuota = quotasByFolderResource
                        .getOrDefault(reserveFolderId, Collections.emptyMap())
                        .getOrDefault(resourceTransfer.getResource().getId(), null);
                validateReserveTransferQuotaInReserveFolder(currentReserveQuota, resourceTransfer, errors, locale,
                        delayValidation);
                QuotaModel currentQuota = quotasByFolderResource
                        .getOrDefault(destinationFolderId, Collections.emptyMap())
                        .getOrDefault(resourceTransfer.getResource().getId(), null);
                validateReserveTransferQuotaInFolder(currentQuota, resourceTransfer, errors, locale);
            });
            if (errors.hasAnyErrors()) {
                return Result.failure(errors.build());
            }
            return Result.success(QuotaTransferPreApplicationData.builder().addQuotas(quotas).build());
        });
    }

    public <T> Result<T> validateQuotaTransfersSum(Stream<ValidatedQuotaResourceTransfer> transfers,
                                                   Locale locale) {
        Map<String, BigInteger> sumByResourceId = new HashMap<>();
        transfers
                .forEach(t -> {
                    ResourceModel resource = t.getResource();
                    sumByResourceId.merge(resource.getId(), BigInteger.valueOf(t.getDelta()), BigInteger::add);
                });
        if (sumByResourceId.values().stream().anyMatch(v -> v.compareTo(BigInteger.ZERO) != 0)) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                            .getMessage("errors.transfer.request.has.non.zero.quota.sum", null, locale)))
                    .build());
        }
        return Result.success(null);
    }

    @SuppressWarnings("ParameterNumber")
    public void validateAccountQuotaTransfer(ValidatedQuotaResourceTransfer resourceTransfer,
                                             AccountsQuotasModel accountsQuotasModel,
                                             @Nullable ValidatedReceivedProvision receivedProvision,
                                             ProviderModel provider,
                                             ErrorCollection.Builder errors, String field, Locale locale,
                                             boolean delayValidation) {
        String extendedField = field + "." + resourceTransfer.getIndex();
        long currentProvided;
        long currentFrozen = getFrozenQuota(accountsQuotasModel);
        long currentAllocated;
        boolean allocatedSupported = ProviderUtil.isAllocatedSupported(provider, resourceTransfer.getResource());
        if (receivedProvision != null) {
            currentProvided = receivedProvision.getProvidedAmount();
            currentAllocated = allocatedSupported ? receivedProvision.getAllocatedAmount() : 0L;
        } else {
            currentProvided = getProvided(accountsQuotasModel);
            currentAllocated = allocatedSupported ? getAllocated(accountsQuotasModel) : 0L;
        }
        long available = Math.max(currentProvided - currentFrozen - currentAllocated, 0L);

        long delta = resourceTransfer.getDelta();
        if (!delayValidation && delta < 0 && Math.abs(delta) > available) {
            String errorMessage = transferRequestErrorMessages.getProvidedQuotaLessThanRequestedErrorMessage(
                    resourceTransfer, currentProvided, currentAllocated, currentFrozen, locale);
            errors.addError(extendedField + ".delta",
                    TypedError.invalid(errorMessage));
            if (available > 0) {
                addSuggestedAmountAccount(errors, available, resourceTransfer, allocatedSupported, locale);
            }
        }
        if (delta == 0L || Units.add(currentProvided, delta).isEmpty()) {
            errors.addError(extendedField + ".delta",
                    TypedError.invalid(messages.getMessage("errors.number.out.of.range", null, locale)));
        }
    }

    public void validateFolderQuotaTransfer(ValidatedQuotaResourceTransfer resourceTransfer,
                                            QuotaModel currentQuota, ErrorCollection.Builder errors,
                                            String field, Locale locale, boolean delayValidation
    ) {
        String extendedField = field + "." + resourceTransfer.getIndex();
        long currentBalance = getBalance(currentQuota);
        long currentAmount = getAmount(currentQuota);
        long currentProvided = currentAmount != 0L ? (currentAmount - currentBalance) : 0L;
        long delta = resourceTransfer.getDelta();
        if (!delayValidation && delta < 0 && Math.abs(delta) > currentBalance) {
            String errorMessage = transferRequestErrorMessages.getNotEnoughBalanceErrorMessage(
                    resourceTransfer, currentBalance, currentProvided, locale);
            errors.addError(extendedField + ".delta", TypedError.invalid(errorMessage));
            if (currentBalance > 0) {
                addSuggestedAmount(errors, currentBalance, resourceTransfer, locale);
            }
        }
        if (delta == 0L || Units.add(currentAmount, delta).isEmpty()
                || Units.add(currentBalance, delta).isEmpty()) {
            errors.addError(extendedField + ".delta",
                    TypedError.invalid(messages.getMessage("errors.number.out.of.range", null, locale)));
        }
    }

    public void validateQuotaTransfer(ResourceQuotaTransfer resourceTransfer, QuotaModel currentQuota,
                                      ResourceModel resource, FolderModel folder, UnitsEnsembleModel unitsEnsemble,
                                      ErrorCollection.Builder errorsEn, ErrorCollection.Builder errorsRu) {
        long currentBalance = currentQuota != null
                ? (currentQuota.getBalance() != null ? currentQuota.getBalance() : 0L) : 0L;
        long currentAmount = currentQuota != null
                ? (currentQuota.getQuota() != null ? currentQuota.getQuota() : 0L) : 0L;
        long currentProvided = currentAmount != 0L ? (currentAmount - currentBalance) : 0L;
        long delta = resourceTransfer.getDelta();
        if (delta < 0 && Math.abs(delta) > currentBalance) {
            AmountDto balanceRu = QuotasHelper.getAmountDto(
                    QuotasHelper.toBigDecimal(currentBalance), resource, unitsEnsemble, Locales.RUSSIAN);
            AmountDto balanceEn = QuotasHelper.getAmountDto(
                    QuotasHelper.toBigDecimal(currentBalance), resource, unitsEnsemble, Locales.ENGLISH);
            AmountDto providedRu = QuotasHelper.getAmountDto(
                    QuotasHelper.toBigDecimal(currentProvided), resource, unitsEnsemble, Locales.RUSSIAN);
            AmountDto providedEn = QuotasHelper.getAmountDto(
                    QuotasHelper.toBigDecimal(currentProvided), resource, unitsEnsemble, Locales.ENGLISH);
            String messageEn = messages
                    .getMessage("errors.not.enough.balance.for.quota.transfer.of.resource.0.in.folder.1",
                            new Object[]{resource.getNameEn(),
                                    folder.getDisplayName()},
                            Locales.ENGLISH)
                    + (currentProvided != 0L ? "\n" + messages
                    .getMessage("errors.not.enough.balance.for.quota.transfer.non.zero.provided",
                            new Object[]{providedEn.getReadableAmount() + " " + providedEn.getReadableUnit()},
                            Locales.ENGLISH) : "")
                    + (currentBalance != 0L  ? "\n" + messages
                    .getMessage("errors.not.enough.balance.for.quota.transfer.non.zero.balance",
                            new Object[]{balanceEn.getReadableAmount() + " " + balanceEn.getReadableUnit()},
                            Locales.ENGLISH) : "");
            String messageRu = messages
                    .getMessage("errors.not.enough.balance.for.quota.transfer.of.resource.0.in.folder.1",
                            new Object[]{resource.getNameRu(),
                                    folder.getDisplayName()},
                            Locales.RUSSIAN)
                    + (currentProvided != 0L ? "\n" + messages
                    .getMessage("errors.not.enough.balance.for.quota.transfer.non.zero.provided",
                            new Object[]{providedRu.getReadableAmount() + " " + providedRu.getReadableUnit()},
                            Locales.RUSSIAN) : "")
                    + (currentBalance != 0L  ? "\n" + messages
                    .getMessage("errors.not.enough.balance.for.quota.transfer.non.zero.balance",
                            new Object[]{balanceRu.getReadableAmount() + " " + balanceRu.getReadableUnit()},
                            Locales.RUSSIAN) : "");
            String buttonTextEn = messages.getMessage(
                    "errors.not.enough.balance.for.quota.transfer.non.zero.balance.button",
                    null, Locales.ENGLISH);
            String buttonTextRu = messages.getMessage(
                    "errors.not.enough.balance.for.quota.transfer.non.zero.balance.button",
                    null, Locales.RUSSIAN);
            errorsEn.addError(TypedError.invalid(messageEn));
            errorsRu.addError(TypedError.invalid(messageRu));

            if (currentBalance > 0) {
                errorsEn.addDetail(resource.getId() + ".suggestedAmount", balanceEn);
                errorsRu.addDetail(resource.getId() + ".suggestedAmount", balanceRu);
                errorsEn.addDetail("suggestedAmountPrompt", buttonTextEn);
                errorsRu.addDetail("suggestedAmountPrompt", buttonTextRu);
            }
        }
        if (delta == 0L || Units.add(currentAmount, delta).isEmpty()
                || Units.add(currentBalance, delta).isEmpty()) {
            errorsEn.addError(TypedError.invalid(messages
                    .getMessage("errors.number.out.of.range.for.quota.transfer.of.resource.0.in.folder.1",
                            new Object[] {resource.getNameEn(),
                                    folder.getDisplayName()},
                            Locales.ENGLISH)));
            errorsRu.addError(TypedError.invalid(messages
                    .getMessage("errors.number.out.of.range.for.quota.transfer.of.resource.0.in.folder.1",
                            new Object[] {resource.getNameRu(),
                                    folder.getDisplayName()},
                            Locales.RUSSIAN)));
        }
    }

    @SuppressWarnings("ParameterNumber")
    public void validateAccountQuotaTransfer(
            ResourceQuotaTransfer resourceTransfer, @Nullable AccountsQuotasModel currentQuota,
            @Nullable ValidatedReceivedProvision receivedProvision, ResourceModel resource,
            AccountModel account, ProviderModel providerModel, UnitsEnsembleModel unitsEnsemble,
            ErrorCollection.Builder errorsEn, ErrorCollection.Builder errorsRu, boolean checkAvailableQuota
    ) {
        boolean allocatedSupported = ProviderUtil.isAllocatedSupported(providerModel, resource);
        long currentProvided;
        long currentAllocated;
        if (receivedProvision != null) {
            currentProvided = receivedProvision.getProvidedAmount();
            currentAllocated = allocatedSupported ? receivedProvision.getAllocatedAmount() : 0L;
        } else {
            currentProvided = getProvided(currentQuota);
            currentAllocated = allocatedSupported ? getAllocated(currentQuota) : 0L;
        }
        long frozenQuota = getFrozenQuota(currentQuota);
        long delta = resourceTransfer.getDelta();
        long currentAvailable = Math.max(currentProvided - currentAllocated - frozenQuota, 0L);
        if (checkAvailableQuota && delta < 0 && Math.abs(delta) > currentAvailable) {
            fillAccountQuotaOverCommitErrors(errorsEn, account, resource, unitsEnsemble, currentProvided,
                    currentAllocated, frozenQuota, Locales.ENGLISH);
            fillAccountQuotaOverCommitErrors(errorsRu, account, resource, unitsEnsemble, currentProvided,
                    currentAllocated, frozenQuota, Locales.RUSSIAN);
        }
        if (delta == 0L || Units.add(currentProvided, delta).isEmpty()) {
            String accountDisplayString = DisplayUtil.getAccountDisplayStringWithId(account);
            errorsEn.addError(TypedError.invalid(messages
                    .getMessage("errors.number.out.of.range.for.provision.transfer.of.resource.0.in.account.1",
                            new Object[] {resource.getNameEn(), accountDisplayString},
                            Locales.ENGLISH)));
            errorsRu.addError(TypedError.invalid(messages
                    .getMessage("errors.number.out.of.range.for.provision.transfer.of.resource.0.in.account.1",
                            new Object[] {resource.getNameRu(), accountDisplayString},
                            Locales.RUSSIAN)));
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void fillAccountQuotaOverCommitErrors(
            ErrorCollection.Builder errors,
            AccountModel account,
            ResourceModel resource,
            UnitsEnsembleModel unitsEnsemble,
            long currentProvided,
            long currentAllocated,
            long currentFrozen,
            Locale locale
    ) {
        long currentAvailable = Math.max(currentProvided - currentAllocated - currentFrozen, 0L);
        AmountDto availableDto = QuotasHelper.getAmountDto(
                QuotasHelper.toBigDecimal(currentAvailable), resource, unitsEnsemble, locale);
        AmountDto frozenDto = QuotasHelper.getAmountDto(
                QuotasHelper.toBigDecimal(currentFrozen), resource, unitsEnsemble, locale);
        AmountDto providedDto = QuotasHelper.getAmountDto(QuotasHelper.toBigDecimal(currentProvided), resource,
                unitsEnsemble, locale);
        AmountDto allocatedDto = QuotasHelper.getAmountDto(QuotasHelper.toBigDecimal(currentAllocated), resource,
                unitsEnsemble, locale);
        String accountString = DisplayUtil.getAccountDisplayString(account);
        String message = messages
                .getMessage("errors.not.enough.available.for.provision.transfer.of.resource.0.in.account.1",
                        new Object[]{resource.getNameEn(), accountString},
                        locale)
                + ("\n" + transferRequestErrorMessages.getProvidedMessage(providedDto, locale))
                + (currentAllocated != 0L ? "\n" + transferRequestErrorMessages.getAllocatedMessage(allocatedDto,
                    locale) : "")
                + (currentFrozen != 0L ? "\n" + transferRequestErrorMessages.getFrozenMessage(frozenDto,
                    locale) : "")
                + (currentAvailable != 0L  ? "\n" + messages
                    .getMessage("errors.not.enough.available.for.provision.transfer.non.zero.available",
                        new Object[]{availableDto.getReadableAmount() + " " + availableDto.getReadableUnit()},
                        locale) : "");
        String buttonTextEn = messages.getMessage(
                "errors.not.enough.available.for.provision.transfer.provided.button",
                null, locale);
        errors.addError(TypedError.invalid(message));

        if (currentAvailable > 0) {
            errors.addDetail(resource.getId() + ".suggestedProvisionAmount", availableDto);
            errors.addDetail("suggestedProvisionAmountPrompt", buttonTextEn);
        }
    }

    public void validateMoveProvisionSupportForProvider(ProviderModel provider,
                                                        ErrorCollection.Builder errors,
                                                        String field,
                                                        Locale locale) {
        if (!provider.getAccountsSettings().isMoveProvisionSupported()) {
            String providerName = Locales.selectObject(provider.getNameEn(), provider.getNameRu(), locale);
            errors.addError(field, TypedError.invalid(messages.getMessage(
                    "errors.provider.move.provision.is.not.supported.0", new Object[]{providerName}, locale)));
        }
    }

    private void validateReserveTransferQuotaInReserveFolder(
            QuotaModel quotaModel,
            ValidatedQuotaResourceTransfer resourceTransfer,
            ErrorCollection.Builder errors,
            Locale locale,
            boolean delayValidation
    ) {
        long currentBalance = getBalance(quotaModel);
        long currentAmount = getAmount(quotaModel);
        long delta = resourceTransfer.getDelta();
        long currentProvided = currentAmount != 0L ? (currentAmount - currentBalance) : 0L;
        if (!delayValidation && delta > currentBalance) {
            String errorMessage = transferRequestErrorMessages.getNotEnoughBalanceErrorMessage(
                    resourceTransfer, currentBalance, currentProvided, locale);
            errors.addError("parameters.reserveTransfer.resourceTransfers." + resourceTransfer.getIndex()
                            + ".delta",
                    TypedError.invalid(errorMessage));
            if (currentBalance > 0) {
                addSuggestedAmount(errors, currentBalance, resourceTransfer, locale);
            }
        }
    }

    private void validateReserveTransferQuotaInFolder(
            QuotaModel quotaModel,
            ValidatedQuotaResourceTransfer resourceTransfer,
            ErrorCollection.Builder errors,
            Locale locale
    ) {
        long currentBalance = getBalance(quotaModel);
        long currentAmount = getAmount(quotaModel);
        long delta = resourceTransfer.getDelta();
        if (Units.add(currentAmount, delta).isEmpty()
                || Units.add(currentBalance, delta).isEmpty()) {
            errors.addError("parameters.reserveTransfer.resourceTransfers." + resourceTransfer.getIndex() + ".delta",
                    TypedError.invalid(messages.getMessage("errors.number.out.of.range", null, locale)));
        }
    }

    static long getBalance(QuotaModel quotaModel) {
        return quotaModel != null
                ? (quotaModel.getBalance() != null ? quotaModel.getBalance() : 0L) : 0L;
    }

    static long getAmount(QuotaModel quotaModel) {
        return quotaModel != null
                ? (quotaModel.getQuota() != null ? quotaModel.getQuota() : 0L) : 0L;
    }

    static long getProvided(AccountsQuotasModel accountsQuotasModel) {
        return accountsQuotasModel != null
                ? (accountsQuotasModel.getProvidedQuota() != null ? accountsQuotasModel.getProvidedQuota() : 0L) : 0L;
    }

    static long getAllocated(AccountsQuotasModel accountsQuotasModel) {
        return accountsQuotasModel != null
                ? (accountsQuotasModel.getAllocatedQuota() != null ? accountsQuotasModel.getAllocatedQuota() : 0L) : 0L;
    }

    static long getFrozenQuota(AccountsQuotasModel accountsQuotasModel) {
        return accountsQuotasModel != null
                ? accountsQuotasModel.getFrozenProvidedQuota() : 0L;
    }

    public Mono<Result<ValidatedTransferRequestsSearchParameters>> validateSearchParameters(
            YdbSession session, TransferRequestsSearchParameters parameters, YaUserDetails currentUser,
            Locale locale) {
        int totalFilters = (parameters.getFilterByCurrentUser().isPresent()
                && parameters.getFilterByCurrentUser().get() ? 1 : 0)
                + (parameters.getFilterByFolderId().isPresent() ? 1 : 0)
                + (parameters.getFilterByServiceId().isPresent() ? 1 : 0);
        if (totalFilters != 1) {
            return Mono.just(Result.failure(ErrorCollection.builder()
                    .addError(TypedError.invalid(messages
                            .getMessage("errors.invalid.transfer.request.filter", null, locale))).build()));
        }
        if (parameters.getFilterByCurrentUser().isPresent()
                && parameters.getFilterByCurrentUser().get() && currentUser.getUser().isEmpty()) {
            return Mono.just(Result.failure(ErrorCollection.builder()
                    .addError(TypedError.invalid(messages
                            .getMessage("errors.invalid.transfer.request.filter", null, locale))).build()));
        }
        return getFilterFolder(session, parameters, currentUser, locale)
                .flatMap(folderR -> getFilterService(session, parameters, currentUser, locale).map(serviceR -> {
                    ErrorCollection.Builder errors = ErrorCollection.builder();
                    ValidatedTransferRequestsSearchParameters.Builder builder
                            = ValidatedTransferRequestsSearchParameters.builder();
                    folderR.doOnSuccess(folderO -> folderO.ifPresent(builder::filterByFolder));
                    folderR.doOnFailure(errors::add);
                    serviceR.doOnSuccess(serviceO -> serviceO.ifPresent(builder::filterByService));
                    serviceR.doOnFailure(errors::add);
                    parameters.getFilterByCurrentUser().ifPresent(builder::filterByCurrentUser);
                    Result<Set<TransferRequestStatus>> statusR = validateFilterStatuses(parameters, locale);
                    statusR.doOnSuccess(statuses -> statuses.forEach(builder::addFilterByStatus));
                    statusR.doOnFailure(errors::add);
                    if (errors.hasAnyErrors()) {
                        return Result.failure(errors.build());
                    }
                    return Result.success(builder.build());
                }));
    }

    public Result<VoteType> validateVoteParameters(FrontTransferRequestVotingDto voteParameters, Locale locale) {
        if (voteParameters.getVoteType().isEmpty()) {
            return Result.failure(ErrorCollection.builder().addError("voteType", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale))).build());
        }
        return switch (voteParameters.getVoteType().get()) {
            case CONFIRM -> Result.success(VoteType.CONFIRM);
            case REJECT -> Result.success(VoteType.REJECT);
            case ABSTAIN -> Result.success(VoteType.ABSTAIN);
            default -> Result.failure(ErrorCollection.builder().addError("voteType", TypedError.invalid(messages
                    .getMessage("errors.invalid.transfer.request.vote.type", null, locale))).build());
        };
    }

    private Result<Set<TransferRequestStatus>> validateFilterStatuses(TransferRequestsSearchParameters parameters,
                                                                      Locale locale) {
        if (parameters.getFilterByStatus().isEmpty() || parameters.getFilterByStatus().get().isEmpty()) {
            return Result.success(Set.of());
        }
        boolean unknownStatus = parameters.getFilterByStatus().get().stream()
                .anyMatch(TransferRequestStatusDto.UNKNOWN::equals);
        Set<TransferRequestStatus> statusSet = new HashSet<>();
        parameters.getFilterByStatus().get().forEach(status -> {
            switch (status) {
                case PENDING -> statusSet.add(TransferRequestStatus.PENDING);
                case APPLIED -> statusSet.add(TransferRequestStatus.APPLIED);
                case REJECTED -> statusSet.add(TransferRequestStatus.REJECTED);
                case CANCELLED -> statusSet.add(TransferRequestStatus.CANCELLED);
                case FAILED -> statusSet.add(TransferRequestStatus.FAILED);
                case STALE -> statusSet.add(TransferRequestStatus.STALE);
                case EXECUTING -> statusSet.add(TransferRequestStatus.EXECUTING);
                case PARTLY_APPLIED -> statusSet.add(TransferRequestStatus.PARTLY_APPLIED);
                default -> { }
            }
        });
        boolean duplicateStatuses = statusSet.size() != parameters.getFilterByStatus().get().size();
        if (unknownStatus || duplicateStatuses) {
            return Result.failure(ErrorCollection.builder().addError("filterByStatus",
                    TypedError.invalid(messages
                            .getMessage("errors.invalid.transfer.request.status", null, locale))).build());
        }
        return Result.success(statusSet);
    }

    private Mono<Result<Optional<FolderModel>>> getFilterFolder(YdbSession session,
                                                                TransferRequestsSearchParameters parameters,
                                                                YaUserDetails currentUser,
                                                                Locale locale) {
        if (parameters.getFilterByFolderId().isEmpty()) {
            return Mono.just(Result.success(Optional.empty()));
        }
        if (!Uuids.isValidUuid(parameters.getFilterByFolderId().get())) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError("filterByFolderId",
                    TypedError.invalid(messages
                            .getMessage("errors.folder.not.found", null, locale))).build()));
        }
        return storeService.getFolder(immediateTx(session),
                parameters.getFilterByFolderId().get()).flatMap(folderO -> {
            if (folderO.isEmpty()) {
                return Mono.just(Result.failure(ErrorCollection.builder().addError("filterByFolderId",
                        TypedError.invalid(messages
                                .getMessage("errors.folder.not.found", null, locale))).build()));
            }
            return securityManagerService.checkReadPermissions(folderO.get().getId(), currentUser, locale,
                    null).map(result -> {
                if (result.isFailure()) {
                    return Result.failure(ErrorCollection.builder().addError("filterByFolderId",
                            TypedError.invalid(messages
                                    .getMessage("errors.access.denied", null, locale))).build());
                }
                return Result.success(folderO);
            });
        });
    }

    private YdbTxSession immediateTx(YdbSession session) {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY);
    }

    private Mono<Result<Optional<ServiceMinimalModel>>> getFilterService(YdbSession session,
                                                                         TransferRequestsSearchParameters parameters,
                                                                         YaUserDetails currentUser,
                                                                         Locale locale) {
        if (parameters.getFilterByServiceId().isEmpty()) {
            return Mono.just(Result.success(Optional.empty()));
        }
        long serviceId;
        try {
            serviceId = Long.parseLong(parameters.getFilterByServiceId().get());
        } catch (NumberFormatException e) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError("filterByServiceId",
                    TypedError.invalid(messages
                            .getMessage("errors.service.not.found", null, locale))).build()));
        }
        return storeService.getService(immediateTx(session), serviceId).flatMap(serviceO -> {
            if (serviceO.isEmpty()) {
                return Mono.just(Result.failure(ErrorCollection.builder().addError("filterByServiceId",
                        TypedError.invalid(messages
                                .getMessage("errors.service.not.found", null, locale))).build()));
            }
            return securityManagerService.checkReadPermissions(serviceO.get().getId(), currentUser, locale,
                    null).map(result -> {
                if (result.isFailure()) {
                    return Result.failure(ErrorCollection.builder().addError("filterByServiceId",
                            TypedError.invalid(messages
                                    .getMessage("errors.access.denied", null, locale))).build());
                }
                return Result.success(serviceO);
            });
        });
    }

    private void validateServiceId(Supplier<Optional<String>> getter, Consumer<Long> setter,
                                   ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<String> serviceIdO = getter.get();
        validateServiceId(serviceIdO.orElse(null), setter, errors, fieldKey, locale);
    }

    private void validateServiceId(@Nullable String value, Consumer<Long> setter,
                                   ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        if (value == null) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        try {
            long serviceId = Long.parseLong(value);
            setter.accept(serviceId);
        } catch (Exception e) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.service.not.found", null, locale)));
        }
    }

    private void validateAccountId(Supplier<Optional<String>> getter, Consumer<String> setter,
                                  ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        validateUuid(getter, setter, errors, fieldKey, locale, "errors.account.not.found");
    }

    private void validateAccountId(@Nullable String value, Consumer<String> setter,
                                   ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        validateUuid(value, setter, errors, fieldKey, locale, "errors.account.not.found");
    }

    private void validateFolderId(Supplier<Optional<String>> getter, Consumer<String> setter,
                                  ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        validateUuid(getter, setter, errors, fieldKey, locale, "errors.folder.not.found");
    }

    private void validateFolderId(@Nullable String value, Consumer<String> setter,
                                  ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        validateUuid(value, setter, errors, fieldKey, locale, "errors.folder.not.found");
    }

    private void validateProviderId(Supplier<Optional<String>> getter, Consumer<String> setter,
                                    ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        validateUuid(getter, setter, errors, fieldKey, locale, "errors.provider.not.found");
    }

    private void validateProviderId(@Nullable String value, Consumer<String> setter,
                                    ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        validateUuid(value, setter, errors, fieldKey, locale, "errors.provider.not.found");
    }

    private void validateUuid(Supplier<Optional<String>> getter, Consumer<String> setter,
                              ErrorCollection.Builder errors, String fieldKey, Locale locale, String error) {
        Optional<String> uuidO = getter.get();
        validateUuid(uuidO.orElse(null), setter, errors, fieldKey, locale, error);
    }

    private void validateUuid(@Nullable String value, Consumer<String> setter,
                              ErrorCollection.Builder errors, String fieldKey, Locale locale, String error) {
        if (value == null) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        if (!Uuids.isValidUuid(value)) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage(error, null, locale)));
            return;
        }
        setter.accept(value);
    }

    private void validateResourceId(Supplier<Optional<String>> getter, Consumer<String> setter,
                                    ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<String> resourceIdO = getter.get();
        if (resourceIdO.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        if (!Uuids.isValidUuid(resourceIdO.get())) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)));
            return;
        }
        setter.accept(resourceIdO.get());
    }

    private void validateUnitId(Supplier<Optional<String>> getter, Consumer<String> setter,
                                ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<String> unitIdO = getter.get();
        if (unitIdO.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        if (!Uuids.isValidUuid(unitIdO.get())) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.unit.not.found", null, locale)));
            return;
        }
        setter.accept(unitIdO.get());
    }

    private void validateUnitKey(Supplier<Optional<String>> getter, Consumer<String> setter,
                                ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<String> unitIdO = getter.get();
        if (unitIdO.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        if (unitIdO.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.unit.not.found", null, locale)));
            return;
        }
        setter.accept(unitIdO.get());
    }

    private void validateDelta(Supplier<Optional<String>> getter, Consumer<BigDecimal> setter,
                               ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<String> deltaO = getter.get();
        if (deltaO.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }

        try {
            BigDecimal t = FrontStringUtil.toBigDecimal(deltaO.get());
            setter.accept(t);
        } catch (Exception e) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.number.invalid.format", null, locale)));
        }
    }

    private void validateOptionalText(Supplier<Optional<String>> getter, Consumer<String> setter,
                                      ErrorCollection.Builder errors, String fieldKey, int maxLength, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            setter.accept(null);
        } else if (text.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    @SuppressWarnings("ParameterNumber")
    private void validateDelta(BigDecimal deltaInput, ResourceModel resource, UnitsEnsembleModel unitsEnsemble,
                               UnitModel unit, String fieldKey, ErrorCollection.Builder errors,
                               Consumer<Long> deltaConsumer, Locale locale, boolean disallowNegative) {
        if (deltaInput.compareTo(BigDecimal.ZERO) == 0) {
            errors.addError(fieldKey, TypedError.invalid(messages.getMessage(
                    "errors.number.must.be.non.zero", null, locale)));
            return;
        }
        if (disallowNegative && deltaInput.signum() < 0) {
            errors.addError(fieldKey, TypedError.invalid(messages.getMessage(
                    "errors.number.must.be.positive", null, locale)));
            return;
        }
        UnitModel minAllowedUnit = Units.getMinAllowedUnit(resource, unitsEnsemble)
                .orElseThrow(() -> new IllegalArgumentException("No allowed units for resource " + resource.getId()));
        BigDecimal deltaInMinAllowedUnitsInteger = Units.convertFromUnitToUnitRoundDownToInteger(deltaInput,
                unitsEnsemble, unit, minAllowedUnit);
        if (deltaInMinAllowedUnitsInteger.compareTo(BigDecimal.ZERO) == 0) {
            errors.addError(fieldKey, TypedError.invalid(messages.getMessage(
                    "errors.number.is.too.small", null, locale)));
            return;
        }
        Optional<Long> deltaInBaseUnitO = Units.convertToBaseUnit(deltaInMinAllowedUnitsInteger, resource,
                unitsEnsemble, minAllowedUnit);
        if (deltaInBaseUnitO.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages.getMessage(
                    "errors.number.out.of.range", null, locale)));
            return;
        }
        deltaConsumer.accept(deltaInBaseUnitO.get());
    }

    private void addSuggestedAmount(ErrorCollection.Builder errors, long currentBalance,
                                    ValidatedQuotaResourceTransfer resourceTransfer, Locale locale) {
        errors.addDetail(resourceTransfer.getResource().getId() + ".suggestedAmount", QuotasHelper.getAmountDto(
                QuotasHelper.toBigDecimal(currentBalance), resourceTransfer.getResource(),
                resourceTransfer.getResourceUnitsEnsemble(), locale));
        errors.addDetail("suggestedAmountPrompt", messages.getMessage(
                "errors.not.enough.balance.for.quota.transfer.non.zero.balance.button",
                null, locale));
    }

    private void addSuggestedAmountAccount(
            ErrorCollection.Builder errors,
            long currentAvailable,
            ValidatedQuotaResourceTransfer resourceTransfer,
            boolean isAllocatedSupported,
            Locale locale
    ) {
        errors.addDetail(resourceTransfer.getResource().getId() + ".suggestedProvisionAmount",
                QuotasHelper.getAmountDto(QuotasHelper.toBigDecimal(currentAvailable), resourceTransfer.getResource(),
                resourceTransfer.getResourceUnitsEnsemble(), locale));
        String code = isAllocatedSupported
                ? "errors.not.enough.available.for.provision.transfer.not.allocated.button"
                : "errors.not.enough.available.for.provision.transfer.provided.button";
        errors.addDetail("suggestedProvisionAmountPrompt", messages.getMessage(
                code,
                null, locale));
    }
}
