package ru.yandex.intranet.d.services.transfer.model;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;

/**
 * Transfer requests with additional data.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ExpandedTransferRequests<T> {

    private final T transferRequests;
    private final Map<String, FolderModel> folders;
    private final Map<String, ServiceMinimalModel> services;
    private final Map<String, ResourceModel> resources;
    private final Map<String, ProviderModel> providers;
    private final Map<String, UnitModel> units;
    private final Map<String, UserModel> users;
    private final Map<String, UnitsEnsembleModel> unitsEnsembles;
    private final Map<String, AccountModel> accounts;
    private final Map<String, ResourceTypeModel> resourceTypes;
    private final Map<String, ResourceSegmentationModel> segmentations;
    private final Map<String, ResourceSegmentModel> segments;
    private final Map<String, AccountSpaceModel> accountsSpaces;

    @SuppressWarnings("ParameterNumber")
    public ExpandedTransferRequests(
            T transferRequests,
            Map<String, FolderModel> folders,
            Map<String, ServiceMinimalModel> services,
            Map<String, ResourceModel> resources,
            Map<String, ProviderModel> providers,
            Map<String, UnitModel> units,
            Map<String, UserModel> users,
            Map<String, UnitsEnsembleModel> unitsEnsembles,
            Map<String, AccountModel> accounts,
            Map<String, ResourceTypeModel> resourceTypes,
            Map<String, ResourceSegmentationModel> segmentations,
            Map<String, ResourceSegmentModel> segments,
            Map<String, AccountSpaceModel> accountsSpaces
    ) {
        this.transferRequests = transferRequests;
        this.folders = folders;
        this.services = services;
        this.resources = resources;
        this.providers = providers;
        this.units = units;
        this.users = users;
        this.unitsEnsembles = unitsEnsembles;
        this.accounts = accounts;
        this.resourceTypes = resourceTypes;
        this.segmentations = segmentations;
        this.segments = segments;
        this.accountsSpaces = accountsSpaces;
    }

    public static <T> Builder<T> builder() {
        return new Builder<>();
    }

    public static <T> Builder<T> builder(ExpandedTransferRequests<T> expandedTransferRequests) {
        return new Builder<>(expandedTransferRequests);
    }

    public T getTransferRequests() {
        return transferRequests;
    }

    public Map<String, FolderModel> getFolders() {
        return folders;
    }

    public Map<String, ServiceMinimalModel> getServices() {
        return services;
    }

    public Map<String, ResourceModel> getResources() {
        return resources;
    }

    public Map<String, ProviderModel> getProviders() {
        return providers;
    }

    public Map<String, UnitModel> getUnits() {
        return units;
    }

    public Map<String, UserModel> getUsers() {
        return users;
    }

    public Map<String, UnitsEnsembleModel> getUnitsEnsembles() {
        return unitsEnsembles;
    }

    public Map<String, AccountModel> getAccounts() {
        return accounts;
    }

    public Map<String, ResourceTypeModel> getResourceTypes() {
        return resourceTypes;
    }

    public Map<String, ResourceSegmentationModel> getSegmentations() {
        return segmentations;
    }

    public Map<String, ResourceSegmentModel> getSegments() {
        return segments;
    }

    public Map<String, AccountSpaceModel> getAccountsSpaces() {
        return accountsSpaces;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ExpandedTransferRequests<?> that = (ExpandedTransferRequests<?>) o;
        return Objects.equals(transferRequests, that.transferRequests) &&
                Objects.equals(folders, that.folders) &&
                Objects.equals(services, that.services) &&
                Objects.equals(resources, that.resources) &&
                Objects.equals(providers, that.providers) &&
                Objects.equals(units, that.units) &&
                Objects.equals(users, that.users) &&
                Objects.equals(unitsEnsembles, that.unitsEnsembles) &&
                Objects.equals(accounts, that.accounts) &&
                Objects.equals(resourceTypes, that.resourceTypes) &&
                Objects.equals(segmentations, that.segmentations) &&
                Objects.equals(segments, that.segments) &&
                Objects.equals(accountsSpaces, that.accountsSpaces);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                transferRequests, folders, services, resources, providers, units, users, unitsEnsembles, accounts,
                resourceTypes, segmentations, segments, accountsSpaces
        );
    }

    public static final class Builder<T> {

        private T transferRequests;
        private final Map<String, FolderModel> folders = new HashMap<>();
        private final Map<String, ServiceMinimalModel> services = new HashMap<>();
        private final Map<String, ResourceModel> resources = new HashMap<>();
        private final Map<String, ProviderModel> providers = new HashMap<>();
        private final Map<String, UnitModel> units = new HashMap<>();
        private final Map<String, UserModel> users = new HashMap<>();
        private final Map<String, UnitsEnsembleModel> unitsEnsembles = new HashMap<>();
        private final Map<String, AccountModel> accounts = new HashMap<>();
        private final Map<String, ResourceTypeModel> resourceTypes = new HashMap<>();
        private final Map<String, ResourceSegmentationModel> segmentations = new HashMap<>();
        private final Map<String, ResourceSegmentModel> segments = new HashMap<>();
        private final Map<String, AccountSpaceModel> accountsSpaces = new HashMap<>();

        private Builder() {
        }

        private Builder(ExpandedTransferRequests<T> expandedTransferRequests) {
            transferRequests(expandedTransferRequests.getTransferRequests());
            addFolders(expandedTransferRequests.getFolders().values());
            addServices(expandedTransferRequests.getServices().values());
            addResources(expandedTransferRequests.getResources().values());
            addProviders(expandedTransferRequests.getProviders().values());
            addUnits(expandedTransferRequests.getUnits().values());
            addUsers(expandedTransferRequests.getUsers().values());
            addUnitsEnsembles(expandedTransferRequests.getUnitsEnsembles().values());
            addUnitsEnsembles(expandedTransferRequests.getUnitsEnsembles().values());
        }

        public Builder<T> transferRequests(T transferRequests) {
            this.transferRequests = transferRequests;
            return this;
        }

        public Builder<T> addFolders(Collection<? extends FolderModel> folderModels) {
            folderModels.forEach(folder -> folders.put(folder.getId(), folder));
            return this;
        }

        public Builder<T> addServices(Collection<? extends ServiceMinimalModel> serviceModels) {
            serviceModels.forEach(service -> services.put(String.valueOf(service.getId()), service));
            return this;
        }

        public Builder<T> addResources(Collection<? extends ResourceModel> resourceModels) {
            resourceModels.forEach(resource -> resources.put(resource.getId(), resource));
            return this;
        }

        public Builder<T> addProviders(Collection<? extends ProviderModel> providerModels) {
            providerModels.forEach(provider -> providers.put(provider.getId(), provider));
            return this;
        }

        public Builder<T> addUnits(Collection<? extends UnitModel> unitModels) {
            unitModels.forEach(unit -> units.put(unit.getId(), unit));
            return this;
        }

        public Builder<T> addUsers(Collection<? extends UserModel> userModels) {
            userModels.forEach(user -> users.put(user.getId(), user));
            return this;
        }

        public Builder<T> addUnitsEnsembles(Collection<? extends UnitsEnsembleModel> unitsEnsembleModels) {
            unitsEnsembleModels.forEach(unitsEnsemble -> unitsEnsembles.put(unitsEnsemble.getId(), unitsEnsemble));
            return this;
        }

        public Builder<T> addAccounts(Collection<? extends AccountModel> accountModels) {
            accountModels.forEach(a -> accounts.put(a.getId(), a));
            return this;
        }

        public Builder<T> addResourceTypes(Collection<? extends ResourceTypeModel> resourceTypeModels) {
            resourceTypeModels.forEach(a -> resourceTypes.put(a.getId(), a));
            return this;
        }

        public Builder<T> addSegmentations(Collection<? extends ResourceSegmentationModel> resourceSegmentationModels) {
            resourceSegmentationModels.forEach(a -> segmentations.put(a.getId(), a));
            return this;
        }

        public Builder<T> addSegments(Collection<? extends ResourceSegmentModel> resourceSegmentModels) {
            resourceSegmentModels.forEach(a -> segments.put(a.getId(), a));
            return this;
        }

        public Builder<T> addAccountsSpaces(Collection<? extends AccountSpaceModel> accountSpaceModels) {
            accountSpaceModels.forEach(a -> accountsSpaces.put(a.getId(), a));
            return this;
        }

        public ExpandedTransferRequests<T> build() {
            Preconditions.checkNotNull(transferRequests, "TransferRequests is required");
            return new ExpandedTransferRequests<>(
                    transferRequests, folders, services, resources, providers,  units, users, unitsEnsembles, accounts,
                    resourceTypes, segmentations, segments, accountsSpaces
            );
        }

    }

}
