package ru.yandex.intranet.d.services.transfer.model

import com.google.common.base.Preconditions.checkNotNull
import ru.yandex.intranet.d.kotlin.AccountId
import ru.yandex.intranet.d.kotlin.FolderId


data class PreValidatedProvisionTransfer(
    val sourceAccountId: String,
    val sourceFolderId: String,
    val sourceServiceId: Long?,
    val destinationAccountId: String,
    val destinationFolderId: String,
    val destinationServiceId: Long?,
    val sourceAccountTransfers: List<PreValidatedQuotaResourceTransfer>,
    val destinationAccountTransfers: List<PreValidatedQuotaResourceTransfer>,
) {
    companion object {
        @JvmStatic
        fun builder() = Builder()
    }

    data class Key(
        val accountIds: Set<AccountId?>,
        val folderIds: Set<FolderId?>
    ) {
        constructor(
            sourceAccountId: AccountId?,
            destinationAccountId: AccountId?,
            sourceFolderId: FolderId?,
            destinationFolderId: FolderId?
        ) : this(
            setOf(sourceAccountId, destinationAccountId),
            setOf(sourceFolderId, destinationFolderId)
        )
    }

    class Builder(
        var sourceAccountId: String? = null,
        var sourceFolderId: String? = null,
        var sourceServiceId: Long? = null,
        var destinationAccountId: String? = null,
        var destinationFolderId: String? = null,
        var destinationServiceId: Long? = null,
        val sourceAccountTransfers: MutableList<PreValidatedQuotaResourceTransfer> = mutableListOf(),
        val destinationAccountTransfers: MutableList<PreValidatedQuotaResourceTransfer> = mutableListOf(),
    ) {
        fun sourceAccountId(sourceAccountId: String) = apply {
            this.sourceAccountId = sourceAccountId
        }

        fun sourceFolderId(sourceFolderId: String) = apply {
            this.sourceFolderId = sourceFolderId
        }

        fun sourceServiceId(sourceServiceId: Long) = apply {
            this.sourceServiceId = sourceServiceId
        }

        fun destinationAccountId(destinationAccountId: String) = apply {
            this.destinationAccountId = destinationAccountId
        }

        fun destinationFolderId(destinationFolderId: String) = apply {
            this.destinationFolderId = destinationFolderId
        }

        fun destinationServiceId(destinationServiceId: Long) = apply {
            this.destinationServiceId = destinationServiceId
        }

        fun addSourceAccountTransfer(sourceAccountTransfer: PreValidatedQuotaResourceTransfer) = apply {
            this.sourceAccountTransfers += sourceAccountTransfer
        }

        fun addDestinationAccountTransfer(destinationAccountTransfer: PreValidatedQuotaResourceTransfer) = apply {
            this.destinationAccountTransfers += destinationAccountTransfer
        }

        fun build() = PreValidatedProvisionTransfer(
            checkNotNull(sourceAccountId, "SourceAccountId is required"),
            checkNotNull(sourceFolderId, "SourceFolderId is required"),
            sourceServiceId,
            checkNotNull(destinationAccountId, "DestinationAccountId is required"),
            checkNotNull(destinationFolderId, "DestinationFolderId is required"),
            destinationServiceId,
            sourceAccountTransfers,
            destinationAccountTransfers,
        )
    }
}
