package ru.yandex.intranet.d.services.transfer.model;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;

/**
 * Pre validated quota transfer.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class PreValidatedQuotaTransfer {

    private final String destinationFolderId;
    private final Long destinationServiceId;
    private final List<PreValidatedQuotaResourceTransfer> transfers;
    private final long index;

    public PreValidatedQuotaTransfer(String destinationFolderId,
                                     Long destinationServiceId,
                                     List<PreValidatedQuotaResourceTransfer> transfers,
                                     long index) {
        this.destinationFolderId = destinationFolderId;
        this.destinationServiceId = destinationServiceId;
        this.transfers = transfers;
        this.index = index;
    }

    public static Builder builder() {
        return new Builder();
    }

    public String getDestinationFolderId() {
        return destinationFolderId;
    }

    public Optional<Long> getDestinationServiceId() {
        return Optional.ofNullable(destinationServiceId);
    }

    public List<PreValidatedQuotaResourceTransfer> getTransfers() {
        return transfers;
    }

    public long getIndex() {
        return index;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PreValidatedQuotaTransfer that = (PreValidatedQuotaTransfer) o;
        return index == that.index &&
                Objects.equals(destinationFolderId, that.destinationFolderId) &&
                Objects.equals(destinationServiceId, that.destinationServiceId) &&
                Objects.equals(transfers, that.transfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(destinationFolderId, destinationServiceId, transfers, index);
    }

    @Override
    public String toString() {
        return "PreValidatedQuotaTransfer{" +
                "destinationFolderId='" + destinationFolderId + '\'' +
                ", destinationServiceId=" + destinationServiceId +
                ", transfers=" + transfers +
                ", index=" + index +
                '}';
    }

    public static final class Builder {

        private final List<PreValidatedQuotaResourceTransfer> transfers = new ArrayList<>();

        private String destinationFolderId;
        private Long destinationServiceId;
        private Long index;

        private Builder() {
        }

        public Builder destinationFolderId(String destinationFolderId) {
            this.destinationFolderId = destinationFolderId;
            return this;
        }

        public Builder destinationServiceId(long destinationServiceId) {
            this.destinationServiceId = destinationServiceId;
            return this;
        }

        public Builder index(long index) {
            this.index = index;
            return this;
        }

        public Builder addTransfer(PreValidatedQuotaResourceTransfer transfer) {
            this.transfers.add(transfer);
            return this;
        }

        public List<PreValidatedQuotaResourceTransfer> getTransfers() {
            return transfers;
        }

        public PreValidatedQuotaTransfer build() {
            Preconditions.checkNotNull(destinationFolderId, "DestinationFolderId is required");
            Preconditions.checkNotNull(index, "Index is required");
            return new PreValidatedQuotaTransfer(destinationFolderId, destinationServiceId, transfers, index);
        }

    }

}
