package ru.yandex.intranet.d.services.transfer.model;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;

/**
 * Pre validated reserve transfer.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
public final class PreValidatedReserveTransfer {
    private final String providerId;
    private final String destinationFolderId;
    private final Long destinationServiceId;
    private final List<PreValidatedQuotaResourceTransfer> resourceTransfers;

    public PreValidatedReserveTransfer(String providerId, String destinationFolderId, Long destinationServiceId,
                                       List<PreValidatedQuotaResourceTransfer> resourceTransfers) {
        this.providerId = providerId;
        this.destinationFolderId = destinationFolderId;
        this.destinationServiceId = destinationServiceId;
        this.resourceTransfers = resourceTransfers;
    }

    public String getProviderId() {
        return providerId;
    }

    public String getDestinationFolderId() {
        return destinationFolderId;
    }

    public Optional<Long> getDestinationServiceId() {
        return Optional.ofNullable(destinationServiceId);
    }

    public List<PreValidatedQuotaResourceTransfer> getResourceTransfers() {
        return resourceTransfers;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PreValidatedReserveTransfer that = (PreValidatedReserveTransfer) o;
        return Objects.equals(providerId, that.providerId) &&
                Objects.equals(destinationFolderId, that.destinationFolderId) &&
                Objects.equals(destinationServiceId, that.destinationServiceId) &&
                Objects.equals(resourceTransfers, that.resourceTransfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(providerId, destinationFolderId, destinationServiceId, resourceTransfers);
    }

    @Override
    public String toString() {
        return "PreValidatedReserveTransfer{" +
                "providerId='" + providerId + '\'' +
                ", destinationFolderId='" + destinationFolderId + '\'' +
                ", destinationServiceId=" + destinationServiceId +
                ", resourceTransfers=" + resourceTransfers +
                '}';
    }

    public static Builder builder() {
        return new Builder();
    }

    public static final class Builder {
        private final List<PreValidatedQuotaResourceTransfer> transfers = new ArrayList<>();

        private String providerId;
        private String destinationFolderId;
        private Long destinationServiceId;

        private Builder() {

        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder destinationFolderId(String destinationFolderId) {
            this.destinationFolderId = destinationFolderId;
            return this;
        }

        public Builder destinationServiceId(long destinationServiceId) {
            this.destinationServiceId = destinationServiceId;
            return this;
        }

        public Builder addTransfer(PreValidatedQuotaResourceTransfer transfer) {
            this.transfers.add(transfer);
            return this;
        }

        public List<PreValidatedQuotaResourceTransfer> getTransfers() {
            return transfers;
        }

        public PreValidatedReserveTransfer build() {
            return new PreValidatedReserveTransfer(
                    Preconditions.checkNotNull(providerId, "ProviderId is required"),
                    Preconditions.checkNotNull(destinationFolderId, "DestinationFolderId is required"),
                    destinationServiceId, transfers);
        }
    }
}
