package ru.yandex.intranet.d.services.transfer.model;

import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;

/**
 * Transfer request creation result.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class TransferRequestCreationResult {

    private final TransferRequestModel request;
    private final List<FolderModel> folders;
    private final List<ServiceMinimalModel> services;
    private final List<ResourceModel> resources;
    private final List<UnitsEnsembleModel> unitsEnsembles;
    private final List<ProviderModel> providers;
    private final Set<UserModel> notifiedUsers;

    public TransferRequestCreationResult(TransferRequestModel request,
                                         List<FolderModel> folders,
                                         List<ServiceMinimalModel> services,
                                         List<ResourceModel> resources,
                                         List<UnitsEnsembleModel> unitsEnsembles,
                                         List<ProviderModel> providers,
                                         Set<UserModel> notifiedUsers) {
        this.request = request;
        this.folders = folders;
        this.services = services;
        this.resources = resources;
        this.unitsEnsembles = unitsEnsembles;
        this.providers = providers;
        this.notifiedUsers = notifiedUsers;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(TransferRequestCreationResult requestCreationResult) {
        return new Builder(requestCreationResult);
    }

    public TransferRequestModel getRequest() {
        return request;
    }

    public List<FolderModel> getFolders() {
        return folders;
    }

    public List<ServiceMinimalModel> getServices() {
        return services;
    }

    public List<ResourceModel> getResources() {
        return resources;
    }

    public List<UnitsEnsembleModel> getUnitsEnsembles() {
        return unitsEnsembles;
    }

    public List<ProviderModel> getProviders() {
        return providers;
    }

    public Set<UserModel> getNotifiedUsers() {
        return notifiedUsers;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestCreationResult that = (TransferRequestCreationResult) o;
        return Objects.equals(request, that.request) &&
                Objects.equals(folders, that.folders) &&
                Objects.equals(services, that.services) &&
                Objects.equals(resources, that.resources) &&
                Objects.equals(unitsEnsembles, that.unitsEnsembles) &&
                Objects.equals(providers, that.providers) &&
                Objects.equals(notifiedUsers, that.notifiedUsers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(request, folders, services, resources, unitsEnsembles, providers, notifiedUsers);
    }

    @Override
    public String toString() {
        return "TransferRequestCreationResult{" +
                "request=" + request +
                ", folders=" + folders +
                ", services=" + services +
                ", resources=" + resources +
                ", unitsEnsembles=" + unitsEnsembles +
                ", providers=" + providers +
                ", notifiedUsers=" + notifiedUsers +
                '}';
    }

    public static final class Builder {

        private TransferRequestModel request;
        private List<FolderModel> folders;
        private List<ServiceMinimalModel> services;
        private List<ResourceModel> resources;
        private List<UnitsEnsembleModel> unitsEnsembles;
        private List<ProviderModel> providers;
        private Set<UserModel> notifiedUsers;

        private Builder() {
        }

        private Builder(TransferRequestCreationResult requestCreationResult) {
            request(requestCreationResult.getRequest());
            folders(requestCreationResult.getFolders());
            services(requestCreationResult.getServices());
            resources(requestCreationResult.getResources());
            unitsEnsembles(requestCreationResult.getUnitsEnsembles());
            providers(requestCreationResult.getProviders());
            notifiedUsers(requestCreationResult.getNotifiedUsers());
        }

        public Builder request(TransferRequestModel request) {
            this.request = request;
            return this;
        }

        public Builder folders(List<FolderModel> folders) {
            this.folders = folders;
            return this;
        }

        public Builder services(List<ServiceMinimalModel> services) {
            this.services = services;
            return this;
        }

        public Builder resources(List<ResourceModel> resources) {
            this.resources = resources;
            return this;
        }

        public Builder unitsEnsembles(List<UnitsEnsembleModel> unitsEnsembles) {
            this.unitsEnsembles = unitsEnsembles;
            return this;
        }

        public Builder providers(List<ProviderModel> providers) {
            this.providers = providers;
            return this;
        }

        public Builder notifiedUsers(Set<UserModel> notifiedUsers) {
            this.notifiedUsers = notifiedUsers;
            return this;
        }

        public TransferRequestCreationResult build() {
            Preconditions.checkNotNull(request, "TransferRequest is required");
            Preconditions.checkNotNull(folders, "Folders is required");
            Preconditions.checkNotNull(services, "Services is required");
            Preconditions.checkNotNull(resources, "Resources is required");
            Preconditions.checkNotNull(unitsEnsembles, "UnitsEnsembles is required");
            Preconditions.checkNotNull(providers, "Providers is required");
            Preconditions.checkNotNull(notifiedUsers, "NotifiedUsers is required");
            return new TransferRequestCreationResult(request, folders, services, resources, unitsEnsembles, providers,
                    notifiedUsers);
        }

    }

}
