package ru.yandex.intranet.d.services.transfer.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.transfers.TransferRequestByFolderModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByResponsibleModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByServiceModel;

/**
 * Transfer request indices.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 * @see ru.yandex.intranet.d.services.transfer.TransferRequestIndicesService
 */
public final class TransferRequestIndices {

    private final List<TransferRequestByResponsibleModel> responsibleIndices;
    private final List<TransferRequestByFolderModel> folderIndices;
    private final List<TransferRequestByServiceModel> serviceIndices;

    public TransferRequestIndices(List<TransferRequestByResponsibleModel> responsibleIndices,
                                  List<TransferRequestByFolderModel> folderIndices,
                                  List<TransferRequestByServiceModel> serviceIndices) {
        this.responsibleIndices = responsibleIndices;
        this.folderIndices = folderIndices;
        this.serviceIndices = serviceIndices;
    }

    public TransferRequestIndices(Set<TransferRequestByResponsibleModel> responsibleIndices,
                                  Set<TransferRequestByFolderModel> folderIndices,
                                  Set<TransferRequestByServiceModel> serviceIndices) {
        this.responsibleIndices = new ArrayList<>(responsibleIndices);
        this.folderIndices = new ArrayList<>(folderIndices);
        this.serviceIndices = new ArrayList<>(serviceIndices);
    }

    public List<TransferRequestByResponsibleModel> getResponsibleIndices() {
        return responsibleIndices;
    }

    public List<TransferRequestByFolderModel> getFolderIndices() {
        return folderIndices;
    }

    public List<TransferRequestByServiceModel> getServiceIndices() {
        return serviceIndices;
    }

    /**
     * Difference.
     *
     * @see ru.yandex.intranet.d.services.transfer.TransferRequestIndicesService
     */
    public static final class Difference {
        private final List<TransferRequestByResponsibleModel> addedResponsible;
        private final List<TransferRequestByResponsibleModel> removedResponsible;
        private final List<TransferRequestByFolderModel> addedFolders;
        private final List<TransferRequestByFolderModel> removedFolders;
        private final List<TransferRequestByServiceModel> addedServices;
        private final List<TransferRequestByServiceModel> removedService;

        public Difference(Collection<TransferRequestByResponsibleModel> addedResponsible,
                          Collection<TransferRequestByResponsibleModel> removedResponsible,
                          Collection<TransferRequestByFolderModel> addedFolders,
                          Collection<TransferRequestByFolderModel> removedFolders,
                          Collection<TransferRequestByServiceModel> addedServices,
                          Collection<TransferRequestByServiceModel> removedService) {
            this.addedResponsible = new ArrayList<>(addedResponsible);
            this.removedResponsible = new ArrayList<>(removedResponsible);
            this.addedFolders = new ArrayList<>(addedFolders);
            this.removedFolders = new ArrayList<>(removedFolders);
            this.addedServices = new ArrayList<>(addedServices);
            this.removedService = new ArrayList<>(removedService);
        }

        public static Difference empty() {
            return new Difference(Collections.emptyList(), Collections.emptyList(), Collections.emptyList(),
                    Collections.emptyList(), Collections.emptyList(), Collections.emptyList());
        }

        public List<TransferRequestByResponsibleModel> getAddedResponsible() {
            return addedResponsible;
        }

        public List<TransferRequestByResponsibleModel> getRemovedResponsible() {
            return removedResponsible;
        }

        public List<TransferRequestByFolderModel> getAddedFolders() {
            return addedFolders;
        }

        public List<TransferRequestByFolderModel> getRemovedFolders() {
            return removedFolders;
        }

        public List<TransferRequestByServiceModel> getAddedServices() {
            return addedServices;
        }

        public List<TransferRequestByServiceModel> getRemovedService() {
            return removedService;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Difference that = (Difference) o;
            return addedResponsible.equals(that.addedResponsible)
                    && removedResponsible.equals(that.removedResponsible)
                    && addedFolders.equals(that.addedFolders)
                    && removedFolders.equals(that.removedFolders)
                    && addedServices.equals(that.addedServices)
                    && removedService.equals(that.removedService);
        }

        @Override
        public int hashCode() {
            return Objects.hash(addedResponsible, removedResponsible,
                    addedFolders, removedFolders, addedServices,
                    removedService);
        }

        @Override
        public String toString() {
            return "Difference{" +
                    "addedResponsible=" + addedResponsible +
                    ", removedResponsible=" + removedResponsible +
                    ", addedFolders=" + addedFolders +
                    ", removedFolders=" + removedFolders +
                    ", addedServices=" + addedServices +
                    ", removedService=" + removedService +
                    '}';
        }

    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestIndices that = (TransferRequestIndices) o;
        return Objects.equals(responsibleIndices, that.responsibleIndices) &&
                Objects.equals(folderIndices, that.folderIndices) &&
                Objects.equals(serviceIndices, that.serviceIndices);
    }

    @Override
    public int hashCode() {
        return Objects.hash(responsibleIndices, folderIndices, serviceIndices);
    }

    @Override
    public String toString() {
        return "TransferRequestIndices{" +
                "responsibleIndices=" + responsibleIndices +
                ", folderIndices=" + folderIndices +
                ", serviceIndices=" + serviceIndices +
                '}';
    }

    public static final class Builder {

        private List<TransferRequestByResponsibleModel> responsibleIndices;
        private List<TransferRequestByFolderModel> folderIndices;
        private List<TransferRequestByServiceModel> serviceIndices;

        private Builder() {
        }

        public Builder responsibleIndices(List<TransferRequestByResponsibleModel> responsibleIndices) {
            this.responsibleIndices = responsibleIndices;
            return this;
        }

        public Builder folderIndices(List<TransferRequestByFolderModel> folderIndices) {
            this.folderIndices = folderIndices;
            return this;
        }

        public Builder serviceIndices(List<TransferRequestByServiceModel> serviceIndices) {
            this.serviceIndices = serviceIndices;
            return this;
        }

        public TransferRequestIndices build() {
            Preconditions.checkNotNull(responsibleIndices, "ResponsibleIndices is required");
            Preconditions.checkNotNull(folderIndices, "FolderIndices is required");
            Preconditions.checkNotNull(serviceIndices, "ServiceIndices is required");
            return new TransferRequestIndices(responsibleIndices, folderIndices, serviceIndices);
        }

    }

}
