package ru.yandex.intranet.d.services.transfer.model;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByFolderModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByResponsibleModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestByServiceModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.operations.model.ValidatedReceivedProvision;

/**
 * Validated create transfer request.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ValidatedCreateTransferRequest {

    private final TransferRequestModel transferRequest;
    private final TransferRequestHistoryModel history;
    private final List<TransferRequestByResponsibleModel> responsibleIndices;
    private final List<TransferRequestByFolderModel> folderIndices;
    private final List<TransferRequestByServiceModel> serviceIndices;
    private final List<QuotaModel> quotas;
    private final List<AccountsQuotasModel> accountsQuotas;
    private final List<FolderModel> folders;
    private final List<AccountModel> accounts;
    private final List<ServiceMinimalModel> services;
    private final List<ResourceModel> resources;
    private final List<UnitsEnsembleModel> unitsEnsembles;
    private final List<ProviderModel> providers;
    private final boolean apply;
    private final Map<String, String> preGeneratedFolderOpLogIdsByFolderId;
    private final Instant now;
    private final Set<UserModel> notifiedUsers;
    private final Set<String> operationIds;
    private final Map<String, Map<String, ValidatedReceivedProvision>> receivedProvisionByAccountResource;

    @SuppressWarnings("ParameterNumber")
    public ValidatedCreateTransferRequest(
            TransferRequestModel transferRequest,
            TransferRequestHistoryModel history,
            List<TransferRequestByResponsibleModel> responsibleIndices,
            List<TransferRequestByFolderModel> folderIndices,
            List<TransferRequestByServiceModel> serviceIndices,
            List<QuotaModel> quotas,
            List<AccountsQuotasModel> accountsQuotas,
            List<FolderModel> folders,
            List<AccountModel> accounts,
            List<ServiceMinimalModel> services,
            List<ResourceModel> resources,
            List<UnitsEnsembleModel> unitsEnsembles,
            List<ProviderModel> providers,
            boolean apply,
            Map<String, String> preGeneratedFolderOpLogIdsByFolderId,
            Instant now,
            Set<UserModel> notifiedUsers,
            Set<String> operationIds,
            Map<String, Map<String, ValidatedReceivedProvision>> receivedProvisionByAccountResource
    ) {
        this.transferRequest = transferRequest;
        this.history = history;
        this.responsibleIndices = responsibleIndices;
        this.folderIndices = folderIndices;
        this.serviceIndices = serviceIndices;
        this.quotas = quotas;
        this.accountsQuotas = accountsQuotas;
        this.folders = folders;
        this.accounts = accounts;
        this.services = services;
        this.resources = resources;
        this.unitsEnsembles = unitsEnsembles;
        this.providers = providers;
        this.apply = apply;
        this.preGeneratedFolderOpLogIdsByFolderId = preGeneratedFolderOpLogIdsByFolderId;
        this.now = now;
        this.notifiedUsers = notifiedUsers;
        this.operationIds = operationIds;
        this.receivedProvisionByAccountResource = receivedProvisionByAccountResource;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(ValidatedCreateTransferRequest original) {
        return new Builder(original);
    }

    public Builder copyBuilder() {
        return new Builder(this);
    }

    public TransferRequestModel getTransferRequest() {
        return transferRequest;
    }

    public TransferRequestHistoryModel getHistory() {
        return history;
    }

    public List<TransferRequestByResponsibleModel> getResponsibleIndices() {
        return responsibleIndices;
    }

    public List<TransferRequestByFolderModel> getFolderIndices() {
        return folderIndices;
    }

    public List<TransferRequestByServiceModel> getServiceIndices() {
        return serviceIndices;
    }

    public List<QuotaModel> getQuotas() {
        return quotas;
    }

    public List<AccountsQuotasModel> getAccountsQuotas() {
        return accountsQuotas;
    }

    public List<FolderModel> getFolders() {
        return folders;
    }

    public List<AccountModel> getAccounts() {
        return accounts;
    }

    public List<ServiceMinimalModel> getServices() {
        return services;
    }

    public List<ResourceModel> getResources() {
        return resources;
    }

    public List<UnitsEnsembleModel> getUnitsEnsembles() {
        return unitsEnsembles;
    }

    public List<ProviderModel> getProviders() {
        return providers;
    }

    public boolean isApply() {
        return apply;
    }

    public Map<String, String> getPreGeneratedFolderOpLogIdsByFolderId() {
        return preGeneratedFolderOpLogIdsByFolderId;
    }

    public Instant getNow() {
        return now;
    }

    public Set<UserModel> getNotifiedUsers() {
        return notifiedUsers;
    }

    public Set<String> getOperationIds() {
        return operationIds;
    }

    public Map<String, Map<String, ValidatedReceivedProvision>> getReceivedProvisionByAccountResource() {
        return receivedProvisionByAccountResource;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ValidatedCreateTransferRequest that = (ValidatedCreateTransferRequest) o;
        return apply == that.apply &&
                Objects.equals(transferRequest, that.transferRequest) &&
                Objects.equals(history, that.history) &&
                Objects.equals(responsibleIndices, that.responsibleIndices) &&
                Objects.equals(folderIndices, that.folderIndices) &&
                Objects.equals(serviceIndices, that.serviceIndices) &&
                Objects.equals(quotas, that.quotas) &&
                Objects.equals(folders, that.folders) &&
                Objects.equals(services, that.services) &&
                Objects.equals(resources, that.resources) &&
                Objects.equals(unitsEnsembles, that.unitsEnsembles) &&
                Objects.equals(providers, that.providers) &&
                Objects.equals(preGeneratedFolderOpLogIdsByFolderId, that.preGeneratedFolderOpLogIdsByFolderId) &&
                Objects.equals(now, that.now) &&
                Objects.equals(operationIds, that.operationIds) &&
                Objects.equals(notifiedUsers, that.notifiedUsers) &&
                Objects.equals(receivedProvisionByAccountResource, that.receivedProvisionByAccountResource);
    }

    @Override
    public int hashCode() {
        return Objects.hash(transferRequest, history, responsibleIndices, folderIndices, serviceIndices, quotas,
                folders, services, resources, unitsEnsembles, providers, apply, preGeneratedFolderOpLogIdsByFolderId,
                now, notifiedUsers, operationIds, receivedProvisionByAccountResource);
    }

    @Override
    public String toString() {
        return "ValidatedCreateTransferRequest{" +
                "transferRequest=" + transferRequest +
                ", history=" + history +
                ", responsibleIndices=" + responsibleIndices +
                ", folderIndices=" + folderIndices +
                ", serviceIndices=" + serviceIndices +
                ", quotas=" + quotas +
                ", folders=" + folders +
                ", services=" + services +
                ", resources=" + resources +
                ", unitsEnsembles=" + unitsEnsembles +
                ", providers=" + providers +
                ", apply=" + apply +
                ", preGeneratedFolderOpLogIdsByFolderId=" + preGeneratedFolderOpLogIdsByFolderId +
                ", now=" + now +
                ", notifiedUsers=" + notifiedUsers +
                ", operationIds=" + operationIds +
                ", receivedProvisionByAccountResource=" + receivedProvisionByAccountResource +
                '}';
    }

    public static final class Builder {

        private final List<TransferRequestByResponsibleModel> responsibleIndices = new ArrayList<>();
        private final List<TransferRequestByFolderModel> folderIndices = new ArrayList<>();
        private final List<TransferRequestByServiceModel> serviceIndices = new ArrayList<>();
        private final Map<String, String> preGeneratedFolderOpLogIdsByFolderId = new HashMap<>();
        private final Set<UserModel> notifiedUsers = new HashSet<>();
        private final Set<String> operationIds = new HashSet<>();
        private final List<AccountModel> accounts = new ArrayList<>();
        private final List<AccountsQuotasModel> accountsQuotas = new ArrayList<>();
        private final Map<String, Map<String, ValidatedReceivedProvision>> receivedProvisionByAccountResource
                = new HashMap<>();

        private TransferRequestModel transferRequest;
        private TransferRequestHistoryModel history;
        private List<QuotaModel> quotas;
        private List<FolderModel> folders;
        private List<ServiceMinimalModel> services;
        private List<ResourceModel> resources;
        private List<UnitsEnsembleModel> unitsEnsembles;
        private List<ProviderModel> providers;
        private Boolean apply;
        private Instant now;

        private Builder() {
        }

        private Builder(ValidatedCreateTransferRequest original) {
            this.transferRequest = original.transferRequest;
            this.history = original.history;
            this.quotas = original.quotas;
            this.folders = original.folders;
            this.services = original.services;
            this.resources = original.resources;
            this.unitsEnsembles = original.unitsEnsembles;
            this.providers = original.providers;
            this.apply = original.apply;
            this.now = original.now;
            this.responsibleIndices.addAll(original.responsibleIndices);
            this.folderIndices.addAll(original.folderIndices);
            this.serviceIndices.addAll(original.serviceIndices);
            this.preGeneratedFolderOpLogIdsByFolderId.putAll(original.preGeneratedFolderOpLogIdsByFolderId);
            this.notifiedUsers.addAll(original.notifiedUsers);
            this.operationIds.addAll(original.operationIds);
            this.accounts.addAll(original.accounts);
            this.accountsQuotas.addAll(original.accountsQuotas);
            this.receivedProvisionByAccountResource.putAll(original.receivedProvisionByAccountResource);
        }

        public Builder transferRequest(TransferRequestModel transferRequest) {
            this.transferRequest = transferRequest;
            return this;
        }

        public Builder history(TransferRequestHistoryModel history) {
            this.history = history;
            return this;
        }

        public Builder addResponsibleIndex(TransferRequestByResponsibleModel index) {
            this.responsibleIndices.add(index);
            return this;
        }

        public Builder addResponsibleIndices(Collection<TransferRequestByResponsibleModel> indices) {
            this.responsibleIndices.addAll(indices);
            return this;
        }

        public Builder addFolderIndex(TransferRequestByFolderModel index) {
            this.folderIndices.add(index);
            return this;
        }

        public Builder addServiceIndex(TransferRequestByServiceModel index) {
            this.serviceIndices.add(index);
            return this;
        }

        public Builder quotas(List<QuotaModel> quotas) {
            this.quotas = quotas;
            return this;
        }

        public Builder folders(List<FolderModel> folders) {
            this.folders = folders;
            return this;
        }

        public Builder addAccounts(List<AccountModel> accounts) {
            this.accounts.addAll(accounts);
            return this;
        }

        public Builder addAccountsQuotas(List<AccountsQuotasModel> accountsQuotas) {
            this.accountsQuotas.addAll(accountsQuotas);
            return this;
        }

        public Builder services(List<ServiceMinimalModel> services) {
            this.services = services;
            return this;
        }

        public Builder resources(List<ResourceModel> resources) {
            this.resources = resources;
            return this;
        }

        public Builder unitsEnsembles(List<UnitsEnsembleModel> unitsEnsembles) {
            this.unitsEnsembles = unitsEnsembles;
            return this;
        }

        public Builder providers(List<ProviderModel> providers) {
            this.providers = providers;
            return this;
        }

        public Builder apply(boolean apply) {
            this.apply = apply;
            return this;
        }

        public Builder putPreGeneratedFolderOpLogId(String folderId, String opLogId) {
            this.preGeneratedFolderOpLogIdsByFolderId.put(folderId, opLogId);
            return this;
        }

        public Builder now(Instant now) {
            this.now = now;
            return this;
        }

        public Builder addNotifiedUser(UserModel notifiedUser) {
            this.notifiedUsers.add(notifiedUser);
            return this;
        }

        public Builder addNotifiedUsers(Collection<? extends UserModel> notifiedUsers) {
            this.notifiedUsers.addAll(notifiedUsers);
            return this;
        }

        public Builder addOperationId(String operationId) {
            this.operationIds.add(operationId);
            return this;
        }

        public Builder addReceivedProvisionByAccountResource(
                Map<String, Map<String, ValidatedReceivedProvision>> receivedProvisionByAccountResource
        ) {
            this.receivedProvisionByAccountResource.putAll(receivedProvisionByAccountResource);
            return this;
        }

        public ValidatedCreateTransferRequest build() {
            Preconditions.checkNotNull(transferRequest, "TransferRequest is required");
            Preconditions.checkNotNull(history, "History is required");
            Preconditions.checkNotNull(apply, "Apply is required");
            Preconditions.checkNotNull(quotas, "Quotas is required");
            Preconditions.checkNotNull(folders, "Folders is required");
            Preconditions.checkNotNull(services, "Services is required");
            Preconditions.checkNotNull(resources, "Resources is required");
            Preconditions.checkNotNull(unitsEnsembles, "UnitsEnsembles is required");
            Preconditions.checkNotNull(providers, "Providers is required");
            Preconditions.checkNotNull(now, "Now is required");
            return new ValidatedCreateTransferRequest(transferRequest, history, responsibleIndices,
                    folderIndices, serviceIndices, quotas, accountsQuotas, folders, accounts, services, resources,
                    unitsEnsembles, providers, apply, preGeneratedFolderOpLogIdsByFolderId, now, notifiedUsers,
                    operationIds, receivedProvisionByAccountResource);
        }

    }

}
