package ru.yandex.intranet.d.services.transfer.model;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;

/**
 * Validated put transfer request.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ValidatedPutTransferRequest {

    private final TransferRequestModel transferRequest;
    private final TransferRequestHistoryModel history;
    private final TransferRequestIndices.Difference indicesDifference;
    private final List<QuotaModel> quotas;
    private final List<AccountsQuotasModel> accountsQuotas;
    private final List<FolderModel> folders;
    private final List<AccountModel> accounts;
    private final List<ServiceMinimalModel> services;
    private final List<ResourceModel> resources;
    private final List<UnitsEnsembleModel> unitsEnsembles;
    private final List<ProviderModel> providers;
    private final boolean apply;
    private final Map<String, String> preGeneratedFolderOpLogIdsByFolderId;
    private final Instant now;
    private final TransferRequestModel oldTransferRequest;
    private final Set<UserModel> newNotifiedUsers;
    private final Set<UserModel> updateNotifiedUsers;
    private final Set<String> operationIds;

    @SuppressWarnings("ParameterNumber")
    public ValidatedPutTransferRequest(TransferRequestModel transferRequest,
                                       TransferRequestHistoryModel history,
                                       TransferRequestIndices.Difference indicesDifference,
                                       List<QuotaModel> quotas,
                                       List<AccountsQuotasModel> accountsQuotas,
                                       List<FolderModel> folders,
                                       List<AccountModel> accounts,
                                       List<ServiceMinimalModel> services,
                                       List<ResourceModel> resources,
                                       List<UnitsEnsembleModel> unitsEnsembles,
                                       List<ProviderModel> providers,
                                       boolean apply,
                                       Map<String, String> preGeneratedFolderOpLogIdsByFolderId,
                                       Instant now,
                                       TransferRequestModel oldTransferRequest,
                                       Set<UserModel> newNotifiedUsers,
                                       Set<UserModel> updateNotifiedUsers,
                                       Set<String> operationIds) {
        this.transferRequest = transferRequest;
        this.history = history;
        this.indicesDifference = indicesDifference;
        this.quotas = quotas;
        this.accountsQuotas = accountsQuotas;
        this.folders = folders;
        this.accounts = accounts;
        this.services = services;
        this.resources = resources;
        this.unitsEnsembles = unitsEnsembles;
        this.providers = providers;
        this.apply = apply;
        this.preGeneratedFolderOpLogIdsByFolderId = preGeneratedFolderOpLogIdsByFolderId;
        this.now = now;
        this.oldTransferRequest = oldTransferRequest;
        this.newNotifiedUsers = newNotifiedUsers;
        this.updateNotifiedUsers = updateNotifiedUsers;
        this.operationIds = operationIds;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(ValidatedPutTransferRequest transferRequest) {
        return new Builder(transferRequest);
    }

    public Builder copyBuilder() {
        return new Builder(this);
    }

    public Optional<TransferRequestModel> getTransferRequest() {
        return Optional.ofNullable(transferRequest);
    }

    public Optional<TransferRequestHistoryModel> getHistory() {
        return Optional.ofNullable(history);
    }

    public List<QuotaModel> getQuotas() {
        return quotas;
    }

    public List<FolderModel> getFolders() {
        return folders;
    }

    public List<AccountsQuotasModel> getAccountsQuotas() {
        return accountsQuotas;
    }

    public List<AccountModel> getAccounts() {
        return accounts;
    }

    public List<ServiceMinimalModel> getServices() {
        return services;
    }

    public List<ResourceModel> getResources() {
        return resources;
    }

    public List<UnitsEnsembleModel> getUnitsEnsembles() {
        return unitsEnsembles;
    }

    public List<ProviderModel> getProviders() {
        return providers;
    }

    public boolean isApply() {
        return apply;
    }

    public Map<String, String> getPreGeneratedFolderOpLogIdsByFolderId() {
        return preGeneratedFolderOpLogIdsByFolderId;
    }

    public Instant getNow() {
        return now;
    }

    public TransferRequestIndices.Difference getIndicesDifference() {
        return indicesDifference;
    }

    public Optional<TransferRequestModel> getOldTransferRequest() {
        return Optional.ofNullable(oldTransferRequest);
    }

    public Set<UserModel> getNewNotifiedUsers() {
        return newNotifiedUsers;
    }

    public Set<UserModel> getUpdateNotifiedUsers() {
        return updateNotifiedUsers;
    }

    public Set<String> getOperationIds() {
        return operationIds;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ValidatedPutTransferRequest that = (ValidatedPutTransferRequest) o;
        return apply == that.apply &&
                Objects.equals(transferRequest, that.transferRequest) &&
                Objects.equals(history, that.history) &&
                Objects.equals(indicesDifference, that.indicesDifference) &&
                Objects.equals(quotas, that.quotas) &&
                Objects.equals(folders, that.folders) &&
                Objects.equals(services, that.services) &&
                Objects.equals(resources, that.resources) &&
                Objects.equals(unitsEnsembles, that.unitsEnsembles) &&
                Objects.equals(providers, that.providers) &&
                Objects.equals(preGeneratedFolderOpLogIdsByFolderId, that.preGeneratedFolderOpLogIdsByFolderId) &&
                Objects.equals(now, that.now) &&
                Objects.equals(oldTransferRequest, that.oldTransferRequest) &&
                Objects.equals(newNotifiedUsers, that.newNotifiedUsers) &&
                Objects.equals(operationIds, that.operationIds) &&
                Objects.equals(updateNotifiedUsers, that.updateNotifiedUsers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(transferRequest, history, indicesDifference, quotas,
                folders, services, resources, unitsEnsembles, providers, apply, preGeneratedFolderOpLogIdsByFolderId,
                now, oldTransferRequest, newNotifiedUsers, updateNotifiedUsers, operationIds);
    }

    @Override
    public String toString() {
        return "ValidatedPutTransferRequest{" +
                "transferRequest=" + transferRequest +
                ", history=" + history +
                ", indicesDifference=" + indicesDifference +
                ", quotas=" + quotas +
                ", folders=" + folders +
                ", services=" + services +
                ", resources=" + resources +
                ", unitsEnsembles=" + unitsEnsembles +
                ", providers=" + providers +
                ", apply=" + apply +
                ", preGeneratedFolderOpLogIdsByFolderId=" + preGeneratedFolderOpLogIdsByFolderId +
                ", now=" + now +
                ", oldTransferRequest=" + oldTransferRequest +
                ", newNotifiedUsers=" + newNotifiedUsers +
                ", updateNotifiedUsers=" + updateNotifiedUsers +
                ", operationIds=" + operationIds +
                '}';
    }

    public static final class Builder {

        private final Map<String, String> preGeneratedFolderOpLogIdsByFolderId = new HashMap<>();
        private final Set<UserModel> newNotifiedUsers = new HashSet<>();
        private final Set<UserModel> updateNotifiedUsers = new HashSet<>();
        private final Set<String> operationIds = new HashSet<>();
        private final List<AccountModel> accounts = new ArrayList<>();
        private final List<AccountsQuotasModel> accountsQuotas = new ArrayList<>();

        private TransferRequestIndices.Difference indicesDifference = TransferRequestIndices.Difference.empty();
        private TransferRequestModel transferRequest;
        private TransferRequestHistoryModel history;
        private List<QuotaModel> quotas;
        private List<FolderModel> folders;
        private List<ServiceMinimalModel> services;
        private List<ResourceModel> resources;
        private List<UnitsEnsembleModel> unitsEnsembles;
        private List<ProviderModel> providers;
        private Boolean apply;
        private Instant now;
        private TransferRequestModel oldTransferRequest;

        private Builder() {
        }

        private Builder(ValidatedPutTransferRequest original) {
            this.indicesDifference = original.indicesDifference;
            this.transferRequest = original.transferRequest;
            this.history = original.history;
            this.quotas = original.quotas;
            this.folders = original.folders;
            this.services = original.services;
            this.resources = original.resources;
            this.unitsEnsembles = original.unitsEnsembles;
            this.providers = original.providers;
            this.apply = original.apply;
            this.now = original.now;
            this.oldTransferRequest = original.oldTransferRequest;
            this.preGeneratedFolderOpLogIdsByFolderId.putAll(original.preGeneratedFolderOpLogIdsByFolderId);
            this.newNotifiedUsers.addAll(original.newNotifiedUsers);
            this.updateNotifiedUsers.addAll(original.updateNotifiedUsers);
            this.operationIds.addAll(original.operationIds);
            this.accounts.addAll(original.accounts);
            this.accountsQuotas.addAll(original.accountsQuotas);
        }

        public Builder transferRequest(TransferRequestModel transferRequest) {
            this.transferRequest = transferRequest;
            return this;
        }

        public Builder history(TransferRequestHistoryModel history) {
            this.history = history;
            return this;
        }

        public Builder indicesDifference(TransferRequestIndices.Difference indicesDifference) {
            this.indicesDifference = indicesDifference;
            return this;
        }

        public Builder quotas(List<QuotaModel> quotas) {
            this.quotas = quotas;
            return this;
        }

        public Builder folders(List<FolderModel> folders) {
            this.folders = folders;
            return this;
        }

        public Builder addAccounts(List<AccountModel> accounts) {
            this.accounts.addAll(accounts);
            return this;
        }

        public Builder addAccountsQuotas(List<AccountsQuotasModel> accountsQuotas) {
            this.accountsQuotas.addAll(accountsQuotas);
            return this;
        }

        public Builder services(List<ServiceMinimalModel> services) {
            this.services = services;
            return this;
        }

        public Builder resources(List<ResourceModel> resources) {
            this.resources = resources;
            return this;
        }

        public Builder unitsEnsembles(List<UnitsEnsembleModel> unitsEnsembles) {
            this.unitsEnsembles = unitsEnsembles;
            return this;
        }

        public Builder providers(List<ProviderModel> providers) {
            this.providers = providers;
            return this;
        }

        public Builder apply(boolean apply) {
            this.apply = apply;
            return this;
        }

        public Builder putPreGeneratedFolderOpLogId(String folderId, String opLogId) {
            this.preGeneratedFolderOpLogIdsByFolderId.put(folderId, opLogId);
            return this;
        }

        public Builder now(Instant now) {
            this.now = now;
            return this;
        }

        public Builder oldTransferRequest(TransferRequestModel oldTransferRequest) {
            this.oldTransferRequest = oldTransferRequest;
            return this;
        }

        public Builder addNewNotifiedUser(UserModel notifiedUser) {
            this.newNotifiedUsers.add(notifiedUser);
            return this;
        }

        public Builder addNewNotifiedUsers(Collection<? extends UserModel> notifiedUsers) {
            this.newNotifiedUsers.addAll(notifiedUsers);
            return this;
        }

        public Builder addUpdateNotifiedUser(UserModel notifiedUser) {
            this.updateNotifiedUsers.add(notifiedUser);
            return this;
        }

        public Builder addUpdateNotifiedUsers(Collection<? extends UserModel> notifiedUsers) {
            this.updateNotifiedUsers.addAll(notifiedUsers);
            return this;
        }

        public Builder addOperationId(String operationId) {
            this.operationIds.add(operationId);
            return this;
        }

        public ValidatedPutTransferRequest build() {
            Preconditions.checkNotNull(apply, "Apply is required");
            Preconditions.checkNotNull(quotas, "Quotas is required");
            Preconditions.checkNotNull(folders, "Folders is required");
            Preconditions.checkNotNull(services, "Services is required");
            Preconditions.checkNotNull(resources, "Resources is required");
            Preconditions.checkNotNull(unitsEnsembles, "UnitsEnsembles is required");
            Preconditions.checkNotNull(providers, "Providers is required");
            Preconditions.checkNotNull(now, "Now is required");
            return new ValidatedPutTransferRequest(transferRequest, history, indicesDifference, quotas, accountsQuotas,
                    folders, accounts, services, resources, unitsEnsembles,
                    providers, apply, preGeneratedFolderOpLogIdsByFolderId, now, oldTransferRequest, newNotifiedUsers,
                    updateNotifiedUsers, operationIds);
        }

    }

}
