package ru.yandex.intranet.d.services.transfer.model;

import java.util.Objects;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;

/**
 * Validated quota resource transfer.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ValidatedQuotaResourceTransfer {

    private final ResourceModel resource;
    private final long delta;
    private final UnitsEnsembleModel resourceUnitsEnsemble;
    private final UnitModel deltaUnit;
    private final long index;

    public ValidatedQuotaResourceTransfer(ResourceModel resource, long delta,
                                          UnitsEnsembleModel resourceUnitsEnsemble, UnitModel deltaUnit, long index) {
        this.resource = resource;
        this.delta = delta;
        this.resourceUnitsEnsemble = resourceUnitsEnsemble;
        this.deltaUnit = deltaUnit;
        this.index = index;
    }

    public static Builder builder() {
        return new Builder();
    }

    public ResourceModel getResource() {
        return resource;
    }

    public long getDelta() {
        return delta;
    }

    public UnitsEnsembleModel getResourceUnitsEnsemble() {
        return resourceUnitsEnsemble;
    }

    public UnitModel getDeltaUnit() {
        return deltaUnit;
    }

    public long getIndex() {
        return index;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ValidatedQuotaResourceTransfer that = (ValidatedQuotaResourceTransfer) o;
        return delta == that.delta &&
                index == that.index &&
                Objects.equals(resource, that.resource) &&
                Objects.equals(resourceUnitsEnsemble, that.resourceUnitsEnsemble) &&
                Objects.equals(deltaUnit, that.deltaUnit);
    }

    @Override
    public int hashCode() {
        return Objects.hash(resource, delta, resourceUnitsEnsemble, deltaUnit, index);
    }

    @Override
    public String toString() {
        return "ValidatedQuotaResourceTransfer{" +
                "resource=" + resource +
                ", delta=" + delta +
                ", resourceUnitsEnsemble=" + resourceUnitsEnsemble +
                ", deltaUnit=" + deltaUnit +
                ", index=" + index +
                '}';
    }

    public static final class Builder {

        private ResourceModel resource;
        private Long delta;
        private UnitsEnsembleModel resourceUnitsEnsemble;
        private UnitModel deltaUnit;
        private Long index;

        private Builder() {
        }

        public Builder resource(ResourceModel resource) {
            this.resource = resource;
            return this;
        }

        public Builder delta(Long delta) {
            this.delta = delta;
            return this;
        }

        public Builder resourceUnitsEnsemble(UnitsEnsembleModel resourceUnitsEnsemble) {
            this.resourceUnitsEnsemble = resourceUnitsEnsemble;
            return this;
        }

        public Builder deltaUnit(UnitModel deltaUnit) {
            this.deltaUnit = deltaUnit;
            return this;
        }

        public Builder index(long index) {
            this.index = index;
            return this;
        }

        public ValidatedQuotaResourceTransfer build() {
            Preconditions.checkNotNull(resource, "Resource is required");
            Preconditions.checkNotNull(delta, "Delta is required");
            Preconditions.checkNotNull(resourceUnitsEnsemble, "ResourceUnitsEnsemble is required");
            Preconditions.checkNotNull(deltaUnit, "DeltaUnit is required");
            Preconditions.checkNotNull(index, "Index is required");
            return new ValidatedQuotaResourceTransfer(resource, delta, resourceUnitsEnsemble, deltaUnit, index);
        }

    }

}
