package ru.yandex.intranet.d.services.transfer.model;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;

/**
 * Validated quota transfer.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ValidatedQuotaTransfer {

    private final FolderModel destinationFolder;
    private final ServiceMinimalModel destinationService;
    private final List<ValidatedQuotaResourceTransfer> transfers;
    private final long index;

    public ValidatedQuotaTransfer(FolderModel destinationFolder, ServiceMinimalModel destinationService,
                                  List<ValidatedQuotaResourceTransfer> transfers, long index) {
        this.destinationFolder = destinationFolder;
        this.destinationService = destinationService;
        this.transfers = transfers;
        this.index = index;
    }

    public static Builder builder() {
        return new Builder();
    }

    public FolderModel getDestinationFolder() {
        return destinationFolder;
    }

    public ServiceMinimalModel getDestinationService() {
        return destinationService;
    }

    public List<ValidatedQuotaResourceTransfer> getTransfers() {
        return transfers;
    }

    public long getIndex() {
        return index;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ValidatedQuotaTransfer that = (ValidatedQuotaTransfer) o;
        return index == that.index &&
                Objects.equals(destinationFolder, that.destinationFolder) &&
                Objects.equals(destinationService, that.destinationService) &&
                Objects.equals(transfers, that.transfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(destinationFolder, destinationService, transfers, index);
    }

    @Override
    public String toString() {
        return "ValidatedQuotaTransfer{" +
                "destinationFolder=" + destinationFolder +
                ", destinationService=" + destinationService +
                ", transfers=" + transfers +
                ", index=" + index +
                '}';
    }

    public static final class Builder {

        private final List<ValidatedQuotaResourceTransfer> transfers = new ArrayList<>();

        private FolderModel destinationFolder;
        private ServiceMinimalModel destinationService;
        private Long index;

        private Builder() {
        }

        public Builder destinationFolder(FolderModel destinationFolder) {
            this.destinationFolder = destinationFolder;
            return this;
        }

        public Builder destinationService(ServiceMinimalModel destinationService) {
            this.destinationService = destinationService;
            return this;
        }

        public Builder addTransfer(ValidatedQuotaResourceTransfer transfer) {
            this.transfers.add(transfer);
            return this;
        }

        public Builder index(long index) {
            this.index = index;
            return this;
        }

        public ValidatedQuotaTransfer build() {
            Preconditions.checkNotNull(destinationFolder, "DestinationFolder is required");
            Preconditions.checkNotNull(destinationService, "DestinationService is required");
            Preconditions.checkNotNull(index, "Index is required");
            return new ValidatedQuotaTransfer(destinationFolder, destinationService, transfers, index);
        }

    }

}
