package ru.yandex.intranet.d.services.transfer.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;

/**
 * Pre validated quota transfer parameters.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ValidatedQuotaTransferParameters {

    private final List<ValidatedQuotaTransfer> transfers;
    private final List<FolderModel> folders;
    private final List<ServiceMinimalModel> services;
    private final List<ResourceModel> resources;
    private final List<UnitsEnsembleModel> unitsEnsembles;
    private final List<ProviderModel> providers;

    public ValidatedQuotaTransferParameters(List<ValidatedQuotaTransfer> transfers,
                                            List<FolderModel> folders,
                                            List<ServiceMinimalModel> services,
                                            List<ResourceModel> resources,
                                            List<UnitsEnsembleModel> unitsEnsembles,
                                            List<ProviderModel> providers) {
        this.transfers = transfers;
        this.folders = folders;
        this.services = services;
        this.resources = resources;
        this.unitsEnsembles = unitsEnsembles;
        this.providers = providers;
    }

    public static Builder builder() {
        return new Builder();
    }

    public List<ValidatedQuotaTransfer> getTransfers() {
        return transfers;
    }

    public List<FolderModel> getFolders() {
        return folders;
    }

    public List<ServiceMinimalModel> getServices() {
        return services;
    }

    public List<ResourceModel> getResources() {
        return resources;
    }

    public List<UnitsEnsembleModel> getUnitsEnsembles() {
        return unitsEnsembles;
    }

    public List<ProviderModel> getProviders() {
        return providers;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ValidatedQuotaTransferParameters that = (ValidatedQuotaTransferParameters) o;
        return Objects.equals(transfers, that.transfers) &&
                Objects.equals(folders, that.folders) &&
                Objects.equals(services, that.services) &&
                Objects.equals(resources, that.resources) &&
                Objects.equals(unitsEnsembles, that.unitsEnsembles) &&
                Objects.equals(providers, that.providers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(transfers, folders, services, resources, unitsEnsembles, providers);
    }

    @Override
    public String toString() {
        return "ValidatedQuotaTransferParameters{" +
                "transfers=" + transfers +
                ", folders=" + folders +
                ", services=" + services +
                ", resources=" + resources +
                ", unitsEnsembles=" + unitsEnsembles +
                ", providers=" + providers +
                '}';
    }

    public static final class Builder {

        private final List<ValidatedQuotaTransfer> transfers = new ArrayList<>();
        private final List<FolderModel> folders = new ArrayList<>();
        private final List<ServiceMinimalModel> services = new ArrayList<>();
        private final List<ResourceModel> resources = new ArrayList<>();
        private final List<UnitsEnsembleModel> unitsEnsembles = new ArrayList<>();
        private final List<ProviderModel> providers = new ArrayList<>();

        private Builder() {
        }

        public Builder addTransfer(ValidatedQuotaTransfer transfer) {
            this.transfers.add(transfer);
            return this;
        }

        public Builder addFolder(FolderModel folder) {
            this.folders.add(folder);
            return this;
        }

        public Builder addFolders(Collection<? extends FolderModel> folders) {
            this.folders.addAll(folders);
            return this;
        }

        public Builder addService(ServiceMinimalModel service) {
            this.services.add(service);
            return this;
        }

        public Builder addServices(Collection<? extends ServiceMinimalModel> services) {
            this.services.addAll(services);
            return this;
        }

        public Builder addResource(ResourceModel resource) {
            this.resources.add(resource);
            return this;
        }

        public Builder addResources(Collection<? extends ResourceModel> resources) {
            this.resources.addAll(resources);
            return this;
        }

        public Builder addUnitsEnsemble(UnitsEnsembleModel unitsEnsemble) {
            this.unitsEnsembles.add(unitsEnsemble);
            return this;
        }

        public Builder addUnitsEnsembles(Collection<? extends UnitsEnsembleModel> unitsEnsembles) {
            this.unitsEnsembles.addAll(unitsEnsembles);
            return this;
        }

        public Builder addProvider(ProviderModel provider) {
            this.providers.add(provider);
            return this;
        }

        public Builder addProviders(Collection<? extends ProviderModel> providers) {
            this.providers.addAll(providers);
            return this;
        }

        public ValidatedQuotaTransferParameters build() {
            return new ValidatedQuotaTransferParameters(transfers, folders, services, resources,
                    unitsEnsembles, providers);
        }

    }

}
