package ru.yandex.intranet.d.services.transfer.model;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;

/**
 * Validated reserve transfer.
 *
 * @author Ruslan Kadriev <aqru@yandex-team.ru>
 */
public final class ValidatedReserveTransfer {
    private final ProviderModel provider;
    private final FolderModel reserveFolder;
    private final FolderModel destinationFolder;
    private final ServiceMinimalModel destinationService;
    private final List<ValidatedQuotaResourceTransfer> transfers;

    public ValidatedReserveTransfer(ProviderModel provider, FolderModel reserveFolder, FolderModel destinationFolder,
                                    ServiceMinimalModel destinationService,
                                    List<ValidatedQuotaResourceTransfer> transfers) {
        this.provider = provider;
        this.reserveFolder = reserveFolder;
        this.destinationFolder = destinationFolder;
        this.destinationService = destinationService;
        this.transfers = transfers;
    }

    public ProviderModel getProvider() {
        return provider;
    }

    public FolderModel getReserveFolder() {
        return reserveFolder;
    }

    public FolderModel getDestinationFolder() {
        return destinationFolder;
    }

    public ServiceMinimalModel getDestinationService() {
        return destinationService;
    }

    public List<ValidatedQuotaResourceTransfer> getTransfers() {
        return transfers;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ValidatedReserveTransfer that = (ValidatedReserveTransfer) o;
        return Objects.equals(provider, that.provider) && Objects.equals(reserveFolder, that.reserveFolder)
                && Objects.equals(destinationFolder, that.destinationFolder)
                && Objects.equals(destinationService, that.destinationService)
                && Objects.equals(transfers, that.transfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(provider, reserveFolder, destinationFolder, destinationService, transfers);
    }

    @Override
    public String toString() {
        return "ValidatedReserveTransfer{" +
                "provider=" + provider +
                ", reserveFolder=" + reserveFolder +
                ", destinationFolder=" + destinationFolder +
                ", destinationService=" + destinationService +
                ", transfers=" + transfers +
                '}';
    }

    public static Builder builder() {
        return new Builder();
    }

    public static final class Builder {
        private final List<ValidatedQuotaResourceTransfer> transfers = new ArrayList<>();

        private ProviderModel provider;
        private FolderModel reserveFolder;
        private FolderModel destinationFolder;
        private ServiceMinimalModel destinationService;

        private Builder() {
        }

        public Builder destinationFolder(FolderModel destinationFolder) {
            this.destinationFolder = destinationFolder;
            return this;
        }

        public Builder reserveFolder(FolderModel reserveFolder) {
            this.reserveFolder = reserveFolder;
            return this;
        }

        public Builder destinationService(ServiceMinimalModel destinationService) {
            this.destinationService = destinationService;
            return this;
        }

        public Builder addTransfer(ValidatedQuotaResourceTransfer transfer) {
            this.transfers.add(transfer);
            return this;
        }

        public Builder provider(ProviderModel provider) {
            this.provider = provider;
            return this;
        }

        public ValidatedReserveTransfer build() {
            return new ValidatedReserveTransfer(
                    Preconditions.checkNotNull(provider),
                    Preconditions.checkNotNull(reserveFolder),
                    Preconditions.checkNotNull(destinationFolder),
                    Preconditions.checkNotNull(destinationService),
                    transfers);
        }
    }
}
