package ru.yandex.intranet.d.services.transfer.model;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestHistoryModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;

/**
 * Validated vote transfer request.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ValidatedVoteTransferRequest {

    private final TransferRequestModel transferRequest;
    private final TransferRequestHistoryModel history;
    private final TransferRequestIndices.Difference indicesDifference;
    private final List<QuotaModel> quotas;
    private final List<AccountsQuotasModel> accountsQuotas;
    private final List<FolderModel> folders;
    private final List<AccountModel> accounts;
    private final List<ServiceMinimalModel> services;
    private final List<ResourceModel> resources;
    private final List<UnitsEnsembleModel> unitsEnsembles;
    private final List<ProviderModel> providers;
    private final boolean apply;
    private final Map<String, String> preGeneratedFolderOpLogIdsByFolderId;
    private final Instant now;
    private final Set<UserModel> notifiedUsers;
    private final Set<String> operationIds;

    @SuppressWarnings("ParameterNumber")
    public ValidatedVoteTransferRequest(TransferRequestModel transferRequest,
                                        TransferRequestHistoryModel history,
                                        TransferRequestIndices.Difference indicesDifference,
                                        List<QuotaModel> quotas,
                                        List<AccountsQuotasModel> accountsQuotas,
                                        List<FolderModel> folders,
                                        List<AccountModel> accounts,
                                        List<ServiceMinimalModel> services,
                                        List<ResourceModel> resources,
                                        List<UnitsEnsembleModel> unitsEnsembles,
                                        List<ProviderModel> providers,
                                        boolean apply,
                                        Map<String, String> preGeneratedFolderOpLogIdsByFolderId,
                                        Instant now,
                                        Set<UserModel> notifiedUsers,
                                        Set<String> operationIds) {
        this.transferRequest = transferRequest;
        this.history = history;
        this.indicesDifference = indicesDifference;
        this.quotas = quotas;
        this.accountsQuotas = accountsQuotas;
        this.folders = folders;
        this.accounts = accounts;
        this.services = services;
        this.resources = resources;
        this.unitsEnsembles = unitsEnsembles;
        this.providers = providers;
        this.apply = apply;
        this.preGeneratedFolderOpLogIdsByFolderId = preGeneratedFolderOpLogIdsByFolderId;
        this.now = now;
        this.notifiedUsers = notifiedUsers;
        this.operationIds = operationIds;
    }

    public static Builder builder() {
        return new Builder();
    }

    public Builder copyBuilder() {
        return new Builder(this);
    }

    public TransferRequestModel getTransferRequest() {
        return transferRequest;
    }

    public TransferRequestHistoryModel getHistory() {
        return history;
    }

    public TransferRequestIndices.Difference getIndicesDifference() {
        return indicesDifference;
    }

    public List<QuotaModel> getQuotas() {
        return quotas;
    }

    public List<FolderModel> getFolders() {
        return folders;
    }

    public List<AccountsQuotasModel> getAccountsQuotas() {
        return accountsQuotas;
    }

    public List<AccountModel> getAccounts() {
        return accounts;
    }

    public List<ServiceMinimalModel> getServices() {
        return services;
    }

    public List<ResourceModel> getResources() {
        return resources;
    }

    public List<UnitsEnsembleModel> getUnitsEnsembles() {
        return unitsEnsembles;
    }

    public List<ProviderModel> getProviders() {
        return providers;
    }

    public boolean isApply() {
        return apply;
    }

    public Map<String, String> getPreGeneratedFolderOpLogIdsByFolderId() {
        return preGeneratedFolderOpLogIdsByFolderId;
    }

    public Instant getNow() {
        return now;
    }

    public Set<UserModel> getNotifiedUsers() {
        return notifiedUsers;
    }

    public Set<String> getOperationIds() {
        return operationIds;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ValidatedVoteTransferRequest that = (ValidatedVoteTransferRequest) o;
        return apply == that.apply &&
                Objects.equals(transferRequest, that.transferRequest) &&
                Objects.equals(history, that.history) &&
                Objects.equals(indicesDifference, that.indicesDifference) &&
                Objects.equals(quotas, that.quotas) &&
                Objects.equals(folders, that.folders) &&
                Objects.equals(services, that.services) &&
                Objects.equals(resources, that.resources) &&
                Objects.equals(unitsEnsembles, that.unitsEnsembles) &&
                Objects.equals(providers, that.providers) &&
                Objects.equals(preGeneratedFolderOpLogIdsByFolderId, that.preGeneratedFolderOpLogIdsByFolderId) &&
                Objects.equals(operationIds, that.operationIds) &&
                Objects.equals(now, that.now);
    }

    @Override
    public int hashCode() {
        return Objects.hash(transferRequest, history, indicesDifference, quotas,
                folders, services, resources, unitsEnsembles, providers, apply, preGeneratedFolderOpLogIdsByFolderId,
                now, operationIds);
    }

    @Override
    public String toString() {
        return "ValidatedVoteTransferRequest{" +
                "transferRequest=" + transferRequest +
                ", history=" + history +
                ", indicesDifference=" + indicesDifference +
                ", quotas=" + quotas +
                ", folders=" + folders +
                ", services=" + services +
                ", resources=" + resources +
                ", unitsEnsembles=" + unitsEnsembles +
                ", providers=" + providers +
                ", apply=" + apply +
                ", preGeneratedFolderOpLogIdsByFolderId=" + preGeneratedFolderOpLogIdsByFolderId +
                ", now=" + now +
                ", operationIds=" + operationIds +
                '}';
    }

    public static final class Builder {

        private final Map<String, String> preGeneratedFolderOpLogIdsByFolderId = new HashMap<>();
        private final Set<String> operationIds = new HashSet<>();
        private final List<AccountModel> accounts = new ArrayList<>();
        private final List<AccountsQuotasModel> accountsQuotas = new ArrayList<>();

        private TransferRequestIndices.Difference indicesDifference = TransferRequestIndices.Difference.empty();
        private TransferRequestModel transferRequest;
        private TransferRequestHistoryModel history;
        private List<QuotaModel> quotas;
        private List<FolderModel> folders;
        private List<ServiceMinimalModel> services;
        private List<ResourceModel> resources;
        private List<UnitsEnsembleModel> unitsEnsembles;
        private List<ProviderModel> providers;
        private Boolean apply;
        private Instant now;
        private Set<UserModel> notifiedUsers = new HashSet<>();

        private Builder() {
        }

        private Builder(ValidatedVoteTransferRequest original) {
            this.indicesDifference = original.indicesDifference;
            this.transferRequest = original.transferRequest;
            this.history = original.history;
            this.quotas = original.quotas;
            this.folders = original.folders;
            this.services = original.services;
            this.resources = original.resources;
            this.unitsEnsembles = original.unitsEnsembles;
            this.providers = original.providers;
            this.apply = original.apply;
            this.now = original.now;
            this.notifiedUsers = original.notifiedUsers;
            this.preGeneratedFolderOpLogIdsByFolderId.putAll(original.preGeneratedFolderOpLogIdsByFolderId);
            this.operationIds.addAll(original.operationIds);
            this.accounts.addAll(original.accounts);
            this.accountsQuotas.addAll(original.accountsQuotas);
        }

        public Builder transferRequest(TransferRequestModel transferRequest) {
            this.transferRequest = transferRequest;
            return this;
        }

        public Builder history(TransferRequestHistoryModel history) {
            this.history = history;
            return this;
        }

        public Builder indicesDifference(TransferRequestIndices.Difference indicesDifference) {
            this.indicesDifference = indicesDifference;
            return this;
        }

        public Builder quotas(List<QuotaModel> quotas) {
            this.quotas = quotas;
            return this;
        }

        public Builder folders(List<FolderModel> folders) {
            this.folders = folders;
            return this;
        }

        public Builder addAccounts(List<AccountModel> accounts) {
            this.accounts.addAll(accounts);
            return this;
        }

        public Builder addAccountsQuotas(List<AccountsQuotasModel> accountsQuotas) {
            this.accountsQuotas.addAll(accountsQuotas);
            return this;
        }

        public Builder services(List<ServiceMinimalModel> services) {
            this.services = services;
            return this;
        }

        public Builder resources(List<ResourceModel> resources) {
            this.resources = resources;
            return this;
        }

        public Builder unitsEnsembles(List<UnitsEnsembleModel> unitsEnsembles) {
            this.unitsEnsembles = unitsEnsembles;
            return this;
        }

        public Builder providers(List<ProviderModel> providers) {
            this.providers = providers;
            return this;
        }

        public Builder apply(boolean apply) {
            this.apply = apply;
            return this;
        }

        public Builder putPreGeneratedFolderOpLogId(String folderId, String opLogId) {
            this.preGeneratedFolderOpLogIdsByFolderId.put(folderId, opLogId);
            return this;
        }

        public Builder now(Instant now) {
            this.now = now;
            return this;
        }

        public Builder notifiedUsers(Set<UserModel> notifiedUsers) {
            this.notifiedUsers = notifiedUsers;
            return this;
        }

        public Builder addOperationId(String operationId) {
            this.operationIds.add(operationId);
            return this;
        }

        public ValidatedVoteTransferRequest build() {
            Preconditions.checkNotNull(transferRequest, "TransferRequest is required");
            Preconditions.checkNotNull(history, "History is required");
            Preconditions.checkNotNull(apply, "Apply is required");
            Preconditions.checkNotNull(quotas, "Quotas is required");
            Preconditions.checkNotNull(folders, "Folders is required");
            Preconditions.checkNotNull(services, "Services is required");
            Preconditions.checkNotNull(resources, "Resources is required");
            Preconditions.checkNotNull(unitsEnsembles, "UnitsEnsembles is required");
            Preconditions.checkNotNull(providers, "Providers is required");
            Preconditions.checkNotNull(now, "Now is required");
            return new ValidatedVoteTransferRequest(transferRequest, history, indicesDifference,
                    quotas, accountsQuotas, folders, accounts, services, resources, unitsEnsembles,
                    providers, apply, preGeneratedFolderOpLogIdsByFolderId, now, notifiedUsers, operationIds);
        }

    }

}
