package ru.yandex.intranet.d.services.transfer.model.dryrun;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import ru.yandex.intranet.d.model.transfers.TransferParameters;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.web.model.QuotaDto;
import ru.yandex.intranet.d.web.model.recipe.AccountQuotaDto;

/**
 * Transfer dry-run request.
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
public final class DryRunTransferRequestModel {

    private final TransferRequestType type;
    private final List<QuotaDto> quotasOld;
    private final List<QuotaDto> quotasNew;
    private final List<AccountQuotaDto> accountQuotasOld;
    private final List<AccountQuotaDto> accountQuotasNew;
    private final TransferParameters parameters;
    private final DryRunTransferParameters dryRunParameters;

    @SuppressWarnings("ParameterNumber")
    public DryRunTransferRequestModel(TransferRequestType type,
                                      List<QuotaDto> quotasOld,
                                      List<QuotaDto> quotasNew,
                                      List<AccountQuotaDto> accountQuotasOld,
                                      List<AccountQuotaDto> accountQuotasNew,
                                      TransferParameters parameters,
                                      DryRunTransferParameters dryRunParameters) {
        this.type = type;
        this.quotasOld = quotasOld;
        this.quotasNew = quotasNew;
        this.accountQuotasOld = accountQuotasOld;
        this.accountQuotasNew = accountQuotasNew;
        this.parameters = parameters;
        this.dryRunParameters = dryRunParameters;
    }

    public static Builder builder() {
        return new Builder();
    }

    public TransferRequestType getType() {
        return type;
    }

    public List<QuotaDto> getQuotasOld() {
        return quotasOld;
    }

    public List<QuotaDto> getQuotasNew() {
        return quotasNew;
    }

    public TransferParameters getParameters() {
        return parameters;
    }

    public List<AccountQuotaDto> getAccountQuotasOld() {
        return accountQuotasOld;
    }

    public List<AccountQuotaDto> getAccountQuotasNew() {
        return accountQuotasNew;
    }

    public Optional<DryRunTransferParameters> getDryRunParameters() {
        return Optional.ofNullable(dryRunParameters);
    }

    public static class Builder {
        private TransferRequestType type;
        private final List<QuotaDto> quotasOld = new ArrayList<>();
        private final List<QuotaDto> quotasNew = new ArrayList<>();
        private final List<AccountQuotaDto> accountQuotasOld = new ArrayList<>();
        private final List<AccountQuotaDto> accountQuotasNew = new ArrayList<>();
        private TransferParameters parameters;
        private DryRunTransferParameters dryRunParameters;

        private Builder() {
        }

        public Builder type(TransferRequestType type) {
            this.type = type;
            return this;
        }

        public Builder quotasOld(List<QuotaDto> quotas) {
            this.quotasOld.addAll(quotas);
            return this;
        }

        public Builder quotasNew(List<QuotaDto> quotas) {
            this.quotasNew.addAll(quotas);
            return this;
        }

        public Builder accountQuotasOld(List<AccountQuotaDto> accountQuotaOld) {
            this.accountQuotasOld.addAll(accountQuotaOld);
            return this;
        }

        public Builder accountQuotasNew(List<AccountQuotaDto> accountQuotasNew) {
            this.accountQuotasNew.addAll(accountQuotasNew);
            return this;
        }

        public Builder parameters(TransferParameters parameters) {
            this.parameters = parameters;
            return this;
        }

        public Builder dryRunParameters(DryRunTransferParameters dryRunParameters) {
            this.dryRunParameters = dryRunParameters;
            return this;
        }

        public DryRunTransferRequestModel build() {
            return new DryRunTransferRequestModel(type, quotasOld, quotasNew, accountQuotasOld, accountQuotasNew,
                    parameters, dryRunParameters);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DryRunTransferRequestModel that = (DryRunTransferRequestModel) o;
        return getType() == that.getType() && Objects.equals(getQuotasOld(), that.getQuotasOld()) &&
                Objects.equals(getQuotasNew(), that.getQuotasNew()) && Objects.equals(getAccountQuotasOld(),
                that.getAccountQuotasOld()) && Objects.equals(getAccountQuotasNew(), that.getAccountQuotasNew()) &&
                Objects.equals(getParameters(), that.getParameters()) && Objects.equals(getDryRunParameters(),
                that.getDryRunParameters());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getType(), getQuotasOld(), getQuotasNew(), getAccountQuotasOld(), getAccountQuotasNew(),
                getParameters(), getDryRunParameters());
    }

    @Override
    public String toString() {
        return "DryRunTransferRequestModel{" +
                "type=" + type +
                ", quotasOld=" + quotasOld +
                ", quotasNew=" + quotasNew +
                ", accountQuotasOld=" + accountQuotasOld +
                ", accountQuotasNew=" + accountQuotasNew +
                ", parameters=" + parameters +
                ", dryRunParameters=" + dryRunParameters +
                '}';
    }
}
