package ru.yandex.intranet.d.services.transfer.model.dryrun;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.transfers.TransferResponsible;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.util.result.ErrorCollection;

/**
 * Validated dry run create transfer request.
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
public final class DryRunTransferRequestResult {

    private final DryRunTransferRequestModel transferRequest;
    private final List<FolderModel> folders;
    private final List<ServiceMinimalModel> services;
    private final List<ResourceModel> resources;
    private final List<UnitsEnsembleModel> unitsEnsembles;
    private final List<ProviderModel> providers;
    private final TransferResponsible transferResponsible;
    private final DryRunTransferRequestWarnings warnings;
    private final DryRunTransferRequestPermissions permissions;
    private final ErrorCollection errors;

    @SuppressWarnings("ParameterNumber")
    public DryRunTransferRequestResult(DryRunTransferRequestModel transferRequest,
                                       List<FolderModel> folders,
                                       List<ServiceMinimalModel> services,
                                       List<ResourceModel> resources,
                                       List<UnitsEnsembleModel> unitsEnsembles,
                                       List<ProviderModel> providers,
                                       TransferResponsible transferResponsible,
                                       DryRunTransferRequestWarnings warnings,
                                       DryRunTransferRequestPermissions permissions, ErrorCollection errors) {
        this.transferRequest = transferRequest;
        this.folders = folders;
        this.services = services;
        this.resources = resources;
        this.unitsEnsembles = unitsEnsembles;
        this.providers = providers;
        this.transferResponsible = transferResponsible;
        this.warnings = warnings;
        this.permissions = permissions;
        this.errors = errors;
    }

    public static Builder builder() {
        return new Builder();
    }

    public DryRunTransferRequestModel getTransferRequest() {
        return transferRequest;
    }

    public List<FolderModel> getFolders() {
        return folders;
    }

    public List<ServiceMinimalModel> getServices() {
        return services;
    }

    public List<ResourceModel> getResources() {
        return resources;
    }

    public List<UnitsEnsembleModel> getUnitsEnsembles() {
        return unitsEnsembles;
    }

    public List<ProviderModel> getProviders() {
        return providers;
    }

    public Optional<TransferResponsible> getTransferResponsible() {
        return Optional.ofNullable(transferResponsible);
    }

    public DryRunTransferRequestWarnings getWarnings() {
        return warnings;
    }

    public Optional<DryRunTransferRequestPermissions> getPermissions() {
        return Optional.ofNullable(permissions);
    }

    public ErrorCollection getErrors() {
        return errors;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DryRunTransferRequestResult that = (DryRunTransferRequestResult) o;
        return Objects.equals(transferRequest, that.transferRequest) &&
                Objects.equals(folders, that.folders) &&
                Objects.equals(services, that.services) &&
                Objects.equals(resources, that.resources) &&
                Objects.equals(unitsEnsembles, that.unitsEnsembles) &&
                Objects.equals(providers, that.providers) &&
                Objects.equals(transferResponsible, that.transferResponsible) &&
                Objects.equals(warnings, that.warnings) &&
                Objects.equals(permissions, that.permissions) &&
                Objects.equals(errors, that.errors);
    }

    @Override
    public int hashCode() {
        return Objects.hash(transferRequest, folders, services, resources, unitsEnsembles, providers,
                transferResponsible, warnings, permissions, errors);
    }

    @Override
    public String toString() {
        return "DryRunTransferRequestResult{" +
                "transferRequest=" + transferRequest +
                ", folders=" + folders +
                ", services=" + services +
                ", resources=" + resources +
                ", unitsEnsembles=" + unitsEnsembles +
                ", providers=" + providers +
                ", transferResponsible=" + transferResponsible +
                ", warnings=" + warnings +
                ", permissions=" + permissions +
                ", errors=" + errors +
                '}';
    }

    public static final class Builder {
        private DryRunTransferRequestModel transferRequest;
        private List<FolderModel> folders;
        private List<ServiceMinimalModel> services;
        private List<ResourceModel> resources;
        private List<UnitsEnsembleModel> unitsEnsembles;
        private List<ProviderModel> providers;
        private TransferResponsible transferResponsible;
        private DryRunTransferRequestWarnings warnings;
        private DryRunTransferRequestPermissions permissions;
        private ErrorCollection errors = ErrorCollection.empty();

        private Builder() {
        }

        public Builder transferRequest(DryRunTransferRequestModel transferRequest) {
            this.transferRequest = transferRequest;
            return this;
        }

        public Builder responsible(TransferResponsible responsible) {
            transferResponsible = responsible;
            return this;
        }

        public Builder warnings(DryRunTransferRequestWarnings warnings) {
            this.warnings = warnings;
            return this;
        }

        public Builder permissions(DryRunTransferRequestPermissions permissions) {
            this.permissions = permissions;
            return this;
        }

        public Builder folders(List<FolderModel> folders) {
            this.folders = folders;
            return this;
        }

        public Builder services(List<ServiceMinimalModel> services) {
            this.services = services;
            return this;
        }

        public Builder resources(List<ResourceModel> resources) {
            this.resources = resources;
            return this;
        }

        public Builder unitsEnsembles(List<UnitsEnsembleModel> unitsEnsembles) {
            this.unitsEnsembles = unitsEnsembles;
            return this;
        }

        public Builder providers(List<ProviderModel> providers) {
            this.providers = providers;
            return this;
        }

        public Builder errors(ErrorCollection errors) {
            this.errors = errors;
            return this;
        }

        public DryRunTransferRequestResult build() {
            Preconditions.checkNotNull(transferRequest, "TransferRequest is required");
            Preconditions.checkNotNull(folders, "Folders is required");
            Preconditions.checkNotNull(services, "Services is required");
            Preconditions.checkNotNull(resources, "Resources is required");
            Preconditions.checkNotNull(unitsEnsembles, "UnitsEnsembles is required");
            Preconditions.checkNotNull(providers, "Providers is required");
            Preconditions.checkNotNull(warnings, "Warnings is required");
            Preconditions.checkNotNull(errors, "Errors is required");
            return new DryRunTransferRequestResult(transferRequest, folders, services, resources,
                    unitsEnsembles, providers, transferResponsible, warnings, permissions, errors);
        }

    }
}
