package ru.yandex.intranet.d.services.transfer.ticket

import org.springframework.stereotype.Component
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.folders.FolderType
import ru.yandex.intranet.d.model.services.ServiceMinimalModel
import ru.yandex.intranet.d.model.transfers.QuotaTransfer
import ru.yandex.intranet.d.model.transfers.ResourceQuotaTransfer
import ru.yandex.intranet.d.model.transfers.TransferRequestModel
import ru.yandex.intranet.d.model.transfers.TransferRequestType
import ru.yandex.intranet.d.model.users.UserModel
import ru.yandex.intranet.d.services.quotas.QuotasHelper
import ru.yandex.intranet.d.services.transfer.TextFormatterService
import ru.yandex.intranet.d.services.transfer.model.ExpandedTransferRequests
import ru.yandex.intranet.d.util.DisplayUtil.getAccountDisplayString
import ru.yandex.intranet.d.util.FrontStringUtil
import java.math.BigDecimal
import java.util.stream.Collectors

/**
 * Service for generate ticket description
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Component
class TransferRequestTicketDescriptionService(
    private val textFormatterService: TextFormatterService
) {

    public fun getSummary(expandedTransferRequests: ExpandedTransferRequests<TransferRequestModel>): String {
        return expandedTransferRequests.transferRequests.summary.orElseThrow()
    }

    public fun getDescription(expandedTransferRequests: ExpandedTransferRequests<TransferRequestModel>): String {
        return when (expandedTransferRequests.transferRequests.type) {
            TransferRequestType.QUOTA_TRANSFER -> getDescriptionForQuotaTransfer(expandedTransferRequests)
            TransferRequestType.RESERVE_TRANSFER -> getDescriptionForReserve(expandedTransferRequests)
            TransferRequestType.PROVISION_TRANSFER -> getDescriptionForProvisionTransfer(expandedTransferRequests)
            else -> throw IllegalArgumentException("Invalid transfer request type: " +
                expandedTransferRequests.transferRequests.type)
        }
    }

    public fun getDescriptionForQuotaTransfer(
        expandedTransferRequests: ExpandedTransferRequests<TransferRequestModel>
    ): String {
        return if (expandedTransferRequests.transferRequests.parameters.quotaTransfers.size == 2) {
            getDescriptionForQuotaTransferBetweenTwoService(expandedTransferRequests)
        } else {
            getDescriptionForQuotaTransferBetweenManyService(expandedTransferRequests)
        }
    }

    private fun getDescriptionForQuotaTransferBetweenTwoService(
        expandedTransferRequests: ExpandedTransferRequests<TransferRequestModel>
    ): String {
        val quotaTransferSet: MutableSet<QuotaTransfer> =
            HashSet(expandedTransferRequests.transferRequests.parameters.quotaTransfers)
        val quotaTransferSourceOptional = expandedTransferRequests.transferRequests.parameters.quotaTransfers
            .stream().filter { transferHasNegativeQuota(it) }.findFirst()
        val quotaTransferSource = quotaTransferSourceOptional.orElseThrow()
        quotaTransferSet.remove(quotaTransferSource)
        val quotaTransferDestinationOptional = quotaTransferSet.stream().findFirst()
        val quotaTransferDestination = quotaTransferDestinationOptional.orElseThrow()
        val sb: StringBuilder = StringBuilder()
            .append("Сервис-источник: ")
            .append(textFormatterService.buildAbcLink(
                expandedTransferRequests.services[quotaTransferSource.destinationServiceId.toString()]!!.slug))
            .append("\nПодтверждающие: ")
            .append(convertServiceResponsibleToString(
                quotaTransferSource.destinationFolderId, expandedTransferRequests))
            .append("\nСервис-получатель: ")
            .append(textFormatterService.buildAbcLink(
                expandedTransferRequests.services[quotaTransferDestination.destinationServiceId.toString()]!!.slug))
            .append("\nПодтверждающие: ")
            .append(convertServiceResponsibleToString(
                quotaTransferDestination.destinationFolderId, expandedTransferRequests))
            .append("\nЗаявка в ABCD: ")
            .append(textFormatterService.buildTransferUrlWithName(expandedTransferRequests.transferRequests))
            .append("\n")
            .append(convertResourceTransfersToString(
                quotaTransferDestination.transfers, expandedTransferRequests))
        expandedTransferRequests.transferRequests.description
            .filter { it.isNotBlank() }
            .ifPresent { sb.append("\n\nКомментарий:\n").append(it) }
        return sb.toString()
    }

    private fun getDescriptionForQuotaTransferBetweenManyService(
        expandedTransferRequests: ExpandedTransferRequests<TransferRequestModel>
    ): String {
        val sb = StringBuilder()
        for (quotaTransfer in expandedTransferRequests.transferRequests.parameters.quotaTransfers) {
            if (transferHasNegativeQuota(quotaTransfer)) {
                sb.append("Сервис-источник: ")
                    .append(textFormatterService.buildAbcLink(
                        expandedTransferRequests.services[quotaTransfer.destinationServiceId.toString()]!!.slug))
                    .append("\nПодтверждающие: ")
                    .append(convertServiceResponsibleToString(
                        quotaTransfer.destinationFolderId, expandedTransferRequests))
                    .append("\n")
                    .append(quotaTransfer.transfers.stream()
                        .filter { it.delta < 0 }
                        .map { convertResourceTransferToString(it, expandedTransferRequests) }
                        .collect(Collectors.joining("\n")))
                    .append("\n")
            }
        }
        for (quotaTransfer in expandedTransferRequests.transferRequests.parameters.quotaTransfers) {
            if (transferHasPositiveQuota(quotaTransfer)) {
                sb.append("Сервис-получатель: ")
                    .append(textFormatterService.buildAbcLink(
                        expandedTransferRequests.services[quotaTransfer.destinationServiceId.toString()]!!.slug))
                    .append("\nПодтверждающие: ")
                    .append(convertServiceResponsibleToString(
                        quotaTransfer.destinationFolderId, expandedTransferRequests))
                    .append("\n")
                    .append(quotaTransfer.transfers.stream()
                        .filter { it.delta > 0 }
                        .map { convertResourceTransferToString(it, expandedTransferRequests) }
                        .collect(Collectors.joining("\n")))
                    .append("\n")
            }
        }
        sb.append("\n\nЗаявка в ABCD: ")
        sb.append(textFormatterService.buildTransferUrlWithName(expandedTransferRequests.transferRequests))
        expandedTransferRequests.transferRequests.description
            .filter { it.isNotBlank() }
            .ifPresent { sb.append("\n\nКомментарий:\n").append(it) }
        return sb.toString()
    }

    public fun getDescriptionForReserve(
        expandedTransferRequests: ExpandedTransferRequests<TransferRequestModel>
    ): String {
        val providerServiceIds = expandedTransferRequests.providers.values.stream()
            .map { it.serviceId }.collect(Collectors.toSet())
        val transferSource = expandedTransferRequests.transferRequests.parameters.quotaTransfers.stream()
            .filter {
                val folder = expandedTransferRequests.folders[it.destinationFolderId]!!
                folder.folderType == FolderType.PROVIDER_RESERVE && providerServiceIds.contains(folder.serviceId)
            }.findFirst().orElseThrow()
        val transferDestination = expandedTransferRequests.transferRequests.parameters.quotaTransfers.stream()
            .filter { it.destinationFolderId != transferSource.destinationFolderId }
            .findFirst().orElseThrow()
        val destinationFolderId = transferDestination.destinationFolderId
        val destinationServiceId = transferDestination.destinationServiceId
        val sourceServiceId = transferSource.destinationServiceId
        val sb = StringBuilder()
            .append("Сервис-получатель: ")
            .append(textFormatterService.buildAbcLink(
                expandedTransferRequests.services[destinationServiceId.toString()]!!.slug))
            .append("\nПодтверждающие: ")
            .append(convertServiceResponsibleToString(destinationFolderId, expandedTransferRequests))
            .append("\nПровайдер-источник: ")
            .append(textFormatterService.buildAbcLink(
                expandedTransferRequests.services[sourceServiceId.toString()]!!.slug))
            .append("\nПодтверждающие от провайдера: ")
            .append(convertReserveResponsibleToString(expandedTransferRequests))
            .append("\nЗаявка в ABCD: ")
            .append(textFormatterService.buildTransferUrlWithName(expandedTransferRequests.transferRequests))
            .append("\n")
            .append(convertResourceTransfersToString(transferDestination.transfers, expandedTransferRequests))
            .append("\n")
        expandedTransferRequests.transferRequests.description
            .filter { it.isNotBlank() }
            .ifPresent { sb.append("\nКомментарий:\n").append(it) }
        return sb.toString()
    }

    public fun getDescriptionForProvisionTransfer(
        expandedTransferRequests: ExpandedTransferRequests<TransferRequestModel>
    ): String {
        val sb = StringBuilder()
        val transfersByAccount: MutableMap<String, MutableList<ResourceQuotaTransfer>> = HashMap()
        val accountsByFolderId: MutableMap<String, MutableSet<AccountModel>> = HashMap()
        val foldersByServiceId: MutableMap<Long, MutableSet<FolderModel>> = HashMap()
        val transfersByAccountPair: MutableMap<Pair<String, String>, MutableList<ResourceQuotaTransfer>> = HashMap()

        for (provisionTransfer in expandedTransferRequests.transferRequests.parameters.provisionTransfers) {
            val sourceAccountTransfers = transfersByAccount
                .computeIfAbsent(provisionTransfer.sourceAccountId) { ArrayList() }
            sourceAccountTransfers.addAll(provisionTransfer.sourceAccountTransfers)
            val destinationAccountTransfers = transfersByAccount
                .computeIfAbsent(provisionTransfer.destinationAccountId) { ArrayList() }
            destinationAccountTransfers.addAll(provisionTransfer.destinationAccountTransfers)
            val sourceAccount = expandedTransferRequests.accounts[provisionTransfer.sourceAccountId]!!
            val destinationAccount = expandedTransferRequests.accounts[provisionTransfer.destinationAccountId]!!
            accountsByFolderId.computeIfAbsent(sourceAccount.folderId) { HashSet() }.add(sourceAccount)
            accountsByFolderId.computeIfAbsent(destinationAccount.folderId) { HashSet() }.add(destinationAccount)
            val sourceFolder = expandedTransferRequests.folders[sourceAccount.folderId]!!
            val destinationFolder = expandedTransferRequests.folders[destinationAccount.folderId]!!
            foldersByServiceId.computeIfAbsent(sourceFolder.serviceId) { HashSet() }.add(sourceFolder)
            foldersByServiceId.computeIfAbsent(destinationFolder.serviceId) { HashSet() }.add(destinationFolder)
            val backwardTransfers = provisionTransfer.sourceAccountTransfers.stream()
                .filter { it.delta > 0 }
                .toList()
            val forwardTransfers = provisionTransfer.destinationAccountTransfers.stream()
                .filter { it.delta > 0 }
                .toList()
            if (forwardTransfers.isNotEmpty()) {
                transfersByAccountPair.computeIfAbsent(Pair(sourceAccount.id, destinationAccount.id)) { ArrayList() }
                    .addAll(forwardTransfers)
            }
            if (backwardTransfers.isNotEmpty()) {
                transfersByAccountPair.computeIfAbsent(Pair(destinationAccount.id, sourceAccount.id)) { ArrayList() }
                    .addAll(backwardTransfers)
            }
        }
        foldersByServiceId.forEach { (serviceId: Long, folders: Set<FolderModel>) ->
            val service = expandedTransferRequests.services[serviceId.toString()]!!
            val folderIds = folders.stream().map { it.id }.collect(Collectors.toSet())
            sb.append("Сервис: ")
                .append(textFormatterService.buildAbcLink(service.slug))
                .append("\nПодтверждающие: ")
                .append(convertServiceResponsibleToString(folderIds, expandedTransferRequests))
            folders.stream()
                .sorted(Comparator.comparing { it.displayName })
                .forEachOrdered { folder: FolderModel ->
                    sb.append("\nФолдер: ")
                        .append(folder.displayName)
                        .append(" (")
                        .append(folder.id)
                        .append(")")
                    val accounts: Set<AccountModel> = accountsByFolderId[folder.id]!!
                    sb.append(if (accounts.size > 1) "\nАккаунты:" else "\nАккаунт:")
                    accounts.stream()
                        .map { getAccountDisplayString(it) }
                        .sorted()
                        .forEachOrdered { accountString: String? -> sb.append("\n* ").append(accountString) }
                    sb.append("\n")
                }
            sb.append("\n\n")
        }
        transfersByAccountPair.forEach { (accountPair: Pair<String, String>, transfers: List<ResourceQuotaTransfer>) ->
            val sourceAccount = expandedTransferRequests.accounts[accountPair.first]!!
            val sourceFolder = expandedTransferRequests.folders[sourceAccount.folderId]!!
            val sourceService = expandedTransferRequests.services[sourceFolder.serviceId.toString()]!!
            val destinationAccount = expandedTransferRequests.accounts[accountPair.second]!!
            val destinationFolder = expandedTransferRequests.folders[destinationAccount.folderId]!!
            val destinationService = expandedTransferRequests.services[destinationFolder.serviceId.toString()]!!
            sb.append(fullAccountName(sourceAccount, sourceFolder, sourceService))
                .append(" -> ")
                .append(fullAccountName(destinationAccount, destinationFolder, destinationService))
                .append("\n")
                .append(
                    transfers.stream()
                        .sorted(Comparator.comparing { it.resourceId })
                        .map { convertResourceTransferToString(it, expandedTransferRequests) }
                        .collect(Collectors.joining("\n")))
                .append("\n\n")
        }
        sb.append("Заявка в ABCD: ")
        sb.append(textFormatterService.buildTransferUrlWithName(expandedTransferRequests.transferRequests))
        expandedTransferRequests.transferRequests.description
            .filter { it.isNotBlank() }
            .ifPresent { sb.append("\n\nКомментарий:\n").append(it) }
        return sb.toString()
    }

    private fun fullAccountName(
        account: AccountModel, accountFolder: FolderModel, service: ServiceMinimalModel
    ): String {
        return (service.slug + ":" + accountFolder.displayName + ":" + getAccountDisplayString(account))
    }

    private fun transferHasNegativeQuota(quotaTransfer: QuotaTransfer): Boolean {
        return transferHasNegativeQuota(quotaTransfer.transfers)
    }

    private fun transferHasNegativeQuota(transfers: Collection<ResourceQuotaTransfer>): Boolean {
        return transfers.stream().anyMatch { it.delta < 0 }
    }

    private fun transferHasPositiveQuota(quotaTransfer: QuotaTransfer): Boolean {
        return transferHasPositiveQuota(quotaTransfer.transfers)
    }

    private fun transferHasPositiveQuota(transfers: Collection<ResourceQuotaTransfer>): Boolean {
        return transfers.stream().anyMatch { it.delta > 0 }
    }

    private fun convertResourceTransfersToString(
        resourceQuotaTransfers: Set<ResourceQuotaTransfer>,
        expandedTransferRequests: ExpandedTransferRequests<TransferRequestModel>
    ): String {
        val resourceTransfersByProvider = resourceQuotaTransfers.stream()
            .filter { it.delta > 0 }
            .collect(Collectors.groupingBy(
                { expandedTransferRequests.resources[it.resourceId]!!.providerId }, Collectors.toSet()
            ))
        val sb = StringBuilder()
        for ((providerId, resourceTransfers) in resourceTransfersByProvider) {
            val providerName = expandedTransferRequests.providers[providerId]!!.nameRu
            sb.append("\n")
            sb.append(textFormatterService.buildBoldText("Провайдер: $providerName"))
            sb.append("\n")
            sb.append(resourceTransfers.stream()
                .map { convertResourceTransferToString(it, expandedTransferRequests) }
                .collect(Collectors.joining("\n")))
        }
        return sb.toString()
    }

    private fun convertResourceTransferToString(
        resourceQuotaTransfer: ResourceQuotaTransfer,
        expandedTransferRequests: ExpandedTransferRequests<TransferRequestModel>
    ): String {
        val resource = expandedTransferRequests.resources[resourceQuotaTransfer.resourceId]!!
        val unitsEnsemble = expandedTransferRequests.unitsEnsembles[resource.unitsEnsembleId]!!
        val decimalWithUnit = QuotasHelper.convertToReadable(
            BigDecimal.valueOf(resourceQuotaTransfer.delta),
            unitsEnsemble.units.stream()
                .sorted(Comparator.comparingLong { it.power })
                .toList(),
            expandedTransferRequests.units[resource.baseUnitId]
        )
        val name = resource.nameRu
        val delta = FrontStringUtil.toString(QuotasHelper.roundForDisplay(decimalWithUnit.amount))
        val unit = decimalWithUnit.unit.shortNameSingularEn
        return "$name: $delta $unit"
    }

    private fun convertServiceResponsibleToString(
        folderId: String, expandedTransferRequests: ExpandedTransferRequests<TransferRequestModel>
    ): String {
        return convertServiceResponsibleToString(setOf(folderId), expandedTransferRequests)
    }

    private fun convertServiceResponsibleToString(
        folderIds: Set<String>, expandedTransferRequests: ExpandedTransferRequests<TransferRequestModel>
    ): String {
        val userResponsibleIds: MutableSet<String> = HashSet()
        for (foldersResponsible in expandedTransferRequests.transferRequests.responsible.responsible) {
            if (foldersResponsible.folderIds.containsAll(folderIds)) {
                userResponsibleIds.addAll(foldersResponsible.responsible.stream()
                    .map { it.responsibleIds }
                    .collect(
                        { HashSet() },
                        { obj: HashSet<String>, c: Set<String> -> obj.addAll(c) },
                        { obj: HashSet<String>, c: HashSet<String> -> obj.addAll(c) }
                    )
                )
            }
        }
        userResponsibleIds.addAll(expandedTransferRequests.transferRequests.responsible.providerResponsible.stream()
            .map { it.responsibleId }
            .collect(Collectors.toSet()))
        return convertResponsibleIdsToStaffLinks(userResponsibleIds, expandedTransferRequests.users)
    }

    private fun convertReserveResponsibleToString(
        expandedTransferRequests: ExpandedTransferRequests<TransferRequestModel>
    ): String {
        val userResponsibleIds: MutableSet<String> = HashSet()
        expandedTransferRequests.transferRequests.responsible.reserveResponsibleModel
            .ifPresent { userResponsibleIds.addAll(it.responsibleIds) }
        userResponsibleIds.addAll(expandedTransferRequests.transferRequests.responsible.providerResponsible.stream()
            .map { it.responsibleId }
            .collect(Collectors.toSet()))
        return convertResponsibleIdsToStaffLinks(userResponsibleIds, expandedTransferRequests.users)
    }

    private fun convertResponsibleIdsToStaffLinks(
        userResponsibleIds: Set<String>,
        userById: Map<String, UserModel>
    ): String {
        return userResponsibleIds.stream()
            .map { userById[it]!! }
            .map { it.passportLogin }
            .filter { it.isPresent }
            .map { it.get() }
            .map { textFormatterService.buildStaffLink(it) }
            .collect(Collectors.joining(", "))
    }
}
