package ru.yandex.intranet.d.services.transfer.ticket;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;

import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.services.ServiceMinimalModel;
import ru.yandex.intranet.d.model.transfers.ProvisionTransfer;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestSubtype;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.services.transfer.model.ExpandedTransferRequests;

/**
 * Service for generate ticket meta information
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Component
public class TransferRequestTicketMetaInfoService {
    private final Environment env;

    public TransferRequestTicketMetaInfoService(Environment env) {
        this.env = env;
    }

    public List<String> getTags(ExpandedTransferRequests<TransferRequestModel> expandedTransferRequests) {
        if (expandedTransferRequests.getTransferRequests().getType() == TransferRequestType.PROVISION_TRANSFER) {
            return getProvisionTransferTags(expandedTransferRequests);
        } else {
            return getCommonTags(expandedTransferRequests);
        }
    }

    private List<String> getCommonTags(ExpandedTransferRequests<TransferRequestModel> expandedTransferRequests) {
        return expandedTransferRequests.getServices().values().stream()
                .map(ServiceMinimalModel::getSlug).toList();
    }

    private List<String> getProvisionTransferTags(
            ExpandedTransferRequests<TransferRequestModel> expandedTransferRequests
    ) {
        Map<String, ServiceMinimalModel> services = expandedTransferRequests.getServices();
        List<String> tags = new ArrayList<>();
        expandedTransferRequests.getTransferRequests()
                .getParameters()
                .getProvisionTransfers()
                .forEach(pt -> {
                    tags.add(services.get(Long.toString(pt.getSourceServiceId())).getSlug());
                    tags.add(services.get(Long.toString(pt.getDestinationServiceId())).getSlug());
                });
        return tags;
    }

    public TrackerTransferType getTransferType(
            ExpandedTransferRequests<TransferRequestModel> expandedTransferRequests
    ) {
        return switch (getTransferRequestSubtype(expandedTransferRequests)) {
            case DEFAULT_QUOTA_TRANSFER -> TrackerTransferType.TRANSFER;
            case DEFAULT_RESERVE_TRANSFER -> TrackerTransferType.RESERVE;
            case DEFAULT_PROVISION_TRANSFER -> TrackerTransferType.PROVISION;
            case RESERVE_PROVISION_TRANSFER -> TrackerTransferType.RESERVE;
            case EXCHANGE_PROVISION_TRANSFER -> TrackerTransferType.EXCHANGE;
            case LOAN_PROVISION_TRANSFER -> TrackerTransferType.LOAN;
            case DEFAULT_FOLDER_TRANSFER, DEFAULT_ACCOUNT_TRANSFER ->
                    throw new IllegalStateException("Unknown transfer request subtype: " +
                            getTransferRequestSubtype(expandedTransferRequests));
        };
    }

    private TransferRequestSubtype getTransferRequestSubtype(
            ExpandedTransferRequests<TransferRequestModel> expandedTransferRequests
    ) {
        return expandedTransferRequests.getTransferRequests().getSubtype().orElseGet(() -> {
            TransferRequestModel transferRequest = expandedTransferRequests.getTransferRequests();
            if (transferRequest.getType() == TransferRequestType.QUOTA_TRANSFER) {
                return TransferRequestSubtype.DEFAULT_QUOTA_TRANSFER;
            }
            if (transferRequest.getType() == TransferRequestType.RESERVE_TRANSFER) {
                return TransferRequestSubtype.DEFAULT_RESERVE_TRANSFER;
            }
            if (transferRequest.getType() == TransferRequestType.PROVISION_TRANSFER) {
                if (transferRequest.getLoanMeta().isPresent()) {
                    return TransferRequestSubtype.LOAN_PROVISION_TRANSFER;
                }
                Set<ProvisionTransfer> provisionTransfers = transferRequest.getParameters().getProvisionTransfers();
                long countTransfersWithReserveAccounts = provisionTransfers
                        .stream()
                        .filter(t -> isReserveAccount(t.getSourceAccountId(), expandedTransferRequests) ||
                                isReserveAccount(t.getDestinationAccountId(), expandedTransferRequests))
                        .count();
                if (countTransfersWithReserveAccounts == 0L) {
                    return TransferRequestSubtype.DEFAULT_PROVISION_TRANSFER;
                } else {
                    Set<String> sourceFolderIds = new HashSet<>();
                    Set<String> destinationFolderIds = new HashSet<>();
                    for (ProvisionTransfer provisionTransfer : provisionTransfers) {
                        sourceFolderIds.add(provisionTransfer.getSourceFolderId());
                        destinationFolderIds.add(provisionTransfer.getDestinationFolderId());
                    }
                    boolean isExchangeTransfer = provisionTransfers.size() == 2 &&
                            sourceFolderIds.size() == 2 &&
                            sourceFolderIds.equals(destinationFolderIds) &&
                            provisionTransfers.stream()
                                    .allMatch(t ->
                                            expandedTransferRequests.getAccounts()
                                                    .get(t.getSourceAccountId()).getReserveType().isPresent() ^
                                            expandedTransferRequests.getAccounts()
                                                    .get(t.getDestinationAccountId()).getReserveType().isPresent());
                    if (isExchangeTransfer) {
                        return TransferRequestSubtype.EXCHANGE_PROVISION_TRANSFER;
                    } else {
                        return TransferRequestSubtype.RESERVE_PROVISION_TRANSFER;
                    }
                }
            }
            throw new IllegalStateException("Invalid transfer request type: " + transferRequest.getType());
        });
    }

    private boolean isReserveAccount(String accountId,
                                     ExpandedTransferRequests<TransferRequestModel> expandedTransferRequests) {
        return expandedTransferRequests.getAccounts().get(accountId).getReserveType().isPresent();
    }

    public List<Long> getComponentIds(ExpandedTransferRequests<TransferRequestModel> expandedTransferRequests) {
        return getComponentIds(getTransferType(expandedTransferRequests),
                expandedTransferRequests.getProviders().values());
    }

    public List<Long> getComponentIds(TrackerTransferType requestType, Collection<ProviderModel> providers) {
        Long componentRequestId = env.getProperty(requestType.getComponentEnvValue(), Long.class);
        if (componentRequestId == null) {
            throw new IllegalStateException("Property " + requestType.getComponentEnvValue() + " not found");
        }
        List<Long> componentIds = providers.stream()
                .map(ProviderModel::getTrackerComponentId).collect(Collectors.toList());
        componentIds.add(componentRequestId);
        return componentIds;
    }

    public String getAuthorLogin(ExpandedTransferRequests<TransferRequestModel> expandedTransferRequests) {
        return expandedTransferRequests.getUsers().get(expandedTransferRequests.getTransferRequests().getCreatedBy())
                .getPassportLogin().orElseThrow();
    }
}
