package ru.yandex.intranet.d.services.transfer.ticket;

import java.util.function.Function;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.transfers.TransferRequestsDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestStatus;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.services.tracker.TrackerClient;
import ru.yandex.intranet.d.services.tracker.TrackerIssueResolution;
import ru.yandex.intranet.d.services.transfer.model.ExpandedTransferRequests;
import ru.yandex.intranet.d.services.transfer.model.TransferRequestCreationResult;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.tracker.TrackerCreateTicketDto;
import ru.yandex.intranet.d.web.model.tracker.TrackerCreateTicketResponseDto;
import ru.yandex.intranet.d.web.model.tracker.TrackerTransitionExecuteDto;
import ru.yandex.intranet.d.web.model.tracker.TrackerUpdateTicketDto;

@Component
public class TransferRequestTicketService {

    private static final Logger LOG = LoggerFactory.getLogger(TransferRequestTicketService.class);

    private final YdbTableClient tableClient;
    private final TrackerClient trackerClient;
    private final String trackerTransferQueue;
    private final TransferRequestsDao transferRequestsDao;
    private final TransferRequestTicketDescriptionService ticketDescriptionService;
    private final TransferRequestTicketMetaInfoService ticketMetaInfoService;

    public TransferRequestTicketService(YdbTableClient tableClient,
                                        TrackerClient trackerClient,
                                        @Value("${tracker.transfer.queue}") String trackerTransferQueue,
                                        TransferRequestsDao transferRequestsDao,
                                        TransferRequestTicketDescriptionService ticketDescriptionService,
                                        TransferRequestTicketMetaInfoService ticketMetaInfoService) {
        this.tableClient = tableClient;
        this.trackerClient = trackerClient;
        this.trackerTransferQueue = trackerTransferQueue;
        this.transferRequestsDao = transferRequestsDao;
        this.ticketDescriptionService = ticketDescriptionService;
        this.ticketMetaInfoService = ticketMetaInfoService;
    }

    public Mono<Result<Tuple2<ExpandedTransferRequests<TransferRequestModel>, TransferRequestCreationResult>>>
    createTicket(Tuple2<ExpandedTransferRequests<TransferRequestModel>, TransferRequestCreationResult> data) {
        ExpandedTransferRequests<TransferRequestModel> expandedTransferRequests = data.getT1();
        TransferRequestType type = expandedTransferRequests.getTransferRequests().getType();

        if (!isValidTransferRequestType(type)) {
            LOG.error("Failed to create ticket. Unexpected transfer request type: {}", type);
            return Mono.just(Result.success(data));
        }
        Mono<Result<TrackerCreateTicketResponseDto>> result = trackerClient.createTicket(
                new TrackerCreateTicketDto(
                        trackerTransferQueue,
                        ticketDescriptionService.getSummary(expandedTransferRequests),
                        ticketDescriptionService.getDescription(expandedTransferRequests),
                        ticketMetaInfoService.getAuthorLogin(expandedTransferRequests),
                        ticketMetaInfoService.getComponentIds(expandedTransferRequests),
                        ticketMetaInfoService.getTags(expandedTransferRequests),
                        createUnique(expandedTransferRequests.getTransferRequests().getId())
                ));
        if (expandedTransferRequests.getTransferRequests().getStatus() == TransferRequestStatus.APPLIED) {
            result = result.flatMap(res ->
                    res.andThenMono(resp -> trackerClient.closeTicket(resp.getKey(), "close",
                                    new TrackerTransitionExecuteDto(TrackerIssueResolution.FIXED)))
                            .map(u -> res));
        }
        return result.flatMap(res -> res.andThenMono(ticket ->
                        updateTrackerIssueKeyInDb(expandedTransferRequests.getTransferRequests(), ticket.getKey())
                                .map(res2 -> res2.apply(u -> updateTrackerIssueKeyInResult(data, ticket.getKey())))))
                //log and ignore if Result.failure
                .map(res -> Result.success(res.match(Function.identity(), error -> {
                    LOG.error("Failed to create ticket. Error: {}", error);
                    return data;
                })))
                //log and ignore if Mono.error
                .doOnError(throwable -> LOG.error("Failed to create ticket.", throwable))
                .onErrorReturn(Result.success(data));
    }

    public <T> Mono<Result<Tuple2<ExpandedTransferRequests<TransferRequestModel>, T>>>
    updateTicket(ExpandedTransferRequests<TransferRequestModel> expandedTransferRequests, T returnT2Data) {
        TransferRequestType type = expandedTransferRequests.getTransferRequests().getType();
        Tuple2<ExpandedTransferRequests<TransferRequestModel>, T> returnData = Tuples.of(
                expandedTransferRequests, returnT2Data
        );
        String trackerIssueKey = expandedTransferRequests.getTransferRequests().getTrackerIssueKey().orElse(null);

        if (!isValidTransferRequestType(type)) {
            LOG.error("Failed to update ticket. Unexpected transfer request type: {}", type);
            return Mono.just(Result.success(returnData));
        }
        if (trackerIssueKey == null || trackerIssueKey.isEmpty()) {
            return Mono.just(Result.failure(ErrorCollection.builder().addError(
                    TypedError.badRequest("Not found trackerIssueKey for ticket update")).build()));
        }
        Mono<Result<Void>> result = trackerClient.updateTicket(trackerIssueKey, new TrackerUpdateTicketDto(
                ticketDescriptionService.getSummary(expandedTransferRequests),
                ticketDescriptionService.getDescription(expandedTransferRequests),
                ticketMetaInfoService.getComponentIds(expandedTransferRequests),
                ticketMetaInfoService.getTags(expandedTransferRequests)
        ));
        return result
                //log and ignore if Result.failure
                .map(res -> {
                    res.doOnFailure(error -> LOG.error("Failed to update ticket. Error: {}", error));
                    return Result.success(returnData);
                })
                //log and ignore if Mono.error
                .doOnError(throwable -> LOG.error("Failed to update ticket.", throwable))
                .onErrorReturn(Result.success(returnData));
    }

    public <T> Mono<Result<T>> closeTicket(TransferRequestModel transferRequestModel, T returnData) {
        String trackerIssueKey = transferRequestModel.getTrackerIssueKey().orElse(null);
        if (trackerIssueKey == null || trackerIssueKey.isEmpty()) {
            LOG.warn("Not found trackerIssueKey for ticket close");
            return Mono.just(Result.success(returnData));
        }

        Mono<Result<Void>> result;
        TransferRequestStatus status = transferRequestModel.getStatus();
        if (status == TransferRequestStatus.APPLIED || status == TransferRequestStatus.PARTLY_APPLIED) {
            result = trackerClient.closeTicket(trackerIssueKey, "close",
                    new TrackerTransitionExecuteDto(TrackerIssueResolution.FIXED));
        } else if (status == TransferRequestStatus.CANCELLED || status == TransferRequestStatus.REJECTED
                || status == TransferRequestStatus.FAILED) {
            result = trackerClient.closeTicket(trackerIssueKey, "close",
                    new TrackerTransitionExecuteDto(TrackerIssueResolution.WONT_FIX));
        } else {
            return Mono.just(Result.success(returnData));
        }

        return result
                //log and ignore if Result.failure
                .map(res -> {
                    res.doOnFailure(error -> LOG.error("Failed to close ticket. Error: {}", error));
                    return Result.success(returnData);
                })
                //log and ignore if Mono.error
                .doOnError(throwable -> LOG.error("Failed to close ticket.", throwable))
                .onErrorReturn(Result.success(returnData));
    }

    private Tuple2<ExpandedTransferRequests<TransferRequestModel>, TransferRequestCreationResult>
    updateTrackerIssueKeyInResult(
            Tuple2<ExpandedTransferRequests<TransferRequestModel>, TransferRequestCreationResult> data,
            String trackerIssueKey) {
        TransferRequestModel transferRequest = TransferRequestModel.builder(data.getT1().getTransferRequests())
                .trackerIssueKey(trackerIssueKey)
                .build();
        data = data.mapT1(expandedTransferRequests -> ExpandedTransferRequests.builder(expandedTransferRequests)
                .transferRequests(transferRequest)
                .build());
        data = data.mapT2(creationResult -> TransferRequestCreationResult.builder(creationResult)
                .request(transferRequest)
                .build());
        return data;
    }

    private Mono<Result<Void>> updateTrackerIssueKeyInDb(TransferRequestModel transferRequest, String key) {
        return tableClient.usingSessionMonoRetryable(session ->
                transferRequestsDao.updateTrackerIssueKeyRetryable(
                        session.asTxCommitRetryable(TransactionMode.SERIALIZABLE_READ_WRITE),
                        transferRequest.getId(), transferRequest.getTenantId(), key));
    }

    private boolean isValidTransferRequestType(TransferRequestType type) {
        return type == TransferRequestType.QUOTA_TRANSFER ||
                type == TransferRequestType.RESERVE_TRANSFER ||
                type == TransferRequestType.PROVISION_TRANSFER;
    }

    private String createUnique(String uniquePart) {
        return trackerTransferQueue + "_" + uniquePart;
    }
}
