package ru.yandex.intranet.d.services.uniques

import kotlinx.coroutines.reactor.awaitSingle
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.accounts.AccountsDao
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao
import ru.yandex.intranet.d.dao.folders.FolderDao
import ru.yandex.intranet.d.dao.quotas.QuotasDao
import ru.yandex.intranet.d.dao.transfers.TransferRequestsDao
import ru.yandex.intranet.d.dao.unique.RequestUniqueDao
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader
import ru.yandex.intranet.d.model.accounts.AccountReserveType
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel
import ru.yandex.intranet.d.model.transfers.TransferRequestModel
import ru.yandex.intranet.d.model.uniques.RequestUniqueEndpoint
import ru.yandex.intranet.d.model.uniques.RequestUniqueIdentity
import ru.yandex.intranet.d.model.uniques.RequestUniqueMetadata
import ru.yandex.intranet.d.model.uniques.RequestUniqueModel
import ru.yandex.intranet.d.model.uniques.RequestUniqueSubject
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.time.Instant
import java.util.*

/**
 * Request unique service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class RequestUniqueService(private val requestUniqueDao: RequestUniqueDao,
                           private val transferRequestsDao: TransferRequestsDao,
                           private val accountsQuotasOperationsDao: AccountsQuotasOperationsDao,
                           private val accountsDao: AccountsDao,
                           private val accountsQuotasDao: AccountsQuotasDao,
                           private val resourcesLoader: ResourcesLoader,
                           private val unitsEnsemblesLoader: UnitsEnsemblesLoader,
                           private val quotasDao: QuotasDao,
                           private val folderDao: FolderDao,
                           @Qualifier("messageSource") private val messages: MessageSource
) {

    fun validateIdempotencyKey(idempotencyKey: String?, locale: Locale): Result<String> {
        if (idempotencyKey != null) {
            val errors = ErrorCollection.builder()
            validateIdempotencyKey(idempotencyKey, errors, locale)
            if (errors.hasAnyErrors()) {
                return Result.failure(errors.build())
            }
            return Result.success(idempotencyKey)
        }
        return Result.success("internal-" + UUID.randomUUID().toString())
    }

    suspend fun checkCreateTransferRequest(session: YdbTxSession, unique: String?,
                                           user: YaUserDetails): TransferRequestModel? {
        return checkTransferRequest(session, unique, user, RequestUniqueEndpoint.CREATE_TRANSFER_REQUEST)
    }

    suspend fun checkPutTransferRequest(session: YdbTxSession, unique: String?,
                                        user: YaUserDetails): TransferRequestModel? {
        return checkTransferRequest(session, unique, user, RequestUniqueEndpoint.PUT_TRANSFER_REQUEST)
    }

    suspend fun checkVoteForTransferRequest(session: YdbTxSession, unique: String?,
                                            user: YaUserDetails): TransferRequestModel? {
        return checkTransferRequest(session, unique, user, RequestUniqueEndpoint.VOTE_FOR_TRANSFER_REQUEST)
    }

    suspend fun checkCancelTransferRequest(session: YdbTxSession, unique: String?,
                                           user: YaUserDetails): TransferRequestModel? {
        return checkTransferRequest(session, unique, user, RequestUniqueEndpoint.CANCEL_TRANSFER_REQUEST)
    }

    suspend fun checkCreateAccount(session: YdbTxSession, unique: String?,
                                   user: YaUserDetails): CreateAccountOperation? {
        if (unique == null) {
            return null
        }
        val id = toIdentity(unique, user, RequestUniqueEndpoint.CREATE_ACCOUNT)
        val requestUnique = requestUniqueDao.getById(session, id) ?: return null
        val operationId = requestUnique.metadata.operationId ?: return null
        val operation = accountsQuotasOperationsDao.getById(session, operationId, id.tenantId)
            .awaitSingle().orElse(null) ?: return null
        if (operation.operationType != AccountsQuotasOperationsModel.OperationType.CREATE_ACCOUNT) {
            return null
        }
        val success = operation.requestStatus.map { v -> v == AccountsQuotasOperationsModel.RequestStatus.OK }
            .orElse(false)
        if (!success) {
            return CreateAccountOperation(null, operation)
        }
        val accountId = operation.requestedChanges.accountCreateParams
            .map {v -> v.accountId}.orElse(null) ?: return null
        val account = accountsDao.getById(session, accountId, id.tenantId)
            .awaitSingle().orElse(null) ?: return null
        return CreateAccountOperation(account, operation)
    }

    suspend fun checkPutAccount(session: YdbTxSession, unique: String?,
                                user: YaUserDetails): PutAccountOperation? {
        if (unique == null) {
            return null
        }
        val id = toIdentity(unique, user, RequestUniqueEndpoint.PUT_ACCOUNT)
        val requestUnique = requestUniqueDao.getById(session, id) ?: return null
        val operationId = requestUnique.metadata.operationId ?: return null
        val operation = accountsQuotasOperationsDao.getById(session, operationId, id.tenantId)
            .awaitSingle().orElse(null) ?: return null
        if (operation.operationType != AccountsQuotasOperationsModel.OperationType.PUT_ACCOUNT) {
            return null
        }
        val accountId = operation.requestedChanges.accountId.orElse(null) ?: return null
        val account = accountsDao.getById(session, accountId, id.tenantId)
            .awaitSingle().orElse(null) ?: return null
        return PutAccountOperation(account, operation)
    }

    suspend fun checkUpdateProvisions(session: YdbTxSession, unique: String?,
                                      user: YaUserDetails): UpdateProvisionOperation? {
        if (unique == null) {
            return null
        }
        val id = toIdentity(unique, user, RequestUniqueEndpoint.UPDATE_PROVISIONS)
        val requestUnique = requestUniqueDao.getById(session, id) ?: return null
        val operationId = requestUnique.metadata.operationId ?: return null
        val operation = accountsQuotasOperationsDao.getById(session, operationId, id.tenantId)
            .awaitSingle().orElse(null) ?: return null
        if (operation.operationType != AccountsQuotasOperationsModel.OperationType.UPDATE_PROVISION) {
            return null
        }
        val operationAccountId = operation.requestedChanges.accountId.orElseThrow()
        val operationAccount = accountsDao.getById(session, operationAccountId, id.tenantId).awaitSingle().orElseThrow()
        val folder = folderDao.getById(session, operationAccount.folderId, id.tenantId).awaitSingle().orElseThrow()
        val success = operation.requestStatus.map { v -> v == AccountsQuotasOperationsModel.RequestStatus.OK }
            .orElse(false)
        if (!success) {
            return UpdateProvisionOperation(null, operation, folder)
        }
        val operationResourceIds = operation.requestedChanges.updatedProvisions
            .map { v -> v.map { u -> u.resourceId } }.orElse(null) ?: return null
        val operationAccountProvisions = accountsQuotasDao.getAllByAccountIds(session, id.tenantId,
            setOf(operationAccountId)).awaitSingle()
        val folderProviderQuotas = quotasDao.getByFoldersAndProvider(session, listOf(operationAccount.folderId),
            id.tenantId, operationAccount.providerId, false).awaitSingle()
        val folderAccounts = accountsDao.getByFoldersForProvider(session, id.tenantId,
            operationAccount.providerId, operationAccount.folderId, false).awaitSingle()
        val folderAccountIds = folderAccounts.map { a -> a.id }.toSet()
        val folderAccountsProvisions = accountsQuotasDao.getAllByAccountIds(session, id.tenantId, folderAccountIds)
            .awaitSingle()
        val operationAccountProvisionsResourceIds = operationAccountProvisions.map { q -> q.resourceId }.toSet()
        val folderProviderQuotasResourceIds = folderProviderQuotas.map { q -> q.resourceId }.toSet()
        val folderAccountsProvisionsResourceIds = folderAccountsProvisions.map { p -> p.resourceId }.toSet()
        val resourceIds = operationAccountProvisionsResourceIds.union(operationResourceIds)
        val folderResourceIds = resourceIds.union(folderProviderQuotasResourceIds)
            .union(folderAccountsProvisionsResourceIds)
        val folderResources = folderResourceIds.chunked(500) { p -> p.map { i -> Tuples.of(i, id.tenantId) } }
            .map { p -> resourcesLoader.getResourcesByIds(session, p).awaitSingle() }.flatten()
        val resources = folderResources.filter { r -> resourceIds.contains(r.id) }
        val folderUnitsEnsembleIds = folderResources.map { r -> r.unitsEnsembleId }
        val unitsEnsembleIds = resources.map { r -> r.unitsEnsembleId }.toSet()
        val folderUnitsEnsembles = folderUnitsEnsembleIds
            .chunked(500) { p -> p.map { i -> Tuples.of(i, id.tenantId) } }
            .map { p -> unitsEnsemblesLoader.getUnitsEnsemblesByIds(session, p).awaitSingle() }.flatten()
        val unitsEnsembles = folderUnitsEnsembles.filter { u -> unitsEnsembleIds.contains(u.id) }
        return UpdateProvisionOperation(UpdateProvisionOperation.UpdatedProvisions(operationAccountProvisions,
            resources, unitsEnsembles, operationAccount, folderProviderQuotas, folderAccounts,
            folderAccountsProvisions, folderResources, folderUnitsEnsembles), operation, folder)
    }

    suspend fun checkProvideReserve(session: YdbTxSession, unique: String?,
                                    user: YaUserDetails, locale: Locale): Result<ProvideReserveOperation?> {
        if (unique == null) {
            return Result.success(null)
        }
        val id = toIdentity(unique, user, RequestUniqueEndpoint.PROVIDE_RESERVE)
        val requestUnique = requestUniqueDao.getById(session, id) ?: return Result.success(null)
        val operationId = requestUnique.metadata.operationId ?: return Result.success(null)
        val operation = accountsQuotasOperationsDao.getById(session, operationId, id.tenantId)
            .awaitSingle().orElse(null) ?: return Result.success(null)
        if (operation.operationType != AccountsQuotasOperationsModel.OperationType.PROVIDE_RESERVE) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.idempotency.key.mismatch", null, locale))).build())
        }
        val success = operation.requestStatus.map { v -> v == AccountsQuotasOperationsModel.RequestStatus.OK }
            .orElse(false)
        if (!success) {
            return Result.success(ProvideReserveOperation(null, operation))
        }
        val accountId = operation.requestedChanges.accountId.orElseThrow()
        val account = accountsDao.getById(session, accountId, id.tenantId).awaitSingle().orElseThrow()
        if (!Objects.equals(account.reserveType.orElse(null), AccountReserveType.PROVIDER) || account.isDeleted) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.provider.reserve.account.not.found", null, locale))).build())
        }
        val accountProvisions = accountsQuotasDao.getAllByAccountIds(session, id.tenantId,
            setOf(accountId)).awaitSingle()
        val accountResourceIds = accountProvisions.map { q -> q.resourceId }.toSet()
        val operationResourceIds = operation.requestedChanges.updatedProvisions
            .map { v -> v.map { u -> u.resourceId } }.orElse(emptyList()).toSet()
        val resourceIds = accountResourceIds + operationResourceIds
        val resources = resourceIds
            .chunked(1000) { p -> p.map { i -> Tuples.of(i, id.tenantId) } }
            .map { p -> resourcesLoader.getResourcesByIds(session, p).awaitSingle() }.flatten()
        val unitsEnsembleIds = resources.map { r -> r.unitsEnsembleId }.toSet()
        val unitsEnsembles = unitsEnsembleIds
            .chunked(500) { p -> p.map { i -> Tuples.of(i, id.tenantId) } }
            .map { p -> unitsEnsemblesLoader.getUnitsEnsemblesByIds(session, p).awaitSingle() }.flatten()
        return Result.success(ProvideReserveOperation(ProvideReserveOperation.ProvideReserveOperationResult(
            accountProvisions, resources, unitsEnsembles, account), operation))
    }

    suspend fun addCreateTransferRequest(session: YdbTxSession, unique: String?, transferRequest: TransferRequestModel,
                                         createdAt: Instant, user: YaUserDetails): Void? {
        if (unique == null) {
            return null
        }
        val model = toRequestUnique(unique, user, RequestUniqueEndpoint.CREATE_TRANSFER_REQUEST, createdAt,
            toMetadata(transferRequest))
        return requestUniqueDao.upsertRetryable(session, model)
    }

    fun addCreateTransferRequestMono(session: YdbTxSession, unique: String?, transferRequest: TransferRequestModel,
                                     createdAt: Instant, user: YaUserDetails): Mono<Void> {
        return mono { addCreateTransferRequest(session, unique, transferRequest, createdAt, user) }
    }

    suspend fun addPutTransferRequest(session: YdbTxSession, unique: String?, transferRequest: TransferRequestModel,
                                      createdAt: Instant, user: YaUserDetails): Void? {
        if (unique == null) {
            return null
        }
        val model = toRequestUnique(unique, user, RequestUniqueEndpoint.PUT_TRANSFER_REQUEST, createdAt,
            toMetadata(transferRequest))
        return requestUniqueDao.upsertRetryable(session, model)
    }

    fun addPutTransferRequestMono(session: YdbTxSession, unique: String?, transferRequest: TransferRequestModel,
                                  createdAt: Instant, user: YaUserDetails): Mono<Void> {
        return mono { addPutTransferRequest(session, unique, transferRequest, createdAt, user) }
    }

    suspend fun addVoteForTransferRequest(session: YdbTxSession, unique: String?, transferRequest: TransferRequestModel,
                                          createdAt: Instant, user: YaUserDetails): Void? {
        if (unique == null) {
            return null
        }
        val model = toRequestUnique(unique, user, RequestUniqueEndpoint.VOTE_FOR_TRANSFER_REQUEST, createdAt,
            toMetadata(transferRequest))
        return requestUniqueDao.upsertRetryable(session, model)
    }

    fun addVoteForTransferRequestMono(session: YdbTxSession, unique: String?, transferRequest: TransferRequestModel,
                                      createdAt: Instant, user: YaUserDetails): Mono<Void> {
        return mono { addVoteForTransferRequest(session, unique, transferRequest, createdAt, user) }
    }

    suspend fun addCancelTransferRequest(session: YdbTxSession, unique: String?, transferRequest: TransferRequestModel,
                                         createdAt: Instant, user: YaUserDetails): Void? {
        if (unique == null) {
            return null
        }
        val model = toRequestUnique(unique, user, RequestUniqueEndpoint.CANCEL_TRANSFER_REQUEST, createdAt,
            toMetadata(transferRequest))
        return requestUniqueDao.upsertRetryable(session, model)
    }

    fun addCancelTransferRequestMono(session: YdbTxSession, unique: String?, transferRequest: TransferRequestModel,
                                     createdAt: Instant, user: YaUserDetails): Mono<Void> {
        return mono { addCancelTransferRequest(session, unique, transferRequest, createdAt, user) }
    }

    suspend fun addCreateAccount(session: YdbTxSession, unique: String?, operation: AccountsQuotasOperationsModel,
                                 createdAt: Instant, user: YaUserDetails): Void? {
        if (unique == null) {
            return null
        }
        val model = toRequestUnique(unique, user, RequestUniqueEndpoint.CREATE_ACCOUNT, createdAt,
            toMetadata(operation))
        return requestUniqueDao.upsertRetryable(session, model)
    }

    suspend fun addPutAccount(session: YdbTxSession, unique: String?, operation: AccountsQuotasOperationsModel,
                              createdAt: Instant, user: YaUserDetails): Void? {
        if (unique == null) {
            return null
        }
        val model = toRequestUnique(unique, user, RequestUniqueEndpoint.PUT_ACCOUNT, createdAt,
            toMetadata(operation))
        return requestUniqueDao.upsertRetryable(session, model)
    }

    suspend fun addUpdateProvisions(session: YdbTxSession, unique: String?, operation: AccountsQuotasOperationsModel,
                                    createdAt: Instant, user: YaUserDetails): Void? {
        if (unique == null) {
            return null
        }
        val model = toRequestUnique(unique, user, RequestUniqueEndpoint.UPDATE_PROVISIONS, createdAt,
            toMetadata(operation))
        return requestUniqueDao.upsertRetryable(session, model)
    }

    suspend fun addProviderReserve(session: YdbTxSession, unique: String?, operation: AccountsQuotasOperationsModel,
                                   createdAt: Instant, user: YaUserDetails) {
        if (unique == null) {
            return
        }
        val model = toRequestUnique(unique, user, RequestUniqueEndpoint.PROVIDE_RESERVE, createdAt,
            toMetadata(operation))
        requestUniqueDao.upsertRetryable(session, model)
    }

    private fun validateIdempotencyKey(idempotencyKey: String?, errors: ErrorCollection.Builder, locale: Locale) {
        val idempotencyKeyLengthLimit = 1024
        if (idempotencyKey != null && idempotencyKey.isBlank()) {
            errors.addError("idempotencyKey", TypedError.invalid(messages
                .getMessage("errors.blank.idempotency.key", null, locale)))
        } else if (idempotencyKey != null && idempotencyKey.length > idempotencyKeyLengthLimit) {
            errors.addError("idempotencyKey", TypedError.invalid(messages
                .getMessage("errors.idempotency.key.is.too.long", null, locale)))
        }
    }

    private suspend fun checkTransferRequest(session: YdbTxSession,
                                             unique: String?,
                                             user: YaUserDetails,
                                             endpoint: RequestUniqueEndpoint): TransferRequestModel? {
        if (unique == null) {
            return null
        }
        val id = toIdentity(unique, user, endpoint)
        val requestUnique = requestUniqueDao.getById(session, id) ?: return null
        val transferId = requestUnique.metadata.transferId ?: return null
        return transferRequestsDao.getById(session, transferId, id.tenantId).awaitSingle().orElse(null)
    }

    private fun toIdentity(unique: String, user: YaUserDetails,
                           endpoint: RequestUniqueEndpoint): RequestUniqueIdentity {
        return RequestUniqueIdentity(Tenants.getTenantId(user), unique, toSubject(user), endpoint)
    }

    private fun toSubject(user: YaUserDetails): RequestUniqueSubject {
        if (user.uid.isPresent) {
            return RequestUniqueSubject.user(user.uid.get())
        }
        if (user.tvmServiceId.isPresent) {
            return RequestUniqueSubject.tvmId(user.tvmServiceId.get().toString())
        }
        return RequestUniqueSubject.empty()
    }

    private fun toMetadata(transferRequest: TransferRequestModel): RequestUniqueMetadata {
        return RequestUniqueMetadata(null, transferRequest.id)
    }

    private fun toMetadata(operation: AccountsQuotasOperationsModel): RequestUniqueMetadata {
        return RequestUniqueMetadata(operation.operationId, null)
    }

    private fun toRequestUnique(unique: String, user: YaUserDetails, endpoint: RequestUniqueEndpoint,
                                createdAt: Instant, metadata: RequestUniqueMetadata): RequestUniqueModel {
        return RequestUniqueModel(Tenants.getTenantId(user), unique, toSubject(user), endpoint, createdAt, metadata)
    }

}
