package ru.yandex.intranet.d.services.units;

import java.math.BigDecimal;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.google.common.collect.Streams;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuple3;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.dao.units.UnitsEnsemblesDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader;
import ru.yandex.intranet.d.model.units.GrammaticalCase;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.services.validators.VersionValidator;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.paging.ContinuationTokens;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.units.GrammaticalCaseDto;
import ru.yandex.intranet.d.web.model.units.UnitCreateDto;
import ru.yandex.intranet.d.web.model.units.UnitPutDto;
import ru.yandex.intranet.d.web.model.units.UnitsEnsembleCreateDto;
import ru.yandex.intranet.d.web.model.units.UnitsEnsemblePatchDto;
import ru.yandex.intranet.d.web.model.units.UnitsEnsemblePutDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Units ensemble service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class UnitsEnsemblesService {

    private static final int MAX_NAME_LENGTH = 256;
    private static final int MAX_DESCRIPTION_LENGTH = 1024;
    private static final int MAX_KEY_LENGTH = 256;
    private static final int MAX_SHORT_NAME_LENGTH = 256;
    private static final int MAX_LONG_NAME_LENGTH = 1024;
    public static final long MAX_POWER = 999999999L;
    public static final long MIN_POWER = -MAX_POWER;
    private static final MultiplierComparator MULTIPLIER_COMPARATOR = new MultiplierComparator();

    private final UnitsEnsemblesDao unitsEnsemblesDao;
    private final ResourcesDao resourcesDao;
    private final YdbTableClient tableClient;
    private final UnitsEnsemblesLoader unitsEnsemblesLoader;
    private final MessageSource messages;
    private final SecurityManagerService securityManagerService;
    private final ObjectReader continuationTokenReader;
    private final ObjectWriter continuationTokenWriter;
    private final VersionValidator versionValidator;

    @SuppressWarnings("ParameterNumber")
    public UnitsEnsemblesService(UnitsEnsemblesDao unitsEnsemblesDao,
                                 ResourcesDao resourcesDao,
                                 YdbTableClient tableClient,
                                 UnitsEnsemblesLoader unitsEnsemblesLoader,
                                 @Qualifier("messageSource") MessageSource messages,
                                 SecurityManagerService securityManagerService,
                                 @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper,
                                 VersionValidator versionValidator
    ) {
        this.unitsEnsemblesDao = unitsEnsemblesDao;
        this.resourcesDao = resourcesDao;
        this.tableClient = tableClient;
        this.unitsEnsemblesLoader = unitsEnsemblesLoader;
        this.messages = messages;
        this.securityManagerService = securityManagerService;
        this.continuationTokenReader = objectMapper.getObjectMapper().readerFor(UnitsEnsembleContinuationToken.class);
        this.continuationTokenWriter = objectMapper.getObjectMapper().writerFor(UnitsEnsembleContinuationToken.class);
        this.versionValidator = versionValidator;
    }

    public Mono<Result<UnitsEnsembleModel>> getById(String id, YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res
            .andThen(v -> validateId(id, locale))
            .andThenMono(u -> tableClient.usingSessionMonoRetryable(session ->
                        unitsEnsemblesDao.getById(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                                id, Tenants.DEFAULT_TENANT_ID).flatMap(r -> {
                            if (r.isPresent() && !r.get().isDeleted()) {
                                return Mono.just(Result.success(skipDeletedUnits(r.get())));
                            }
                            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                                    .getMessage("errors.units.ensemble.not.found", null, locale)))
                                    .build();
                            return Mono.just(Result.failure(error));
                        })
                )
            ));
    }

    public Mono<Result<UnitModel>> getUnitById(String id, String unitId, YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res
                .andThen(v -> validateId(id, locale))
                .andThen(v -> validateUnitId(unitId, locale))
                .andThenMono(u -> tableClient.usingSessionMonoRetryable(session ->
                                unitsEnsemblesDao.getById(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                                        id, Tenants.DEFAULT_TENANT_ID).flatMap(r -> {
                                    if (r.isEmpty() || r.get().isDeleted()) {
                                        ErrorCollection error = ErrorCollection.builder()
                                                .addError(TypedError.notFound(messages
                                                        .getMessage("errors.units.ensemble.not.found", null, locale)))
                                                .build();
                                        return Mono.just(Result.failure(error));
                                    }
                                    Optional<UnitModel> unit = r.get().unitById(unitId);
                                    if (unit.isEmpty() || unit.get().isDeleted()) {
                                        ErrorCollection error = ErrorCollection.builder()
                                                .addError(TypedError.notFound(messages
                                                        .getMessage("errors.unit.not.found", null, locale)))
                                                .build();
                                        return Mono.just(Result.failure(error));
                                    }
                                    return Mono.just(Result.success(unit.get()));
                                })
                        )
                ));
    }

    public Mono<Result<Page<UnitsEnsembleModel>>> getPage(PageRequest pageRequest,
                                                          YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(r -> r.andThenMono(u -> {
            Result<PageRequest.Validated<UnitsEnsembleContinuationToken>> pageValidation
                    = pageRequest.validate(continuationTokenReader, messages, locale);
            return pageValidation.andThenDo(p -> validateContinuationToken(p, locale)).andThenMono(p -> {
                int limit = p.getLimit();
                String fromId = p.getContinuationToken().map(UnitsEnsembleContinuationToken::getId).orElse(null);
                return tableClient.usingSessionMonoRetryable(session ->
                        unitsEnsemblesDao
                                .getByTenant(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                                        Tenants.DEFAULT_TENANT_ID, fromId, limit + 1, false)
                                .map(values -> values.size() > limit
                                        ? Page.page(skipDeletedUnits(values.subList(0, limit)),
                                        prepareToken(values.get(limit - 1)))
                                        : Page.lastPage(skipDeletedUnits(values)))
                                .map(Result::success));
            });
        }));
    }

    public Mono<Result<UnitsEnsembleModel>> create(UnitsEnsembleCreateDto unitsEnsemble,
                                                   YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkDictionaryWritePermissions(currentUser, locale).andThenMono(u -> {
            String newId = UUID.randomUUID().toString();
            return tableClient.usingSessionMonoRetryable(session ->
                    session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts ->
                            validateCreation(unitsEnsemble, newId, locale, ts)
                                    .flatMap(r -> r.andThenMono(validated ->
                                                    unitsEnsemblesDao.upsertUnitsEnsembleRetryable(ts, validated)
                                                            .doOnSuccess(v -> unitsEnsemblesLoader.update(validated))
                                                            .thenReturn(Result.success(validated))
                                            )
                                    )
                    )
            );
        });
    }

    public Mono<Result<UnitsEnsembleModel>> put(String id, Long version, UnitsEnsemblePutDto unitsEnsemble,
                                                YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkDictionaryWritePermissions(currentUser, locale).andThen(
                v -> validateId(id, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> unitsEnsemblesDao.getByIdStartTx(ts, id, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> {
                                    if (r.isEmpty() || r.get().isDeleted()) {
                                        ErrorCollection error = ErrorCollection.builder().addError(TypedError
                                                .notFound(messages.getMessage("errors.units.ensemble.not.found",
                                                        null, locale))).build();
                                        return Mono.just(Result.<Tuple2<UnitsEnsembleModel, Boolean>>failure(error));
                                    }
                                    return Mono.just(validatePut(unitsEnsemble, version, r.get(), locale));
                                },
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return unitsEnsemblesDao.updateUnitsEnsembleRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> unitsEnsemblesLoader.update(m.getT1()))
                                                        .thenReturn(Result.success(skipDeletedUnits(m.getT1())));
                                            } else {
                                                return ts.commitTransaction()
                                                        .thenReturn(Result.success(skipDeletedUnits(m.getT1())));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    public Mono<Result<UnitsEnsembleModel>> patch(String id, Long version, UnitsEnsemblePatchDto unitsEnsemble,
                                                  YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkDictionaryWritePermissions(currentUser, locale).andThen(
                v -> validateId(id, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> unitsEnsemblesDao.getByIdStartTx(ts, id, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> {
                                    if (r.isEmpty() || r.get().isDeleted()) {
                                        ErrorCollection error = ErrorCollection.builder().addError(TypedError
                                                .notFound(messages.getMessage("errors.units.ensemble.not.found",
                                                        null, locale))).build();
                                        return Mono.just(Result.<Tuple2<UnitsEnsembleModel, Boolean>>failure(error));
                                    }
                                    return Mono.just(validatePatch(unitsEnsemble, version, r.get(), locale));
                                },
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return unitsEnsemblesDao.updateUnitsEnsembleRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> unitsEnsemblesLoader.update(m.getT1()))
                                                        .thenReturn(Result.success(skipDeletedUnits(m.getT1())));
                                            } else {
                                                return ts.commitTransaction()
                                                        .thenReturn(Result.success(skipDeletedUnits(m.getT1())));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                )
        );
    }

    public Mono<Result<Void>> delete(String id, Long version, YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkDictionaryWritePermissions(currentUser, locale).andThen(v ->
                validateId(id, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> unitsEnsemblesDao.getByIdStartTx(ts, id, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> {
                                    if (r.isEmpty() || r.get().isDeleted()) {
                                        ErrorCollection error = ErrorCollection.builder().addError(TypedError
                                                .notFound(messages.getMessage("errors.units.ensemble.not.found",
                                                        null, locale))).build();
                                        return Mono.just(Result.failure(error));
                                    }
                                    return validateDelete(ts, version, r.get(), locale);
                                },
                                (ts, r) -> r.match(m -> unitsEnsemblesDao.updateUnitsEnsembleRetryable(ts, m)
                                                .doOnSuccess(v -> unitsEnsemblesLoader.update(m))
                                                .thenReturn(r.toVoid()),
                                        e -> ts.commitTransaction().thenReturn(r.toVoid()))
                        )
                )
        );
    }

    public Mono<Result<UnitsEnsembleModel>> createUnit(String ensembleId, Long version, UnitCreateDto unit,
                                                       YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkDictionaryWritePermissions(currentUser, locale).andThen(v ->
                validateId(ensembleId, locale)).andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> unitsEnsemblesDao.getByIdStartTx(ts, ensembleId, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> {
                                    if (r.isEmpty() || r.get().isDeleted()) {
                                        ErrorCollection error = ErrorCollection.builder().addError(TypedError
                                                .notFound(messages.getMessage("errors.units.ensemble.not.found",
                                                        null, locale))).build();
                                        return Mono.just(Result.<UnitsEnsembleModel>failure(error));
                                    }
                                    return Mono.just(validateCreateUnit(unit, version, r.get(), locale));
                                },
                                (ts, r) -> r.match(m -> unitsEnsemblesDao.updateUnitsEnsembleRetryable(ts, m)
                                                .doOnSuccess(v -> unitsEnsemblesLoader.update(m))
                                                .thenReturn(r.apply(this::skipDeletedUnits)),
                                        e -> ts.commitTransaction().thenReturn(r.apply(this::skipDeletedUnits)))
                        )
                )
        );
    }

    public Mono<Result<UnitsEnsembleModel>> putUnit(String ensembleId, long version, String unitId, UnitPutDto unit,
                                                    YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkDictionaryWritePermissions(currentUser, locale)
                .andThen(v -> validateId(ensembleId, locale))
                .andThen(v -> validateUnitId(unitId, locale)).andThenMono(u ->
                    tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> unitsEnsemblesDao.getByIdStartTx(ts, ensembleId, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> {
                                    if (r.isEmpty() || r.get().isDeleted()) {
                                        ErrorCollection error = ErrorCollection.builder().addError(TypedError
                                                .notFound(messages.getMessage("errors.units.ensemble.not.found",
                                                        null, locale))).build();
                                        return Mono.just(Result.<Tuple2<UnitsEnsembleModel, Boolean>>failure(error));
                                    }
                                    Optional<UnitModel> existingUnit = r.get().unitById(unitId);
                                    if (existingUnit.isEmpty() || existingUnit.get().isDeleted()) {
                                        ErrorCollection error = ErrorCollection.builder().addError(TypedError
                                                .notFound(messages.getMessage("errors.unit.not.found",
                                                        null, locale))).build();
                                        return Mono.just(Result.<Tuple2<UnitsEnsembleModel, Boolean>>failure(error));
                                    }
                                    return Mono.just(validatePutUnit(unit, version, r.get(),
                                            existingUnit.get(), locale));
                                },
                                (ts, r) -> r.match(
                                        m -> {
                                            if (m.getT2()) {
                                                return unitsEnsemblesDao.updateUnitsEnsembleRetryable(ts, m.getT1())
                                                        .doOnSuccess(v -> unitsEnsemblesLoader.update(m.getT1()))
                                                        .thenReturn(Result.success(skipDeletedUnits(m.getT1())));
                                            } else {
                                                return ts.commitTransaction()
                                                        .thenReturn(Result.success(skipDeletedUnits(m.getT1())));
                                            }
                                        },
                                        e -> ts.commitTransaction().thenReturn(Result.failure(e))
                                )
                        )
                    )
                );
    }

    public Mono<Result<UnitsEnsembleModel>> deleteUnit(String ensembleId, long version, String unitId,
                                                       YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkDictionaryWritePermissions(currentUser, locale)
                .andThen(v -> validateId(ensembleId, locale))
                .andThen(v -> validateUnitId(unitId, locale)).andThenMono(u ->
                    tableClient.usingSessionMonoRetryable(session ->
                        session.usingCompTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                                ts -> unitsEnsemblesDao.getByIdStartTx(ts, ensembleId, Tenants.DEFAULT_TENANT_ID),
                                (ts, r) -> {
                                    if (r.isEmpty() || r.get().isDeleted()) {
                                        ErrorCollection error = ErrorCollection.builder().addError(TypedError
                                                .notFound(messages.getMessage("errors.units.ensemble.not.found",
                                                        null, locale))).build();
                                        return Mono.just(Result.<UnitsEnsembleModel>failure(error));
                                    }
                                    Optional<UnitModel> existingUnit = r.get().unitById(unitId);
                                    if (existingUnit.isEmpty() || existingUnit.get().isDeleted()) {
                                        ErrorCollection error = ErrorCollection.builder().addError(TypedError
                                                .notFound(messages.getMessage("errors.unit.not.found",
                                                        null, locale))).build();
                                        return Mono.just(Result.<UnitsEnsembleModel>failure(error));
                                    }
                                    return Mono.just(validateDeleteUnit(version, r.get(),
                                            existingUnit.get(), locale));
                                },
                                (ts, r) -> r.match(m -> unitsEnsemblesDao.updateUnitsEnsembleRetryable(ts, m)
                                                .doOnSuccess(v -> unitsEnsemblesLoader.update(m))
                                                .thenReturn(r.apply(this::skipDeletedUnits)),
                                        e -> ts.commitTransaction().thenReturn(r.apply(this::skipDeletedUnits)))
                        )
                    )
                );
    }

    private String prepareToken(UnitsEnsembleModel lastItem) {
        return ContinuationTokens.encode(new UnitsEnsembleContinuationToken(lastItem.getId()),
                continuationTokenWriter);
    }

    private Result<Void> validateId(String resourceId, Locale locale) {
        if (!Uuids.isValidUuid(resourceId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.units.ensemble.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<Void> validateUnitId(String resourceId, Locale locale) {
        if (!Uuids.isValidUuid(resourceId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.unit.not.found", null, locale)))
                    .build();
            return Result.failure(error);
        }
        return Result.success(null);
    }

    private Result<Void> validateContinuationToken(PageRequest.Validated<UnitsEnsembleContinuationToken> pageRequest,
                                                   Locale locale) {
        if (pageRequest.getContinuationToken().isEmpty()) {
            return Result.success(null);
        }
        return validateId(pageRequest.getContinuationToken().get().getId(), locale);
    }

    private List<UnitsEnsembleModel> skipDeletedUnits(List<UnitsEnsembleModel> ensembles) {
        return ensembles.stream().map(this::skipDeletedUnits).collect(Collectors.toList());
    }

    private UnitsEnsembleModel skipDeletedUnits(UnitsEnsembleModel ensemble) {
        return new UnitsEnsembleModel(ensemble.getId(), ensemble.getTenantId(), ensemble.getVersion(),
                ensemble.getNameEn(), ensemble.getNameRu(), ensemble.getDescriptionEn(), ensemble.getDescriptionRu(),
                ensemble.isFractionsAllowed(), ensemble.isDeleted(),
                ensemble.getUnits().stream().filter(u -> !u.isDeleted()).collect(Collectors.toSet()),
                ensemble.getKey());
    }

    private Mono<Result<UnitsEnsembleModel>> validateCreation(UnitsEnsembleCreateDto input, String newId,
                                                              Locale locale, YdbTxSession session) {
        return validateKey(session, input::getKey, "key", locale).map(keyR -> {
            UnitsEnsembleModel.Builder builder = UnitsEnsembleModel.builder();
            ErrorCollection.Builder errors = ErrorCollection.builder();
            builder.id(newId);
            builder.tenantId(Tenants.DEFAULT_TENANT_ID);
            builder.version(0L);
            builder.deleted(false);
            keyR.doOnFailure(errors::add);
            keyR.doOnSuccess(builder::key);
            validateText(input::getNameEn, builder::nameEn, errors, "nameEn", MAX_NAME_LENGTH, locale);
            validateText(input::getNameRu, builder::nameRu, errors, "nameRu", MAX_NAME_LENGTH, locale);
            validateText(input::getDescriptionEn, builder::descriptionEn, errors, "descriptionEn",
                    MAX_DESCRIPTION_LENGTH, locale);
            validateText(input::getDescriptionRu, builder::descriptionRu, errors, "descriptionRu",
                    MAX_DESCRIPTION_LENGTH, locale);
            validateRequired(input::getFractionsAllowed, builder::fractionsAllowed, errors, "fractionsAllowed", locale);
            if (input.getUnits().isEmpty()) {
                errors.addError("units", TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
            } else {
                validateUnits(input.getUnits().get(), builder, errors, locale);
            }
            if (errors.hasAnyErrors()) {
                return Result.failure(errors.build());
            }
            return Result.success(builder.build());
        });
    }

    private void validateUnits(List<UnitCreateDto> units, UnitsEnsembleModel.Builder builder,
                               ErrorCollection.Builder errors, Locale locale) {
        if (units.isEmpty()) {
            errors.addError("units", TypedError.invalid(messages
                    .getMessage("errors.non.empty.list.is.required", null, locale)));
            return;
        }
        List<String> keys = units.stream().filter(Objects::nonNull).map(UnitCreateDto::getKey)
                .flatMap(Optional::stream).collect(Collectors.toList());
        if (new HashSet<>(keys).size() != keys.size()) {
            errors.addError("units", TypedError.invalid(messages
                    .getMessage("errors.duplicate.keys.are.forbidden", null, locale)));
        }
        for (int i = 0; i < units.size(); i++) {
            UnitModel.Builder unitBuilder = UnitModel.builder();
            ErrorCollection.Builder unitErrors = ErrorCollection.builder();
            UnitCreateDto unit = units.get(i);
            if (unit == null) {
                errors.addError("units." + i, TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                continue;
            }
            validateText(unit::getKey, unitBuilder::key, unitErrors, "units." + i + ".key", MAX_KEY_LENGTH, locale);
            validateText(unit::getShortNameSingularEn, unitBuilder::shortNameSingularEn, unitErrors,
                    "units." + i + ".shortNameSingularEn", MAX_SHORT_NAME_LENGTH, locale);
            validateText(unit::getShortNamePluralEn, unitBuilder::shortNamePluralEn, unitErrors,
                    "units." + i + ".shortNamePluralEn", MAX_SHORT_NAME_LENGTH, locale);
            validateText(unit::getLongNameSingularEn, unitBuilder::longNameSingularEn, unitErrors,
                    "units." + i + ".longNameSingularEn", MAX_LONG_NAME_LENGTH, locale);
            validateText(unit::getLongNamePluralEn, unitBuilder::longNamePluralEn, unitErrors,
                    "units." + i + ".longNamePluralEn", MAX_LONG_NAME_LENGTH, locale);
            validateCases(unit::getShortNameSingularRu, unitBuilder::putShortNameSingularRu, unitErrors,
                    "units." + i + ".shortNameSingularRu", MAX_SHORT_NAME_LENGTH, locale);
            validateCases(unit::getShortNamePluralRu, unitBuilder::putShortNamePluralRu, unitErrors,
                    "units." + i + ".shortNamePluralRu", MAX_SHORT_NAME_LENGTH, locale);
            validateCases(unit::getLongNameSingularRu, unitBuilder::putLongNameSingularRu, unitErrors,
                    "units." + i + ".longNameSingularRu", MAX_LONG_NAME_LENGTH, locale);
            validateCases(unit::getLongNamePluralRu, unitBuilder::putLongNamePluralRu, unitErrors,
                    "units." + i + ".longNamePluralRu", MAX_LONG_NAME_LENGTH, locale);
            validatePositiveNumber(unit::getBase, unitBuilder::base, unitErrors, "units." + i + ".base", locale);
            validateNumberRange(unit::getPower, unitBuilder::power, MIN_POWER, MAX_POWER,
                    unitErrors, "units." + i + ".power", locale);
            if (unitErrors.hasAnyErrors()) {
                errors.add(unitErrors);
            } else {
                unitBuilder.id(UUID.randomUUID().toString());
                unitBuilder.deleted(false);
                builder.addUnit(unitBuilder.build());
            }
        }
    }

    private void validateCases(Supplier<Optional<Map<GrammaticalCaseDto, String>>> getter,
                               BiConsumer<GrammaticalCase, String> setter, ErrorCollection.Builder errors,
                               String fieldKey, int maxLength, Locale locale) {
        Optional<Map<GrammaticalCaseDto, String>> cases = getter.get();
        if (cases.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        if (cases.get().containsKey(GrammaticalCaseDto.UNKNOWN)) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.unknown.grammatical.case", null, locale)));
        }
        long suppliedCases = cases.get().keySet().stream().filter(v -> v != GrammaticalCaseDto.UNKNOWN).count();
        long knownCases = EnumSet.allOf(GrammaticalCaseDto.class).stream()
                .filter(v -> v != GrammaticalCaseDto.UNKNOWN).count();
        if (suppliedCases != knownCases) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.missing.grammatical.cases", null, locale)));
        }
        cases.get().forEach((key, value) -> {
            Optional<GrammaticalCase> grammaticalCase = toGrammaticalCase(key);
            if (grammaticalCase.isPresent()) {
                validateText(() -> Optional.ofNullable(value), text -> setter.accept(grammaticalCase.get(), text),
                        errors, fieldKey + "." + key.name(), maxLength, locale);
            } else {
                validateText(() -> Optional.ofNullable(value), text -> { },
                        errors, fieldKey + "." + key.name(), maxLength, locale);
            }
        });
    }

    private void validateText(Supplier<Optional<String>> getter, Consumer<String> setter,
                              ErrorCollection.Builder errors, String fieldKey, int maxLength, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (text.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    private void validatePositiveNumber(Supplier<Optional<Long>> getter, Consumer<Long> setter,
                                        ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<Long> value = getter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (value.get() <= 0) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.number.must.be.positive", null, locale)));
        } else {
            setter.accept(value.get());
        }
    }

    private void validateNumberRange(Supplier<Optional<Long>> getter, Consumer<Long> setter, long min, long max,
                                     ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<Long> value = getter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (value.get() < min || value.get() > max) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.number.out.of.range", null, locale)));
        } else {
            setter.accept(value.get());
        }
    }

    private <T> void validateRequired(Supplier<Optional<T>> getter, Consumer<T> setter,
                                      ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<T> value = getter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            setter.accept(value.get());
        }
    }

    private Optional<GrammaticalCase> toGrammaticalCase(GrammaticalCaseDto value) {
        switch (value) {
            case NOMINATIVE:
                return Optional.of(GrammaticalCase.NOMINATIVE);
            case GENITIVE:
                return Optional.of(GrammaticalCase.GENITIVE);
            case DATIVE:
                return Optional.of(GrammaticalCase.DATIVE);
            case ACCUSATIVE:
                return Optional.of(GrammaticalCase.ACCUSATIVE);
            case INSTRUMENTAL:
                return Optional.of(GrammaticalCase.INSTRUMENTAL);
            case PREPOSITIONAL:
                return Optional.of(GrammaticalCase.PREPOSITIONAL);
            case UNKNOWN:
                return Optional.empty();
            default:
                throw new IllegalArgumentException("Unexpected grammatical case DTO: " + value);
        }
    }

    private Result<Tuple2<UnitsEnsembleModel, Boolean>> validatePut(UnitsEnsemblePutDto input, Long version,
                                                                    UnitsEnsembleModel existing, Locale locale) {
        UnitsEnsembleModel.Builder builder = UnitsEnsembleModel.builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        builder.id(existing.getId());
        builder.tenantId(existing.getTenantId());
        builder.deleted(existing.isDeleted());
        builder.fractionsAllowed(existing.isFractionsAllowed());
        builder.key(existing.getKey());
        existing.getUnits().forEach(builder::addUnit);
        versionValidator.validate(() -> Optional.ofNullable(version), existing::getVersion, builder::version, errors,
                "version", locale);
        validateText(input::getNameEn, builder::nameEn, errors, "nameEn", MAX_NAME_LENGTH, locale);
        validateText(input::getNameRu, builder::nameRu, errors, "nameRu", MAX_NAME_LENGTH, locale);
        validateText(input::getDescriptionEn, builder::descriptionEn, errors, "descriptionEn",
                MAX_DESCRIPTION_LENGTH, locale);
        validateText(input::getDescriptionRu, builder::descriptionRu, errors, "descriptionRu",
                MAX_DESCRIPTION_LENGTH, locale);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }

        if (builder.hasChanges(existing)) {
            return Result.success(Tuples.of(builder.build(), true));
        }
        return Result.success(Tuples.of(existing, false));
    }

    private Result<Tuple2<UnitsEnsembleModel, Boolean>> validatePatch(UnitsEnsemblePatchDto unitsEnsemble, Long version,
                                                                      UnitsEnsembleModel existingUnitsEnsemble,
                                                                      Locale locale) {
        UnitsEnsembleModel.Builder builder = UnitsEnsembleModel.builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        builder.id(existingUnitsEnsemble.getId());
        builder.tenantId(existingUnitsEnsemble.getTenantId());
        builder.deleted(existingUnitsEnsemble.isDeleted());
        builder.fractionsAllowed(existingUnitsEnsemble.isFractionsAllowed());
        builder.key(existingUnitsEnsemble.getKey());
        existingUnitsEnsemble.getUnits().forEach(builder::addUnit);
        versionValidator.validate(() -> Optional.ofNullable(version), existingUnitsEnsemble::getVersion,
                builder::version, errors, "version", locale);

        unitsEnsemble.getNameEn().acceptNewValueOptionalToSetter(s -> validateText(() -> s, builder::nameEn, errors,
                "nameEn", MAX_NAME_LENGTH, locale), existingUnitsEnsemble.getNameEn());
        unitsEnsemble.getNameRu().acceptNewValueOptionalToSetter(s -> validateText(() -> s, builder::nameRu, errors,
                "nameRu", MAX_NAME_LENGTH, locale), existingUnitsEnsemble.getNameRu());
        unitsEnsemble.getDescriptionEn().acceptNewValueOptionalToSetter(s -> validateText(() -> s,
                builder::descriptionEn, errors, "descriptionEn", MAX_DESCRIPTION_LENGTH, locale),
                existingUnitsEnsemble.getDescriptionEn());
        unitsEnsemble.getDescriptionRu().acceptNewValueOptionalToSetter(s -> validateText(() -> s,
                builder::descriptionRu, errors, "descriptionRu", MAX_DESCRIPTION_LENGTH, locale),
                existingUnitsEnsemble.getDescriptionRu());
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }

        if (builder.hasChanges(existingUnitsEnsemble)) {
            return Result.success(Tuples.of(builder.build(), true));
        }
        return Result.success(Tuples.of(existingUnitsEnsemble, false));
    }

    private Mono<Result<UnitsEnsembleModel>> validateDelete(YdbTxSession session, Long version,
                                                            UnitsEnsembleModel existing, Locale locale) {
        UnitsEnsembleModel.Builder builder = UnitsEnsembleModel.builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        builder.id(existing.getId());
        builder.tenantId(existing.getTenantId());
        builder.deleted(true);
        builder.fractionsAllowed(existing.isFractionsAllowed());
        builder.key(existing.getKey());
        builder.nameEn(existing.getNameEn());
        builder.nameRu(existing.getNameRu());
        builder.descriptionEn(existing.getDescriptionEn());
        builder.descriptionRu(existing.getDescriptionRu());
        existing.getUnits().forEach(builder::addUnit);
        versionValidator.validate(() -> Optional.ofNullable(version), existing::getVersion, builder::version, errors,
                "version", locale);
        return resourcesDao.existsByUnitsEnsembleId(session, existing.getId(), Tenants.DEFAULT_TENANT_ID, false)
                .map(exists -> {
                    if (exists) {
                        errors.addError(TypedError.invalid(messages
                                .getMessage("errors.ensemble.is.still.used", null, locale)));
                    }
                    if (errors.hasAnyErrors()) {
                        return Result.failure(errors.build());
                    }
                    return Result.success(builder.build());
                });
    }

    private Result<UnitsEnsembleModel> validateCreateUnit(UnitCreateDto input, Long version,
                                                          UnitsEnsembleModel existing, Locale locale) {
        UnitsEnsembleModel.Builder builder = UnitsEnsembleModel.builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        builder.id(existing.getId());
        builder.tenantId(existing.getTenantId());
        builder.deleted(existing.isDeleted());
        builder.fractionsAllowed(existing.isFractionsAllowed());
        builder.key(existing.getKey());
        builder.nameEn(existing.getNameEn());
        builder.nameRu(existing.getNameRu());
        builder.descriptionEn(existing.getDescriptionEn());
        builder.descriptionRu(existing.getDescriptionRu());
        existing.getUnits().forEach(builder::addUnit);
        versionValidator.validate(() -> Optional.ofNullable(version), existing::getVersion, builder::version, errors,
                "version", locale);
        validateAddUnit(input, existing.getUnits().stream().filter(u -> !u.isDeleted()).collect(Collectors.toSet()),
                builder, errors, locale);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(builder.build());
    }

    private void validateAddUnit(UnitCreateDto unit, Set<UnitModel> existingUnits,
                                 UnitsEnsembleModel.Builder builder, ErrorCollection.Builder errors, Locale locale) {
        UnitModel.Builder unitBuilder = UnitModel.builder();
        ErrorCollection.Builder unitErrors = ErrorCollection.builder();
        String newId = UUID.randomUUID().toString();
        unitBuilder.id(newId);
        unitBuilder.deleted(false);
        validateKeyUniqueness(unit, existingUnits, unitErrors, locale);
        validateUnitOrdering(unit::getBase, unit::getPower, newId, existingUnits, unitErrors, locale);
        validateText(unit::getKey, unitBuilder::key, unitErrors, "key", MAX_KEY_LENGTH, locale);
        validateText(unit::getShortNameSingularEn, unitBuilder::shortNameSingularEn, unitErrors,
                "shortNameSingularEn", MAX_SHORT_NAME_LENGTH, locale);
        validateText(unit::getShortNamePluralEn, unitBuilder::shortNamePluralEn, unitErrors,
                "shortNamePluralEn", MAX_SHORT_NAME_LENGTH, locale);
        validateText(unit::getLongNameSingularEn, unitBuilder::longNameSingularEn, unitErrors,
                "longNameSingularEn", MAX_LONG_NAME_LENGTH, locale);
        validateText(unit::getLongNamePluralEn, unitBuilder::longNamePluralEn, unitErrors,
                "longNamePluralEn", MAX_LONG_NAME_LENGTH, locale);
        validateCases(unit::getShortNameSingularRu, unitBuilder::putShortNameSingularRu, unitErrors,
                "shortNameSingularRu", MAX_SHORT_NAME_LENGTH, locale);
        validateCases(unit::getShortNamePluralRu, unitBuilder::putShortNamePluralRu, unitErrors,
                "shortNamePluralRu", MAX_SHORT_NAME_LENGTH, locale);
        validateCases(unit::getLongNameSingularRu, unitBuilder::putLongNameSingularRu, unitErrors,
                "longNameSingularRu", MAX_LONG_NAME_LENGTH, locale);
        validateCases(unit::getLongNamePluralRu, unitBuilder::putLongNamePluralRu, unitErrors,
                "longNamePluralRu", MAX_LONG_NAME_LENGTH, locale);
        validatePositiveNumber(unit::getBase, unitBuilder::base, unitErrors, "base", locale);
        validateNumberRange(unit::getPower, unitBuilder::power, MIN_POWER, MAX_POWER,
                unitErrors, "power", locale);
        if (unitErrors.hasAnyErrors()) {
            errors.add(unitErrors);
        } else {
            builder.addUnit(unitBuilder.build());
        }
    }

    private void validateKeyUniqueness(UnitCreateDto unit, Set<UnitModel> existingUnits,
                                       ErrorCollection.Builder unitErrors, Locale locale) {
        Set<String> existingKeys = existingUnits.stream().map(UnitModel::getKey).collect(Collectors.toSet());
        if (unit.getKey().isPresent() && existingKeys.contains(unit.getKey().get())) {
            unitErrors.addError("key", TypedError.invalid(messages
                    .getMessage("errors.duplicate.keys.are.forbidden", null, locale)));
        }
    }

    private void validateUnitOrdering(Supplier<Optional<Long>> baseGetter, Supplier<Optional<Long>> powerGetter,
                                      String newId, Set<UnitModel> existingUnits,
                                      ErrorCollection.Builder unitErrors, Locale locale) {
        Optional<Long> base = baseGetter.get();
        Optional<Long> power = powerGetter.get();
        if (base.isPresent() && power.isPresent() && base.get() > 0
                && power.get() >= MIN_POWER && power.get() <= MAX_POWER) {
            List<Tuple3<String, Long, Long>> sortedExistingMultipliers = existingUnits.stream()
                    .map(u -> Tuples.of(u.getId(), u.getBase(), u.getPower()))
                    .sorted(Comparator.comparing(t -> Tuples.of(t.getT2(), t.getT3()), MULTIPLIER_COMPARATOR))
                    .collect(Collectors.toList());
            List<Tuple3<String, Long, Long>> sortedUpdatedMultipliers = Streams
                    .concat(sortedExistingMultipliers.stream(), Stream.of(Tuples.of(newId, base.get(), power.get())))
                    .sorted(Comparator.comparing(t -> Tuples.of(t.getT2(), t.getT3()), MULTIPLIER_COMPARATOR))
                    .collect(Collectors.toList());
            Tuple3<String, Long, Long> updatedMin = sortedUpdatedMultipliers.get(0);
            Tuple3<String, Long, Long> existingMin = sortedExistingMultipliers.get(0);
            if (!isSameMultiplier(updatedMin.getT2(), existingMin.getT2(), updatedMin.getT3(), existingMin.getT3())) {
                unitErrors.addError(TypedError.invalid(messages
                        .getMessage("errors.smallest.unit.modification.is.forbidden", null, locale)));
            }
        }
    }

    private boolean isSameMultiplier(long baseL, long baseR, long powerL, long powerR) {
        if (baseL == baseR && powerL == powerR) {
            return true;
        }
        if (baseL == 1L && baseR == 1L) {
            return true;
        }
        if (powerL == 0L && powerR == 0L) {
            return true;
        }
        if ((powerL > 0 && powerR > 0) || (powerL < 0 && powerR < 0)) {
            return BigDecimal.valueOf(baseL).pow(Math.abs((int) powerL))
                    .compareTo(BigDecimal.valueOf(baseR).pow(Math.abs((int) powerR))) == 0;
        }
        return false;
    }

    private Result<Tuple2<UnitsEnsembleModel, Boolean>> validatePutUnit(UnitPutDto input, long version,
                                                                        UnitsEnsembleModel existing,
                                                                        UnitModel existingUnit, Locale locale) {
        UnitsEnsembleModel.Builder builder = UnitsEnsembleModel.builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        builder.id(existing.getId());
        builder.tenantId(existing.getTenantId());
        builder.deleted(existing.isDeleted());
        builder.fractionsAllowed(existing.isFractionsAllowed());
        builder.key(existing.getKey());
        builder.nameEn(existing.getNameEn());
        builder.nameRu(existing.getNameRu());
        builder.descriptionEn(existing.getDescriptionEn());
        builder.descriptionRu(existing.getDescriptionRu());
        existing.getUnits().stream().filter(u -> !Objects.equals(u.getId(), existingUnit.getId()))
                .forEach(builder::addUnit);
        versionValidator.validate(() -> Optional.of(version), existing::getVersion, builder::version, errors,
                "version", locale);
        validateUpdateUnit(input, existing.getUnits().stream()
                .filter(u -> !u.isDeleted()).collect(Collectors.toSet()), existingUnit, builder, errors, locale);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        if (builder.hasChanges(existing)) {
            return Result.success(Tuples.of(builder.build(), true));
        }
        return Result.success(Tuples.of(existing, false));
    }

    private void validateUpdateUnit(UnitPutDto unit, Set<UnitModel> existingUnits, UnitModel existingUnit,
                                    UnitsEnsembleModel.Builder builder, ErrorCollection.Builder errors,
                                    Locale locale) {
        UnitModel.Builder unitBuilder = UnitModel.builder();
        ErrorCollection.Builder unitErrors = ErrorCollection.builder();
        unitBuilder.id(existingUnit.getId());
        unitBuilder.key(existingUnit.getKey());
        unitBuilder.deleted(existingUnit.isDeleted());
        validatePutUnitOrdering(unit, existingUnits, existingUnit, unitErrors, locale);
        validateText(unit::getShortNameSingularEn, unitBuilder::shortNameSingularEn, unitErrors,
                "shortNameSingularEn", MAX_SHORT_NAME_LENGTH, locale);
        validateText(unit::getShortNamePluralEn, unitBuilder::shortNamePluralEn, unitErrors,
                "shortNamePluralEn", MAX_SHORT_NAME_LENGTH, locale);
        validateText(unit::getLongNameSingularEn, unitBuilder::longNameSingularEn, unitErrors,
                "longNameSingularEn", MAX_LONG_NAME_LENGTH, locale);
        validateText(unit::getLongNamePluralEn, unitBuilder::longNamePluralEn, unitErrors,
                "longNamePluralEn", MAX_LONG_NAME_LENGTH, locale);
        validateCases(unit::getShortNameSingularRu, unitBuilder::putShortNameSingularRu, unitErrors,
                "shortNameSingularRu", MAX_SHORT_NAME_LENGTH, locale);
        validateCases(unit::getShortNamePluralRu, unitBuilder::putShortNamePluralRu, unitErrors,
                "shortNamePluralRu", MAX_SHORT_NAME_LENGTH, locale);
        validateCases(unit::getLongNameSingularRu, unitBuilder::putLongNameSingularRu, unitErrors,
                "longNameSingularRu", MAX_LONG_NAME_LENGTH, locale);
        validateCases(unit::getLongNamePluralRu, unitBuilder::putLongNamePluralRu, unitErrors,
                "longNamePluralRu", MAX_LONG_NAME_LENGTH, locale);
        validatePositiveNumber(unit::getBase, unitBuilder::base, unitErrors, "base", locale);
        validateNumberRange(unit::getPower, unitBuilder::power, MIN_POWER, MAX_POWER,
                unitErrors, "power", locale);
        if (unitErrors.hasAnyErrors()) {
            errors.add(unitErrors);
        } else {
            builder.addUnit(unitBuilder.build());
        }
    }

    private void validatePutUnitOrdering(UnitPutDto unit, Set<UnitModel> existingUnits, UnitModel existingUnit,
                                         ErrorCollection.Builder unitErrors, Locale locale) {
        if (unit.getBase().isPresent() && unit.getPower().isPresent() && unit.getBase().get() > 0
                && unit.getPower().get() >= MIN_POWER && unit.getPower().get() <= MAX_POWER) {
            long newBase = unit.getBase().get();
            long newPower = unit.getPower().get();
            long currentBase = existingUnit.getBase();
            long currentPower = existingUnit.getPower();
            boolean changedMultiplier = newBase != currentBase || newPower != currentPower;
            if (!changedMultiplier) {
                return;
            }
            if (isSameMultiplier(newBase, currentBase, newPower, currentPower)) {
                return;
            }
            List<Tuple3<String, Long, Long>> sortedExistingMultipliers = existingUnits.stream()
                    .map(u -> Tuples.of(u.getId(), u.getBase(), u.getPower()))
                    .sorted(Comparator.comparing(t -> Tuples.of(t.getT2(), t.getT3()), MULTIPLIER_COMPARATOR))
                    .collect(Collectors.toList());
            Tuple3<String, Long, Long> existingMin = sortedExistingMultipliers.get(0);
            List<Tuple3<String, Long, Long>> sortedUpdatedMultipliers = Streams
                    .concat(sortedExistingMultipliers.stream()
                                    .filter(t -> !Objects.equals(t.getT1(), existingUnit.getId())),
                            Stream.of(Tuples.of(existingUnit.getId(), newBase, newPower)))
                    .sorted(Comparator.comparing(t -> Tuples.of(t.getT2(), t.getT3()), MULTIPLIER_COMPARATOR))
                    .collect(Collectors.toList());
            Tuple3<String, Long, Long> updatedMin = sortedUpdatedMultipliers.get(0);
            if (!isSameMultiplier(existingMin.getT2(), updatedMin.getT2(), existingMin.getT3(), updatedMin.getT3())) {
                unitErrors.addError(TypedError.invalid(messages
                        .getMessage("errors.smallest.unit.modification.is.forbidden", null, locale)));
            }
        }
    }

    private Result<UnitsEnsembleModel> validateDeleteUnit(long version, UnitsEnsembleModel existing,
                                                          UnitModel existingUnit, Locale locale) {
        UnitsEnsembleModel.Builder builder = UnitsEnsembleModel.builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        builder.id(existing.getId());
        builder.tenantId(existing.getTenantId());
        builder.deleted(existing.isDeleted());
        builder.fractionsAllowed(existing.isFractionsAllowed());
        builder.key(existing.getKey());
        builder.nameEn(existing.getNameEn());
        builder.nameRu(existing.getNameRu());
        builder.descriptionEn(existing.getDescriptionEn());
        builder.descriptionRu(existing.getDescriptionRu());
        existing.getUnits().stream().filter(u -> !Objects.equals(u.getId(), existingUnit.getId()))
                .forEach(builder::addUnit);
        versionValidator.validate(() -> Optional.of(version), existing::getVersion, builder::version, errors,
                "version", locale);
        validateRemoveUnit(existing.getUnits().stream()
                .filter(u -> !u.isDeleted()).collect(Collectors.toSet()), existingUnit, builder, errors, locale);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(builder.build());
    }

    private void validateRemoveUnit(Set<UnitModel> existingUnits, UnitModel existingUnit,
                                    UnitsEnsembleModel.Builder builder, ErrorCollection.Builder errors,
                                    Locale locale) {
        UnitModel.Builder unitBuilder = UnitModel.builder();
        ErrorCollection.Builder unitErrors = ErrorCollection.builder();
        unitBuilder.id(existingUnit.getId());
        unitBuilder.key(existingUnit.getKey());
        unitBuilder.deleted(true);
        unitBuilder.shortNameSingularEn(existingUnit.getShortNameSingularEn());
        unitBuilder.shortNamePluralEn(existingUnit.getShortNamePluralEn());
        unitBuilder.longNameSingularEn(existingUnit.getLongNameSingularEn());
        unitBuilder.longNamePluralEn(existingUnit.getLongNamePluralEn());
        existingUnit.getShortNameSingularRu().forEach(unitBuilder::putShortNameSingularRu);
        existingUnit.getShortNamePluralRu().forEach(unitBuilder::putShortNamePluralRu);
        existingUnit.getLongNameSingularRu().forEach(unitBuilder::putLongNameSingularRu);
        existingUnit.getLongNamePluralRu().forEach(unitBuilder::putLongNamePluralRu);
        unitBuilder.base(existingUnit.getBase());
        unitBuilder.power(existingUnit.getPower());
        validateRemoveUnitOrdering(existingUnits, existingUnit, unitErrors, locale);
        if (unitErrors.hasAnyErrors()) {
            errors.add(unitErrors);
        } else {
            builder.addUnit(unitBuilder.build());
        }
    }

    private void validateRemoveUnitOrdering(Set<UnitModel> existingUnits, UnitModel existingUnit,
                                            ErrorCollection.Builder unitErrors, Locale locale) {
        List<Tuple3<String, Long, Long>> sortedExistingMultipliers = existingUnits.stream()
                .map(u -> Tuples.of(u.getId(), u.getBase(), u.getPower()))
                .sorted(Comparator.comparing(t -> Tuples.of(t.getT2(), t.getT3()), MULTIPLIER_COMPARATOR))
                .collect(Collectors.toList());
        Tuple3<String, Long, Long> existingMin = sortedExistingMultipliers.get(0);
        List<Tuple3<String, Long, Long>> sortedUpdatedMultipliers = sortedExistingMultipliers.stream()
                .filter(t -> !Objects.equals(t.getT1(), existingUnit.getId()))
                .sorted(Comparator.comparing(t -> Tuples.of(t.getT2(), t.getT3()), MULTIPLIER_COMPARATOR))
                .collect(Collectors.toList());
        Tuple3<String, Long, Long> updatedMin = sortedUpdatedMultipliers.get(0);
        if (!isSameMultiplier(existingMin.getT2(), updatedMin.getT2(), existingMin.getT3(), updatedMin.getT3())) {
            unitErrors.addError(TypedError.invalid(messages
                    .getMessage("errors.smallest.unit.modification.is.forbidden", null, locale)));
        }
    }

    private Mono<Result<String>> validateKey(YdbTxSession session, Supplier<Optional<String>> getter,
                                             String fieldKey, Locale locale) {
        Optional<String> value = getter.get();
        if (value.isEmpty()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (value.get().isBlank()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        if (value.get().length() > MAX_KEY_LENGTH) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return unitsEnsemblesDao.existsByKey(session, Tenants.DEFAULT_TENANT_ID, value.get(), false).map(exists -> {
            if (exists) {
                ErrorCollection error = ErrorCollection.builder().addError(fieldKey,
                        TypedError.conflict(messages
                                .getMessage("errors.units.ensemble.key.already.exists", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(value.get());
        });
    }

    private static final class MultiplierComparator implements Comparator<Tuple2<Long, Long>> {

        @Override
        public int compare(Tuple2<Long, Long> l, Tuple2<Long, Long> r) {
            if (Objects.equals(l.getT1(), r.getT1()) && Objects.equals(l.getT2(), r.getT2())) {
                return 0;
            }
            if (l.getT2() == 0 && r.getT2() == 0) {
                return 0;
            }
            if (l.getT1() == 1 && r.getT1() == 1) {
                return 0;
            }
            if (l.getT2() < 0 && r.getT2() > 0) {
                return -1;
            }
            if (l.getT2() > 0 && r.getT2() < 0) {
                return 1;
            }
            if (Objects.equals(l.getT2(), r.getT2()) && l.getT2() > 0 && r.getT2() > 0) {
                return l.getT1().compareTo(r.getT1());
            }
            if (Objects.equals(l.getT2(), r.getT2()) && l.getT2() < 0 && r.getT2() < 0) {
                return r.getT1().compareTo(l.getT1());
            }
            if (Objects.equals(l.getT1(), r.getT1())) {
                return l.getT2().compareTo(r.getT2());
            }
            if (l.getT2() < 0 && r.getT2() < 0) {
                return BigDecimal.valueOf(r.getT1()).pow(Math.abs(r.getT2().intValue()))
                        .compareTo(BigDecimal.valueOf(l.getT1()).pow(Math.abs(l.getT2().intValue())));
            } else {
                return BigDecimal.valueOf(l.getT1()).pow(l.getT2().intValue())
                        .compareTo(BigDecimal.valueOf(r.getT1()).pow(r.getT2().intValue()));
            }
        }

    }

}
