package ru.yandex.intranet.d.services.usage

import com.google.common.util.concurrent.ThreadFactoryBuilder
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.ExecutorCoroutineDispatcher
import kotlinx.coroutines.asCoroutineDispatcher
import kotlinx.coroutines.withContext
import mu.KotlinLogging
import org.springframework.stereotype.Component
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.ScheduledThreadPoolExecutor
import java.util.concurrent.TimeUnit
import javax.annotation.PreDestroy

private val logger = KotlinLogging.logger {}

/**
 * Usage sync compute dispatcher.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class UsageSyncComputeDispatcher {

    private val scheduler: ScheduledExecutorService
    private val computeDispatcher: ExecutorCoroutineDispatcher

    init {
        val threadFactory = ThreadFactoryBuilder()
            .setDaemon(true)
            .setNameFormat("usage-sync-compute-pool-%d")
            .setUncaughtExceptionHandler { t: Thread, e: Throwable? ->
                logger.error(e) { "Uncaught exception in usage sync compute pool thread $t" }
            }
            .build()
        val scheduledThreadPoolExecutor = ScheduledThreadPoolExecutor((Runtime.getRuntime().availableProcessors() / 4)
            .coerceAtLeast(2), threadFactory)
        scheduledThreadPoolExecutor.removeOnCancelPolicy = true
        scheduler = scheduledThreadPoolExecutor
        computeDispatcher = scheduledThreadPoolExecutor.asCoroutineDispatcher()
    }

    @PreDestroy
    fun preDestroy() {
        logger.info("Stopping usage sync compute dispatcher...")
        computeDispatcher.close()
        try {
            scheduler.awaitTermination(1, TimeUnit.SECONDS)
        } catch (e: InterruptedException) {
            Thread.currentThread().interrupt()
        }
        scheduler.shutdownNow()
        logger.info("Stopped usage sync compute dispatcher")
    }

    suspend fun <T> execute(block: suspend CoroutineScope.() -> T): T {
        return withContext(computeDispatcher) { block() }
    }

}
