package ru.yandex.intranet.d.services.usage

import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.toList
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.inside.yt.kosher.cypress.YPath
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.annotation.YTreeField
import ru.yandex.inside.yt.kosher.impl.ytree.`object`.annotation.YTreeObject
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.providers.ProvidersDao
import ru.yandex.intranet.d.dao.resources.ResourcesDao
import ru.yandex.intranet.d.dao.resources.segmentations.ResourceSegmentationsDao
import ru.yandex.intranet.d.dao.resources.segments.ResourceSegmentsDao
import ru.yandex.intranet.d.dao.resources.types.ResourceTypesDao
import ru.yandex.intranet.d.dao.services.ServicesDao
import ru.yandex.intranet.d.dao.units.UnitsEnsemblesDao
import ru.yandex.intranet.d.dao.usage.ServiceUsageDao
import ru.yandex.intranet.d.dao.usage.UsageEpochsDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.kotlin.ProviderId
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.kotlin.ServiceId
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.elapsed
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel
import ru.yandex.intranet.d.model.services.ServiceSlugNode
import ru.yandex.intranet.d.model.units.UnitModel
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel
import ru.yandex.intranet.d.model.usage.ServiceUsageAmounts
import ru.yandex.intranet.d.model.usage.ServiceUsageKey
import ru.yandex.intranet.d.model.usage.ServiceUsageModel
import ru.yandex.intranet.d.model.usage.UsageAmount
import ru.yandex.intranet.d.model.usage.UsageEpochKey
import ru.yandex.intranet.d.model.usage.UsageEpochModel
import ru.yandex.intranet.d.services.integration.yt.YtReader
import ru.yandex.intranet.d.services.settings.RuntimeSettingsService
import ru.yandex.intranet.d.services.units.UnitsComparator
import ru.yandex.intranet.d.util.dispatchers.CustomDispatcher
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.util.units.Units
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import ru.yandex.monlib.metrics.labels.Labels
import ru.yandex.monlib.metrics.registry.MetricRegistry
import java.math.BigDecimal
import java.math.BigInteger
import java.math.RoundingMode
import java.time.Clock
import java.time.Duration
import java.time.Instant
import java.time.ZoneId
import java.time.format.DateTimeFormatter
import java.util.*
import java.util.concurrent.atomic.AtomicLong
import java.util.concurrent.atomic.AtomicReference

private val logger = KotlinLogging.logger {}

/**
 * YP usage sync service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class YpUsageSyncService(
    @Qualifier("messageSource") private val messages: MessageSource,
    @Qualifier("backgroundHeavyDispatcher") private val backgroundHeavyDispatcher: CustomDispatcher,
    @Qualifier("ytReaderHahn") private val ytReader: YtReader,
    private val usageSyncComputeDispatcher: UsageSyncComputeDispatcher,
    private val tableClient: YdbTableClient,
    private val runtimeSettingsService: RuntimeSettingsService,
    private val providersDao: ProvidersDao,
    private val resourceTypesDao: ResourceTypesDao,
    private val resourceSegmentationsDao: ResourceSegmentationsDao,
    private val resourceSegmentsDao: ResourceSegmentsDao,
    private val resourcesDao: ResourcesDao,
    private val servicesDao: ServicesDao,
    private val usageEpochsDao: UsageEpochsDao,
    private val serviceUsageDao: ServiceUsageDao,
    private val unitsEnsemblesDao: UnitsEnsemblesDao
) {

    private val lastSuccess: AtomicReference<Instant> = AtomicReference(Instant.now())

    init {
        MetricRegistry.root().lazyGaugeInt64("cron.jobs.duration_since_last_success_millis",
            Labels.of("job", "SyncYpUsageJob")) { Duration.between(lastSuccess.get(), Instant.now()).toMillis() }
    }

    fun doSyncWithValidationMono(user: YaUserDetails, locale: Locale, clock: Clock): Mono<Result<Unit>> {
        return mono { doSyncWithValidation(user, locale, clock) }
    }

    suspend fun doSync(clock: Clock) {
        val syncParameters = meter({ prepareSyncParameters() }, "YP usage synchronization, prepare parameters")
        if (syncParameters == null) {
            logger.warn { "YP usage synchronization, skipped due to invalid settings" }
            return
        }
        if (!syncParameters.enabled) {
            return
        }
        val serviceNodes = meter({ getAllServiceSlugNodes() }, "YP usage synchronization, load services")
        val hierarchy = meter({ usageSyncComputeDispatcher.execute { prepareHierarchy(serviceNodes) } },
            "YP usage synchronization, prepare service tree")
        logger.info { "YP usage synchronization, loaded service tree with ${hierarchy.roots.size} roots " +
            "and ${hierarchy.nodesCount} services" }
        val tableDate = prepareTableDate(clock)
        val tablePath = prepareTablePath(tableDate)
        meter({ validateYtAvailable() }, "YP usage synchronization, validating YT availability")
        if (!meter({ checkTableExists(tablePath) }, "YP usage synchronization, checking table existence")) {
            logger.warn { "YP usage synchronization, skipped due to $tablePath not exists" }
            return
        }
        val ytTable = meter({ loadYtTable(tablePath) }, "YP usage synchronization, loading YT table")
        val groupedUsage = meter({ usageSyncComputeDispatcher.execute { groupUsage(ytTable) } },
            "YP usage synchronization, group usages")
        val matchingResources = findMatchingResources(groupedUsage, syncParameters)
        val resourceEpochs = meter({ incrementAndGetEpochs(syncParameters.ypProvider.id,
            matchingResources.map { it.id }) }, "YP usage synchronization, increment and get epochs")
        val now = Instant.now(clock)
        val usageHierarchy = meter({ usageSyncComputeDispatcher.execute {
            prepareUsageHierarchy(hierarchy, groupedUsage, resourceEpochs, now, syncParameters) } },
            "YP usage synchronization, aggregating usage")
        meter({ storeAccumulatedUsage(usageHierarchy) }, "YP usage synchronization, updating usage, " +
            "${usageHierarchy.serviceUsages.size} service usages")
        val cleanupStats = meter({ cleanupPreviousEpochs(resourceEpochs) },
            "YP usage synchronization, clean up old epochs data")
        logger.info { "YP usage synchronization, " +
            "cleaned up ${cleanupStats.staleServiceUsages} service usages" }
        lastSuccess.set(Instant.now(clock))
    }

    private suspend fun getAllServiceSlugNodes(): List<ServiceSlugNode> {
        return dbSessionRetryable(tableClient) {
            servicesDao.getAllServiceSlugNodes(session).awaitSingle()
        }!!
    }

    private fun prepareHierarchy(nodes: List<ServiceSlugNode>): ServiceHierarchy {
        val roots = mutableListOf<ServiceId>()
        val childrenByParent = mutableMapOf<ServiceId, MutableSet<ServiceId>>()
        val slugByNode = mutableMapOf<ServiceId, String>()
        nodes.forEach {
            if (it.parentId == null) {
                roots.add(it.id)
            } else {
                childrenByParent.computeIfAbsent(it.parentId) { mutableSetOf() }.add(it.id)
            }
            slugByNode[it.id] = it.slug
        }
        return ServiceHierarchy(roots, childrenByParent, nodes.size, slugByNode)
    }

    private fun prepareTableDate(clock: Clock): String {
        return Instant.now(clock).atZone(ZoneId.of("Europe/Moscow")).toLocalDate().minusDays(1)
            .format(DateTimeFormatter.ISO_LOCAL_DATE)
    }

    private fun prepareTablePath(tableDate: String): String {
        return "//home/runtimecloud/cpu_usage_tables/$tableDate"
    }

    private suspend fun validateYtAvailable() {
        ytReader.checkClusterLiveness()
    }

    private suspend fun checkTableExists(tableName: String): Boolean {
        return ytReader.checkNodeExists(YPath.simple(tableName))
    }

    private suspend fun loadYtTable(tableName: String): List<YpUsageTableRow> {
        return ytReader.readTable(YPath.simple(tableName)
            .withColumns("abc_slug", "potenentially_idle_vcpu", "dc", "deploy_engine", "segment"),
                    YpUsageTableRow::class.java)
            .filter { it.abcSlug != null && it.potentiallyIdleVCpu != null && it.dc != null
                && it.segment != null && it.deployEngine == "YP" }.toList()
    }

    private fun groupUsage(table: List<YpUsageTableRow>): GroupedUsage {
        // ABC slug, cluster key, node segment key
        val usage = mutableMapOf<String, MutableMap<String, MutableMap<String, MutableList<Double>>>>()
        // Cluster key, node segment keys
        val keys = mutableMapOf<String, MutableSet<String>>()
        table.forEach {
            usage.computeIfAbsent(it.abcSlug!!) { mutableMapOf() }
                .computeIfAbsent(it.dc!!) { mutableMapOf() }
                .computeIfAbsent(it.segment!!) { mutableListOf() }.add(it.potentiallyIdleVCpu!!)
            keys.computeIfAbsent(it.dc) { mutableSetOf() }.add(it.segment)
        }
        return GroupedUsage(usage, keys)
    }

    private fun findMatchingResources(usage: GroupedUsage, syncParameters: SyncParameters): List<ResourceModel> {
        val result = mutableListOf<ResourceModel>()
        usage.clusterSegments.forEach { (clusterKey, segmentKeys) ->
            val clusterResources = syncParameters.cpuResourcesByClusterAndSegment[clusterKey] ?: emptyMap()
            segmentKeys.forEach { segmentKey ->
                val segmentResource = clusterResources[segmentKey]
                if (segmentResource != null) {
                    result.add(segmentResource)
                }
            }
        }
        return result
    }

    private suspend fun incrementAndGetEpochs(providerId: ProviderId,
                                              resourceIds: Collection<ResourceId>): Map<ResourceId, UsageEpochModel> {
        return dbSessionRetryable(tableClient) {
            rwTxRetryable {
                val keys = resourceIds.map { UsageEpochKey(Tenants.DEFAULT_TENANT_ID, providerId, it) }
                val epochs = usageEpochsDao.getByIds(txSession, keys).associateBy { it.key }
                val incrementedEpochs = keys.map {
                    val currentEpoch = epochs.getOrDefault(it, UsageEpochModel(it, -1))
                    UsageEpochModel(it, currentEpoch.epoch + 1)
                }
                usageEpochsDao.upsertManyRetryable(txSession, incrementedEpochs)
                incrementedEpochs.associateBy { it.key.resourceId }
            }!!
        }!!
    }

    private fun prepareUsageHierarchy(serviceHierarchy: ServiceHierarchy,
                                      usage: GroupedUsage,
                                      epochs: Map<ResourceId, UsageEpochModel>,
                                      now: Instant,
                                      syncParameters: SyncParameters): AccumulatedUsages {
        // Prepare aggregates using post-order depth first traversal
        val serviceUsages = mutableListOf<ServiceUsageModel>()
        // More than one root may exist
        serviceHierarchy.roots.forEach { root ->
            // Traversal to accumulate aggregates
            traverseForAggregates(serviceHierarchy, usage, epochs, now, root, serviceUsages, syncParameters)
        }
        return AccumulatedUsages(serviceUsages)
    }

    private fun traverseForAggregates(hierarchy: ServiceHierarchy,
                                      usage: GroupedUsage,
                                      epochs: Map<ResourceId, UsageEpochModel>,
                                      now: Instant,
                                      root: ServiceId,
                                      serviceUsage: MutableList<ServiceUsageModel>,
                                      syncParameters: SyncParameters) {
        val stack = ArrayDeque<TraversalStackFrame>()
        stack.push(
            TraversalStackFrame(
                root, (hierarchy.childrenByParent[root] ?: setOf()).toMutableSet(), mutableMapOf(), null
            )
        )
        var counter = 0
        val failsafeLimit = hierarchy.nodesCount.toLong() * 3
        while (!stack.isEmpty()) {
            val currentStackFrame = stack.peek()
            if (currentStackFrame.unvisitedChildren.isEmpty()) {
                // Visit this node
                val nodeAggregates = if (currentStackFrame.visitedChildren.isEmpty()) {
                    // Leaf node
                    aggregateLeafServiceResources(hierarchy, usage, epochs,
                        currentStackFrame.id, now, syncParameters, serviceUsage)
                } else {
                    // Non-leaf node
                    aggregateServiceResources(hierarchy, usage,
                        currentStackFrame.visitedChildren.values, epochs, currentStackFrame.id, now,
                        syncParameters, serviceUsage)
                }
                if (currentStackFrame.parentFrame != null) {
                    // Non-root node
                    currentStackFrame.parentFrame.unvisitedChildren.remove(currentStackFrame.id)
                    currentStackFrame.parentFrame.visitedChildren[currentStackFrame.id] = nodeAggregates
                }
                stack.pop()
            } else {
                // Go deeper to next child
                val nextChildNode = currentStackFrame.unvisitedChildren.first()
                stack.push(
                    TraversalStackFrame(nextChildNode,
                        (hierarchy.childrenByParent[nextChildNode] ?: setOf()).toMutableSet(), mutableMapOf(),
                        currentStackFrame
                    )
                )
            }
            counter++
            if (counter > failsafeLimit) {
                throw IllegalStateException("Too many tree traversal iterations")
            }
        }
    }

    private fun aggregateLeafServiceResources(hierarchy: ServiceHierarchy,
                                              usage: GroupedUsage,
                                              epochs: Map<ResourceId, UsageEpochModel>,
                                              serviceId: ServiceId,
                                              now: Instant,
                                              syncParameters: SyncParameters,
                                              serviceUsages: MutableList<ServiceUsageModel>,
    ): Map<ResourceId, BigInteger> {
        val result = mutableMapOf<ResourceId, BigInteger>()
        val serviceSlug = hierarchy.slugByNode[serviceId]!!
        val usagesForService = usage.usage[serviceSlug] ?: emptyMap()
        usagesForService.forEach { (clusterKey, usagesBySegmentKey) ->
            usagesBySegmentKey.forEach { (segmentKey, usages) ->
                val resource = (syncParameters.cpuResourcesByClusterAndSegment[clusterKey] ?: emptyMap())[segmentKey]
                if (resource != null) {
                    val unusedSum = usages.sumOf { BigDecimal.valueOf(if (it < 0.0) { 0.0 } else { it }) }
                    val unusedBaseUnits = Units.convert(unusedSum, syncParameters.coresUnit,
                        syncParameters.cpuBaseUnit).setScale(0, RoundingMode.HALF_UP).toBigInteger()
                    if (unusedBaseUnits > BigInteger.ZERO) {
                        result[resource.id] = unusedBaseUnits
                        serviceUsages.add(ServiceUsageModel(
                            key = ServiceUsageKey(
                                tenantId = resource.tenantId,
                                serviceId = serviceId,
                                providerId = syncParameters.ypProvider.id,
                                resourceId = resource.id
                            ),
                            lastUpdate = now,
                            epoch = epochs[resource.id]!!.epoch,
                            usage = ServiceUsageAmounts(
                                own = UsageAmount(
                                    value = null,
                                    average = null,
                                    min = null,
                                    max = null,
                                    median = null,
                                    variance = null,
                                    accumulated = null,
                                    accumulatedDuration = null,
                                    histogram = null,
                                    values = null,
                                    valuesX = null,
                                    valuesY = null,
                                    unused = unusedBaseUnits
                                ),
                                subtree = null,
                                total = null
                            )
                        ))
                    }
                }
            }
        }
        return result
    }

    private fun aggregateServiceResources(hierarchy: ServiceHierarchy,
                                          usage: GroupedUsage,
                                          childAggregates: Collection<Map<ResourceId, BigInteger>>,
                                          epochs: Map<ResourceId, UsageEpochModel>,
                                          serviceId: ServiceId,
                                          now: Instant,
                                          syncParameters: SyncParameters,
                                          serviceUsages: MutableList<ServiceUsageModel>,
    ): Map<ResourceId, BigInteger> {
        val result = mutableMapOf<ResourceId, BigInteger>()
        val serviceSlug = hierarchy.slugByNode[serviceId]!!
        val usagesForService = usage.usage[serviceSlug] ?: emptyMap()
        val serviceOwnUsage = mutableMapOf<ResourceId, BigInteger>()
        usagesForService.forEach { (clusterKey, usagesBySegmentKey) ->
            usagesBySegmentKey.forEach { (segmentKey, usages) ->
                val resource = (syncParameters.cpuResourcesByClusterAndSegment[clusterKey] ?: emptyMap())[segmentKey]
                if (resource != null) {
                    val unusedSum = usages.sumOf { BigDecimal.valueOf(if (it < 0.0) { 0.0 } else { it }) }
                    val unusedBaseUnits = Units.convert(unusedSum, syncParameters.coresUnit,
                        syncParameters.cpuBaseUnit).setScale(0, RoundingMode.HALF_UP).toBigInteger()
                    if (unusedBaseUnits > BigInteger.ZERO) {
                        serviceOwnUsage[resource.id] = unusedBaseUnits
                    }
                }
            }
        }
        val serviceSubtreeUsage = mutableMapOf<ResourceId, BigInteger>()
        childAggregates.forEach { childUsage ->
            childUsage.forEach { (resourceId, unused) ->
                val sum = (serviceSubtreeUsage[resourceId] ?: BigInteger.ZERO).add(unused)
                if (sum > BigInteger.ZERO) {
                    serviceSubtreeUsage[resourceId] = sum
                }
            }
        }
        val serviceTotalUsage = mutableMapOf<ResourceId, BigInteger>()
        serviceTotalUsage.putAll(serviceOwnUsage)
        serviceSubtreeUsage.forEach { (resourceId, unused) ->
            val sum = (serviceTotalUsage[resourceId] ?: BigInteger.ZERO).add(unused)
            if (sum > BigInteger.ZERO) {
                serviceTotalUsage[resourceId] = sum
            }
        }
        result.putAll(serviceTotalUsage)
        val allResourceIds = serviceOwnUsage.keys + serviceSubtreeUsage.keys + serviceTotalUsage.keys
        allResourceIds.forEach { resourceId ->
            val ownUnused = serviceOwnUsage[resourceId]
            val subtreeUnused = serviceSubtreeUsage[resourceId]
            val totalUnused = serviceTotalUsage[resourceId]
            val ownUsage = if (ownUnused != null && ownUnused > BigInteger.ZERO) {
                UsageAmount(
                    value = null,
                    average = null,
                    min = null,
                    max = null,
                    median = null,
                    variance = null,
                    accumulated = null,
                    accumulatedDuration = null,
                    histogram = null,
                    values = null,
                    valuesX = null,
                    valuesY = null,
                    unused = ownUnused
                )
            } else {
                null
            }
            val subtreeUsage = if (subtreeUnused != null && subtreeUnused > BigInteger.ZERO) {
                UsageAmount(
                    value = null,
                    average = null,
                    min = null,
                    max = null,
                    median = null,
                    variance = null,
                    accumulated = null,
                    accumulatedDuration = null,
                    histogram = null,
                    values = null,
                    valuesX = null,
                    valuesY = null,
                    unused = subtreeUnused
                )
            } else {
                null
            }
            val totalUsage = if (subtreeUnused != null && totalUnused != null && totalUnused > BigInteger.ZERO) {
                UsageAmount(
                    value = null,
                    average = null,
                    min = null,
                    max = null,
                    median = null,
                    variance = null,
                    accumulated = null,
                    accumulatedDuration = null,
                    histogram = null,
                    values = null,
                    valuesX = null,
                    valuesY = null,
                    unused = totalUnused
                )
            } else {
                null
            }
            if (ownUsage != null || subtreeUsage != null || totalUsage != null) {
                serviceUsages.add(ServiceUsageModel(
                    key = ServiceUsageKey(
                        tenantId = syncParameters.ypProvider.tenantId,
                        serviceId = serviceId,
                        providerId = syncParameters.ypProvider.id,
                        resourceId = resourceId
                    ),
                    lastUpdate = now,
                    epoch = epochs[resourceId]!!.epoch,
                    usage = ServiceUsageAmounts(
                        own = ownUsage,
                        subtree = subtreeUsage,
                        total = totalUsage
                    )
                ))
            }
        }
        return result
    }

    private suspend fun storeAccumulatedUsage(usages: AccumulatedUsages) {
        dbSessionRetryable(tableClient) {
            usages.serviceUsages.chunked(250)
                .forEach { serviceUsageDao.upsertManyRetryable(rwSingleRetryableCommit(), it) }
        }
    }

    private suspend fun cleanupPreviousEpochs(epochs: Map<ResourceId, UsageEpochModel>): CleanupStats {
        val staleServiceUsagesCounter = AtomicLong(0L)
        epochs.values.forEach { epoch ->
            var nextServiceUsage = dbSessionRetryable(tableClient) {
                rwTxRetryableOptimized(
                    { serviceUsageDao.getKeysForOlderEpochsFirstPage(txSession, epoch.key.tenantId,
                        epoch.key.providerId, epoch.key.resourceId, epoch.epoch, 250) },
                    { page -> page },
                    { page ->
                        if (page.keys.isEmpty()) {
                            txSession.commitTransaction().awaitSingleOrNull()
                        } else {
                            staleServiceUsagesCounter.addAndGet(page.keys.size.toLong())
                            serviceUsageDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                        }
                        page.nextFrom
                    })
            }
            while (nextServiceUsage != null) {
                nextServiceUsage = dbSessionRetryable(tableClient) {
                    rwTxRetryableOptimized(
                        { serviceUsageDao.getKeysForOlderEpochsNextPage(txSession, nextServiceUsage!!, 250) },
                        { page -> page },
                        { page ->
                            if (page.keys.isEmpty()) {
                                txSession.commitTransaction().awaitSingleOrNull()
                            } else {
                                staleServiceUsagesCounter.addAndGet(page.keys.size.toLong())
                                serviceUsageDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                            }
                            page.nextFrom
                        })
                }
            }
        }
        return CleanupStats(staleServiceUsagesCounter.get())
    }

    private suspend fun doSyncWithValidation(user: YaUserDetails, locale: Locale, clock: Clock): Result<Unit> = binding {
        validatePermissions(user, locale).bind()
        launchInBackground { doSync(clock) }
        Result.success(Unit)
    }

    private fun launchInBackground(block: suspend () -> Unit) {
        try {
            backgroundHeavyDispatcher.launch {
                try {
                    block()
                } catch (e: Exception) {
                    logger.error(e) { "Error during YP usage sync" }
                    if (e is CancellationException) {
                        throw e
                    }
                }
            }
        } catch (e: Exception) {
            logger.error(e) { "Error during YP usage sync" }
        }
    }

    private fun validatePermissions(user: YaUserDetails, locale: Locale): Result<Unit> {
        if (user.user.isEmpty || !user.user.get().dAdmin) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.forbidden(messages
                .getMessage("errors.access.denied", null, locale))).build())
        }
        return Result.success(Unit)
    }

    private suspend fun prepareSyncParameters(): SyncParameters? {
        val knownProviders = runtimeSettingsService.getKnownProvidersImmediate(Tenants.DEFAULT_TENANT_ID)
        if (knownProviders?.yp == null) {
            logger.warn { "YP is not defined as known provider, YP usage synchronization will be skipped" }
            return null
        }
        val ypUsageSyncSettings = runtimeSettingsService.getYpUsageSyncSettingsImmediate(Tenants.DEFAULT_TENANT_ID)
        if (ypUsageSyncSettings?.cpuResourceType == null || ypUsageSyncSettings.clusterSegmentation == null
            || ypUsageSyncSettings.nodeSegmentSegmentation == null || ypUsageSyncSettings.defaultNodeSegment == null
            || ypUsageSyncSettings.devNodeSegment == null || ypUsageSyncSettings.coresUnitsEnsemble == null
            || ypUsageSyncSettings.coresUnit == null || ypUsageSyncSettings.syncEnabled == null) {
            logger.warn { "YP usage sync settings are undefined or incomplete, YP usage synchronization will be skipped" }
            return null
        }
        if (knownProviders.ypPreStable != null && (ypUsageSyncSettings.preStableCpuResourceType == null
                || ypUsageSyncSettings.preStableClusterSegmentation == null
                || ypUsageSyncSettings.preStableNodeSegmentSegmentation == null
                || ypUsageSyncSettings.preStableDefaultNodeSegment == null
                || ypUsageSyncSettings.preStableDevNodeSegment == null)) {
            logger.warn { "YP usage sync settings are undefined or incomplete, YP usage synchronization will be skipped" }
            return null
        }
        val ypProviderO = dbSessionRetryable(tableClient) {
            providersDao.getById(roStaleSingleRetryableCommit(), knownProviders.yp,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (ypProviderO.isEmpty || ypProviderO.get().isDeleted) {
            logger.warn { "YP provider not found, YP usage synchronization will be skipped" }
            return null
        }
        val ypProvider = ypProviderO.get()
        val ypPreStableProvider = if (knownProviders.ypPreStable != null) {
            val ypPreStableProviderO = dbSessionRetryable(tableClient) {
                providersDao.getById(roStaleSingleRetryableCommit(), knownProviders.ypPreStable,
                    Tenants.DEFAULT_TENANT_ID).awaitSingle()
            }!!
            if (ypPreStableProviderO.isEmpty || ypPreStableProviderO.get().isDeleted) {
                logger.warn { "YP pre-stable provider not found, YP usage synchronization will be skipped" }
                return null
            }
            ypPreStableProviderO.get()
        } else {
            null
        }
        val cpuResourceTypeO = dbSessionRetryable(tableClient) {
            resourceTypesDao.getById(roStaleSingleRetryableCommit(), ypUsageSyncSettings.cpuResourceType,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (cpuResourceTypeO.isEmpty || cpuResourceTypeO.get().isDeleted) {
            logger.warn { "YP cpu resource type not found, YP usage synchronization will be skipped" }
            return null
        }
        val cpuResourceType = cpuResourceTypeO.get()
        if (cpuResourceType.providerId != ypProvider.id) {
            logger.warn { "Inconsistent cpu resource type, YP usage synchronization will be skipped" }
            return null
        }
        val preStableCpuResourceType = if (ypPreStableProvider != null) {
            val preStableCpuResourceTypeO = dbSessionRetryable(tableClient) {
                resourceTypesDao.getById(roStaleSingleRetryableCommit(), ypUsageSyncSettings.preStableCpuResourceType,
                    Tenants.DEFAULT_TENANT_ID).awaitSingle()
            }!!
            if (preStableCpuResourceTypeO.isEmpty || preStableCpuResourceTypeO.get().isDeleted) {
                logger.warn { "YP pre-stable cpu resource type not found, YP usage synchronization will be skipped" }
                return null
            }
            val preStableCpuResourceType = preStableCpuResourceTypeO.get()
            if (preStableCpuResourceType.providerId != ypPreStableProvider.id) {
                logger.warn { "Inconsistent pre-stable cpu resource type, YP usage synchronization will be skipped" }
                return null
            }
            preStableCpuResourceType
        } else {
            null
        }
        val segmentationIds = listOfNotNull(ypUsageSyncSettings.clusterSegmentation,
            ypUsageSyncSettings.nodeSegmentSegmentation, ypUsageSyncSettings.preStableClusterSegmentation,
            ypUsageSyncSettings.preStableNodeSegmentSegmentation).map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }
        val segmentations = dbSessionRetryable(tableClient) {
            resourceSegmentationsDao.getByIds(roStaleSingleRetryableCommit(), segmentationIds).awaitSingle()
        }!!.associateBy { it.id }
        if (segmentations[ypUsageSyncSettings.clusterSegmentation]?.isDeleted != false
            || segmentations[ypUsageSyncSettings.nodeSegmentSegmentation]?.isDeleted != false) {
            logger.warn { "YP resource segmentations not found, YP usage synchronization will be skipped" }
            return null
        }
        val clusterSegmentation = segmentations[ypUsageSyncSettings.clusterSegmentation]!!
        val nodeSegmentSegmentation = segmentations[ypUsageSyncSettings.nodeSegmentSegmentation]!!
        if (clusterSegmentation.providerId != ypProvider.id || nodeSegmentSegmentation.providerId != ypProvider.id) {
            logger.warn { "Inconsistent YP resource segmentations, YP usage synchronization will be skipped" }
            return null
        }
        val preStableClusterSegmentation = if (ypPreStableProvider != null) {
            if (segmentations[ypUsageSyncSettings.preStableClusterSegmentation]?.isDeleted != false) {
                logger.warn { "YP pre-stable resource segmentations not found, YP usage synchronization will be skipped" }
                return null
            }
            val preStableClusterSegmentation = segmentations[ypUsageSyncSettings.preStableClusterSegmentation]!!
            if (preStableClusterSegmentation.providerId != ypPreStableProvider.id) {
                logger.warn { "Inconsistent YP pre-stable resource segmentations, YP usage synchronization will be skipped" }
                return null
            }
            preStableClusterSegmentation
        } else {
            null
        }
        val preStableNodeSegmentSegmentation = if (ypPreStableProvider != null) {
            if (segmentations[ypUsageSyncSettings.preStableNodeSegmentSegmentation]?.isDeleted != false) {
                logger.warn { "YP pre-stable resource segmentations not found, YP usage synchronization will be skipped" }
                return null
            }
            val preStableNodeSegmentSegmentation = segmentations[ypUsageSyncSettings.preStableNodeSegmentSegmentation]!!
            if (preStableNodeSegmentSegmentation.providerId != ypPreStableProvider.id) {
                logger.warn { "Inconsistent YP pre-stable resource segmentations, YP usage synchronization will be skipped" }
                return null
            }
            preStableNodeSegmentSegmentation
        } else {
            null
        }
        val segmentIds = listOfNotNull(ypUsageSyncSettings.defaultNodeSegment, ypUsageSyncSettings.devNodeSegment,
            ypUsageSyncSettings.preStableDefaultNodeSegment, ypUsageSyncSettings.preStableDevNodeSegment)
            .map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }
        val segments = dbSessionRetryable(tableClient) {
            resourceSegmentsDao.getByIds(roStaleSingleRetryableCommit(), segmentIds).awaitSingle()
        }!!.associateBy { it.id }
        if (segments[ypUsageSyncSettings.defaultNodeSegment]?.isDeleted != false
            || segments[ypUsageSyncSettings.devNodeSegment]?.isDeleted != false) {
            logger.warn { "YP resource segments not found, YP usage synchronization will be skipped" }
            return null
        }
        val defaultSegment = segments[ypUsageSyncSettings.defaultNodeSegment]!!
        val devSegment = segments[ypUsageSyncSettings.devNodeSegment]!!
        if (defaultSegment.segmentationId != nodeSegmentSegmentation.id
            || devSegment.segmentationId != nodeSegmentSegmentation.id) {
            logger.warn { "Inconsistent YP resource segments, YP usage synchronization will be skipped" }
            return null
        }
        val preStableDefaultSegment = if (preStableNodeSegmentSegmentation != null) {
            if (segments[ypUsageSyncSettings.preStableDefaultNodeSegment]?.isDeleted != false) {
                logger.warn { "YP pre-stable resource segments not found, YP usage synchronization will be skipped" }
                return null
            }
            val preStableDefaultSegment = segments[ypUsageSyncSettings.preStableDefaultNodeSegment]!!
            if (preStableDefaultSegment.segmentationId != preStableNodeSegmentSegmentation.id) {
                logger.warn { "Inconsistent YP pre-stable resource segments, YP usage synchronization will be skipped" }
                return null
            }
            preStableDefaultSegment
        } else {
            null
        }
        val preStableDevSegment = if (preStableNodeSegmentSegmentation != null) {
            if (segments[ypUsageSyncSettings.preStableDevNodeSegment]?.isDeleted != false) {
                logger.warn { "YP pre-stable resource segments not found, YP usage synchronization will be skipped" }
                return null
            }
            val preStableDevSegment = segments[ypUsageSyncSettings.preStableDevNodeSegment]!!
            if (preStableDevSegment.segmentationId != preStableNodeSegmentSegmentation.id) {
                logger.warn { "Inconsistent YP pre-stable resource segments, YP usage synchronization will be skipped" }
                return null
            }
            preStableDevSegment
        } else {
            null
        }
        val ypResources = dbSessionRetryable(tableClient) {
            resourcesDao.getAllByProviderResourceType(roStaleSingleRetryableCommit(), ypProvider.id,
                cpuResourceType.id, Tenants.DEFAULT_TENANT_ID, false).awaitSingle()
        }!!.filter { !it.isDeleted }
        val ypCpuResources = ypResources.filter { resource ->
            resource.segments.any { it.segmentationId == clusterSegmentation.id }
                && resource.segments.any { it.segmentationId == nodeSegmentSegmentation.id }
        }
        if (ypCpuResources.isEmpty()) {
            logger.warn { "No cpu resources in YP, YP usage synchronization will be skipped" }
            return null
        }
        val resourceSegmentIds = ypCpuResources.flatMap { r -> r.segments
            .filter { s -> s.segmentationId == clusterSegmentation.id || s.segmentationId == nodeSegmentSegmentation.id }
            .map { s -> s.segmentId } }.distinct()
        val resourceSegments = dbSessionRetryable(tableClient) {
            resourceSegmentIds.chunked(1000).map { p -> resourceSegmentsDao.getByIds(
                roStaleSingleRetryableCommit(), p.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }
                .flatten()
        }!!.associateBy { it.id }
        val ypCpuResourcesByClusterAndSegment = ypCpuResources
            .groupBy { r -> resourceSegments[r.segments
                .first { s -> s.segmentationId == clusterSegmentation.id }.segmentId]!!.key }
            .mapValues { e -> e.value.associateBy { v -> resourceSegments[v.segments
                .first { s -> s.segmentationId == nodeSegmentSegmentation.id }.segmentId]!!.key } }
        val preStableCpuResourcesByClusterAndSegment = if (ypPreStableProvider != null) {
            val preStableYpResources = dbSessionRetryable(tableClient) {
                resourcesDao.getAllByProviderResourceType(roStaleSingleRetryableCommit(), ypPreStableProvider.id,
                    preStableCpuResourceType!!.id, Tenants.DEFAULT_TENANT_ID, false).awaitSingle()
            }!!.filter { !it.isDeleted }
            val preStableYpCpuResources = preStableYpResources.filter { resource ->
                resource.segments.any { it.segmentationId == preStableClusterSegmentation!!.id }
                    && resource.segments.any { it.segmentationId == preStableNodeSegmentSegmentation!!.id }
            }
            if (preStableYpCpuResources.isEmpty()) {
                logger.warn { "No cpu resources in pre-stable YP, YP usage synchronization will be skipped" }
                return null
            }
            val preStableResourceSegmentIds = preStableYpCpuResources.flatMap { r -> r.segments
                .filter { s -> s.segmentationId == preStableClusterSegmentation!!.id
                    || s.segmentationId == preStableNodeSegmentSegmentation!!.id }
                .map { s -> s.segmentId } }.distinct()
            val preStableResourceSegments = dbSessionRetryable(tableClient) {
                preStableResourceSegmentIds.chunked(1000).map { p -> resourceSegmentsDao.getByIds(
                    roStaleSingleRetryableCommit(), p.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }
                    .flatten()
            }!!.associateBy { it.id }
            preStableYpCpuResources
                .groupBy { r -> preStableResourceSegments[r.segments
                    .first { s -> s.segmentationId == preStableClusterSegmentation!!.id }.segmentId]!!.key }
                .mapValues { e -> e.value.associateBy { v -> preStableResourceSegments[v.segments
                    .first { s -> s.segmentationId == preStableNodeSegmentSegmentation!!.id }.segmentId]!!.key } }
        } else {
            null
        }
        val coresUnitsEnsembleO = dbSessionRetryable(tableClient) {
            unitsEnsemblesDao.getById(roStaleSingleRetryableCommit(), ypUsageSyncSettings.coresUnitsEnsemble,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (coresUnitsEnsembleO.isEmpty || coresUnitsEnsembleO.get().isDeleted) {
            logger.warn { "Cores units ensemble not found, YP usage synchronization will be skipped" }
            return null
        }
        val coresUnitsEnsemble = coresUnitsEnsembleO.get()
        if (coresUnitsEnsemble.id != cpuResourceType.unitsEnsembleId) {
            logger.warn { "Inconsistent cores units ensemble, YP usage synchronization will be skipped" }
            return null
        }
        if (preStableCpuResourceType != null && coresUnitsEnsemble.id != preStableCpuResourceType.unitsEnsembleId) {
            logger.warn { "Inconsistent cores units ensemble for pre-stable, YP usage synchronization will be skipped" }
            return null
        }
        val coresUnitO = coresUnitsEnsemble.unitById(ypUsageSyncSettings.coresUnit)
        if (coresUnitO.isEmpty || coresUnitO.get().isDeleted) {
            logger.warn { "Cores unit not found, YP usage synchronization will be skipped" }
            return null
        }
        val coresUnit = coresUnitO.get()
        val cpuBaseUnit = if (cpuResourceType.baseUnitId != null) {
            coresUnitsEnsemble.unitById(cpuResourceType.baseUnitId).orElseThrow()
        } else {
            UnitsComparator.getBaseUnit(coresUnitsEnsemble)
        }
        val preStableCpuBaseUnit = if (preStableCpuResourceType != null) {
            if (preStableCpuResourceType.baseUnitId != null) {
                coresUnitsEnsemble.unitById(preStableCpuResourceType.baseUnitId).orElseThrow()
            } else {
                UnitsComparator.getBaseUnit(coresUnitsEnsemble)
            }
        } else {
            null
        }
        return SyncParameters(
            cpuResourcesByClusterAndSegment = ypCpuResourcesByClusterAndSegment,
            preStableCpuResourcesByClusterAndSegment = preStableCpuResourcesByClusterAndSegment,
            ypProvider = ypProvider,
            preStableYpProvider = ypPreStableProvider,
            enabled = ypUsageSyncSettings.syncEnabled,
            coresUnitsEnsemble = coresUnitsEnsemble,
            coresUnit = coresUnit,
            cpuBaseUnit = cpuBaseUnit,
            preStableCpuBaseUnit = preStableCpuBaseUnit,
            defaultSegment = defaultSegment,
            devSegment = devSegment,
            clusterSegmentation = clusterSegmentation,
            nodeSegmentSegmentation = nodeSegmentSegmentation,
            preStableDefaultSegment = preStableDefaultSegment,
            preStableDevSegment = preStableDevSegment,
            preStableClusterSegmentation = preStableClusterSegmentation,
            preStableNodeSegmentSegmentation = preStableNodeSegmentSegmentation
        )
    }

    private inline fun <T> meter(statement: () -> T, label: String): T {
        return elapsed(statement)
                { millis, success -> logger.info { "$label: duration = $millis ms, success = $success" } }
    }

    private data class SyncParameters(
        // Resource type = cpu, map by cluster key and segment key
        val cpuResourcesByClusterAndSegment: Map<String, Map<String, ResourceModel>>,
        val preStableCpuResourcesByClusterAndSegment: Map<String, Map<String, ResourceModel>>?,
        val ypProvider: ProviderModel,
        val preStableYpProvider: ProviderModel?,
        val enabled: Boolean,
        val coresUnitsEnsemble: UnitsEnsembleModel,
        val coresUnit: UnitModel,
        val cpuBaseUnit: UnitModel,
        val preStableCpuBaseUnit: UnitModel?,
        val defaultSegment: ResourceSegmentModel,
        val devSegment: ResourceSegmentModel,
        val clusterSegmentation: ResourceSegmentationModel,
        val nodeSegmentSegmentation: ResourceSegmentationModel,
        val preStableDefaultSegment: ResourceSegmentModel?,
        val preStableDevSegment: ResourceSegmentModel?,
        val preStableClusterSegmentation: ResourceSegmentationModel?,
        val preStableNodeSegmentSegmentation: ResourceSegmentationModel?
    )

    private data class ServiceHierarchy(
        val roots: List<ServiceId>,
        val childrenByParent: Map<ServiceId, Set<ServiceId>>,
        val nodesCount: Int,
        val slugByNode: Map<ServiceId, String>
    )

    @YTreeObject
    data class YpUsageTableRow(
        @YTreeField(key = "abc_slug")
        val abcSlug: String?,
        @YTreeField(key = "potenentially_idle_vcpu")
        val potentiallyIdleVCpu: Double?,
        @YTreeField(key = "dc")
        val dc: String?,
        @YTreeField(key = "deploy_engine")
        val deployEngine: String?,
        @YTreeField(key = "segment")
        val segment: String?
    )

    private data class GroupedUsage(
        // ABC slug, cluster key, node segment key
        val usage: Map<String, Map<String, Map<String, List<Double>>>>,
        // Cluster key, node segment keys
        val clusterSegments: Map<String, Set<String>>
    )

    private data class AccumulatedUsages(
        val serviceUsages: List<ServiceUsageModel>
    )

    private data class CleanupStats(
        val staleServiceUsages: Long,
    )

    private data class TraversalStackFrame(
        val id: ServiceId,
        val unvisitedChildren: MutableSet<ServiceId>,
        val visitedChildren: MutableMap<ServiceId, Map<ResourceId, BigInteger>>,
        val parentFrame: TraversalStackFrame?
    )

}
