package ru.yandex.intranet.d.services.usage

import com.google.protobuf.util.Timestamps
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import kotlinx.coroutines.withContext
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.accounts.AccountsDao
import ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao
import ru.yandex.intranet.d.dao.providers.ProvidersDao
import ru.yandex.intranet.d.dao.resources.ResourcesDao
import ru.yandex.intranet.d.dao.resources.segmentations.ResourceSegmentationsDao
import ru.yandex.intranet.d.dao.resources.segments.ResourceSegmentsDao
import ru.yandex.intranet.d.dao.resources.types.ResourceTypesDao
import ru.yandex.intranet.d.dao.services.ServicesDao
import ru.yandex.intranet.d.dao.units.UnitsEnsemblesDao
import ru.yandex.intranet.d.dao.usage.AccountUsageDao
import ru.yandex.intranet.d.dao.usage.FolderUsageDao
import ru.yandex.intranet.d.dao.usage.ServiceUsageDao
import ru.yandex.intranet.d.dao.usage.UsageEpochsDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.kotlin.EpochSeconds
import ru.yandex.intranet.d.kotlin.ExternalAccountKey
import ru.yandex.intranet.d.kotlin.FolderId
import ru.yandex.intranet.d.kotlin.ProviderId
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.kotlin.ServiceId
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.elapsed
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel
import ru.yandex.intranet.d.model.accounts.ServiceAccountKeys
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel
import ru.yandex.intranet.d.model.services.ServiceNode
import ru.yandex.intranet.d.model.units.UnitModel
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel
import ru.yandex.intranet.d.model.usage.AccountUsageKey
import ru.yandex.intranet.d.model.usage.AccountUsageModel
import ru.yandex.intranet.d.model.usage.FolderUsageKey
import ru.yandex.intranet.d.model.usage.FolderUsageModel
import ru.yandex.intranet.d.model.usage.ServiceUsageAmounts
import ru.yandex.intranet.d.model.usage.ServiceUsageKey
import ru.yandex.intranet.d.model.usage.ServiceUsageModel
import ru.yandex.intranet.d.model.usage.UsageAmount
import ru.yandex.intranet.d.model.usage.UsageEpochKey
import ru.yandex.intranet.d.model.usage.UsageEpochModel
import ru.yandex.intranet.d.services.integration.solomon.SolomonClient
import ru.yandex.intranet.d.services.integration.solomon.SolomonResponse
import ru.yandex.intranet.d.services.settings.RuntimeSettingsService
import ru.yandex.intranet.d.services.units.UnitsComparator
import ru.yandex.intranet.d.util.dispatchers.CustomDispatcher
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import ru.yandex.monitoring.api.v3.Downsampling
import ru.yandex.monitoring.api.v3.ReadMetricsDataRequest
import ru.yandex.monitoring.api.v3.ReadMetricsDataResponse
import ru.yandex.monitoring.api.v3.Timeseries
import ru.yandex.monlib.metrics.labels.Labels
import ru.yandex.monlib.metrics.registry.MetricRegistry
import java.math.BigDecimal
import java.math.BigInteger
import java.time.Clock
import java.time.Duration
import java.time.Instant
import java.time.ZoneOffset
import java.util.*
import java.util.concurrent.atomic.AtomicLong
import java.util.concurrent.atomic.AtomicReference
import java.util.regex.Pattern

private val logger = KotlinLogging.logger {}

/**
 * YT usage sync service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class YtUsageSyncService(
    @Qualifier("messageSource") private val messages: MessageSource,
    @Qualifier("backgroundHeavyDispatcher") private val backgroundHeavyDispatcher: CustomDispatcher,
    private val usageSyncComputeDispatcher: UsageSyncComputeDispatcher,
    private val tableClient: YdbTableClient,
    private val runtimeSettingsService: RuntimeSettingsService,
    private val providersDao: ProvidersDao,
    private val resourceTypesDao: ResourceTypesDao,
    private val resourceSegmentationsDao: ResourceSegmentationsDao,
    private val resourceSegmentsDao: ResourceSegmentsDao,
    private val resourcesDao: ResourcesDao,
    private val accountsSpacesDao: AccountsSpacesDao,
    private val solomonClient: SolomonClient,
    private val servicesDao: ServicesDao,
    private val accountsDao: AccountsDao,
    private val usageEpochsDao: UsageEpochsDao,
    private val serviceUsageDao: ServiceUsageDao,
    private val folderUsageDao: FolderUsageDao,
    private val accountUsageDao: AccountUsageDao,
    private val unitsEnsemblesDao: UnitsEnsemblesDao
) {

    private val lastSuccess: AtomicReference<Instant> = AtomicReference(Instant.now())

    init {
        MetricRegistry.root().lazyGaugeInt64("cron.jobs.duration_since_last_success_millis",
            Labels.of("job", "SyncYtUsageJob")) { Duration.between(lastSuccess.get(), Instant.now()).toMillis() }
    }

    fun doSyncWithValidationMono(user: YaUserDetails, locale: Locale, clock: Clock): Mono<Result<Unit>> {
        return mono { doSyncWithValidation(user, locale, clock) }
    }

    suspend fun doSync(clock: Clock) {
        val syncParameters = meter({ prepareSyncParameters() }, "YT usage synchronization, prepare parameters")
        if (syncParameters == null) {
            logger.warn { "YT usage synchronization, skipped due to invalid settings" }
            return
        }
        if (!syncParameters.enabled) {
            return
        }
        val serviceNodes = meter({ getAllServiceSlugNodes() }, "YT usage synchronization, load services")
        val hierarchy = meter({ usageSyncComputeDispatcher.execute { prepareHierarchy(serviceNodes) } },
            "YT usage synchronization, prepare service tree")
        logger.info { "YT usage synchronization, loaded service tree with ${hierarchy.roots.size} roots " +
            "and ${hierarchy.nodesCount} services" }
        val usageInterval = prepareUsageInterval(clock)
        val clusters = syncParameters.cpuAccountsSpacesByCluster.keys
        clusters.forEach { cluster ->
            val cpuStrongResource = syncParameters.cpuStrongResourcesByCluster[cluster]
            if (cpuStrongResource == null) {
                logger.warn { "YT usage synchronization, no cpu strong resource for cluster $cluster, skipping this cluster" }
                return@forEach
            }
            val now = Instant.now(clock)
            val accounts = meter({ getAccounts(syncParameters, cluster) },
                "YT usage synchronization, load $cluster cluster accounts")
            val serviceAccounts = meter({ usageSyncComputeDispatcher.execute { prepareServiceAccounts(accounts) } },
                "YT usage synchronization, prepare service accounts for $cluster cluster, ${accounts.size} accounts total")
            val resourceEpochs = meter({ incrementAndGetEpochs(syncParameters.ytProvider.id,
                listOf(cpuStrongResource.id)) }, "YT usage synchronization, increment and get epochs for $cluster cluster")
            val accumulatedUsages = meter({ usageSyncComputeDispatcher.execute { loadClusterUsage(cluster, usageInterval,
                hierarchy, serviceAccounts, resourceEpochs[cpuStrongResource.id]!!.epoch, now, cpuStrongResource,
                syncParameters) } },
                "YT usage synchronization, loading $cluster cluster usage")
            meter({ storeAccumulatedUsage(accumulatedUsages) }, "YT usage synchronization, updating $cluster cluster usage, " +
                    "${accumulatedUsages.serviceUsages.size} service usages, " +
                    "${accumulatedUsages.folderUsages.size} folder usages and " +
                    "${accumulatedUsages.accountsUsages.size} account usages")
            val cleanupStats = meter({ cleanupPreviousEpochs(resourceEpochs) },
                "YT usage synchronization, clean up old epochs data for $cluster cluster")
            logger.info { "YT usage synchronization, $cluster cluster, " +
                "cleaned up ${cleanupStats.staleServiceUsages} service usages, " +
                "${cleanupStats.staleFolderUsages} folder usages " +
                "and ${cleanupStats.staleAccountUsages} account usages" }
        }
        lastSuccess.set(Instant.now())
    }

    private suspend fun loadClusterUsage(cluster: String,
                                         usageInterval: UsageInterval,
                                         serviceHierarchy: ServiceHierarchy,
                                         serviceAccounts: Map<ServiceId, List<ServiceAccountKeys>>,
                                         epoch: Long,
                                         now: Instant,
                                         resource: ResourceModel,
                                         syncParameters: SyncParameters): AccumulatedUsages {
        // Prepare aggregates using post-order depth first traversal
        val accountsUsages = mutableListOf<AccountUsageModel>()
        val serviceUsages = mutableListOf<ServiceUsageModel>()
        val folderUsages = mutableListOf<FolderUsageModel>()
        // More than one root may exist
        serviceHierarchy.roots.forEach { root ->
            // Traversal to accumulate aggregates
            traverseForAggregates(epoch, serviceHierarchy, now, root, usageInterval, serviceAccounts, cluster,
                accountsUsages, serviceUsages, folderUsages, resource, syncParameters)
        }
        return AccumulatedUsages(accountsUsages, serviceUsages, folderUsages)
    }

    private suspend fun storeAccumulatedUsage(usages: AccumulatedUsages) {
        dbSessionRetryable(tableClient) {
            usages.serviceUsages.chunked(250)
                .forEach { serviceUsageDao.upsertManyRetryable(rwSingleRetryableCommit(), it) }
            usages.accountsUsages.chunked(250)
                .forEach { accountUsageDao.upsertManyRetryable(rwSingleRetryableCommit(), it) }
            usages.folderUsages.chunked(250)
                .forEach { folderUsageDao.upsertManyRetryable(rwSingleRetryableCommit(), it) }
        }
    }

    private suspend fun incrementAndGetEpochs(providerId: ProviderId,
                                              resourceIds: Collection<ResourceId>): Map<ResourceId, UsageEpochModel> {
        return dbSessionRetryable(tableClient) {
            rwTxRetryable {
                val keys = resourceIds.map { UsageEpochKey(Tenants.DEFAULT_TENANT_ID, providerId, it) }
                val epochs = usageEpochsDao.getByIds(txSession, keys).associateBy { it.key }
                val incrementedEpochs = keys.map {
                    val currentEpoch = epochs.getOrDefault(it, UsageEpochModel(it, -1))
                    UsageEpochModel(it, currentEpoch.epoch + 1)
                }
                usageEpochsDao.upsertManyRetryable(txSession, incrementedEpochs)
                incrementedEpochs.associateBy { it.key.resourceId }
            }!!
        }!!
    }

    private suspend fun cleanupPreviousEpochs(epochs: Map<ResourceId, UsageEpochModel>): CleanupStats {
        val staleServiceUsagesCounter = AtomicLong(0L)
        val staleAccountUsagesCounter = AtomicLong(0L)
        val staleFolderUsagesCounter = AtomicLong(0L)
        epochs.values.forEach { epoch ->
            var nextServiceUsage = dbSessionRetryable(tableClient) {
                rwTxRetryableOptimized(
                    { serviceUsageDao.getKeysForOlderEpochsFirstPage(txSession, epoch.key.tenantId,
                        epoch.key.providerId, epoch.key.resourceId, epoch.epoch, 250) },
                    { page -> page },
                    { page ->
                        if (page.keys.isEmpty()) {
                            txSession.commitTransaction().awaitSingleOrNull()
                        } else {
                            staleServiceUsagesCounter.addAndGet(page.keys.size.toLong())
                            serviceUsageDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                        }
                        page.nextFrom
                    })
            }
            while (nextServiceUsage != null) {
                nextServiceUsage = dbSessionRetryable(tableClient) {
                    rwTxRetryableOptimized(
                        { serviceUsageDao.getKeysForOlderEpochsNextPage(txSession, nextServiceUsage!!, 250) },
                        { page -> page },
                        { page ->
                            if (page.keys.isEmpty()) {
                                txSession.commitTransaction().awaitSingleOrNull()
                            } else {
                                staleServiceUsagesCounter.addAndGet(page.keys.size.toLong())
                                serviceUsageDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                            }
                            page.nextFrom
                        })
                }
            }
            var nextAccountUsage = dbSessionRetryable(tableClient) {
                rwTxRetryableOptimized(
                    { accountUsageDao.getKeysForOlderEpochsFirstPage(txSession, epoch.key.tenantId,
                        epoch.key.resourceId, epoch.epoch, 250) },
                    { page -> page },
                    { page ->
                        if (page.keys.isEmpty()) {
                            txSession.commitTransaction().awaitSingleOrNull()
                        } else {
                            staleAccountUsagesCounter.addAndGet(page.keys.size.toLong())
                            accountUsageDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                        }
                        page.nextFrom
                    })
            }
            while (nextAccountUsage != null) {
                nextAccountUsage = dbSessionRetryable(tableClient) {
                    rwTxRetryableOptimized(
                        { accountUsageDao.getKeysForOlderEpochsNextPage(txSession,
                            nextAccountUsage!!, 250) },
                        { page -> page },
                        { page ->
                            if (page.keys.isEmpty()) {
                                txSession.commitTransaction().awaitSingleOrNull()
                            } else {
                                staleAccountUsagesCounter.addAndGet(page.keys.size.toLong())
                                accountUsageDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                            }
                            page.nextFrom
                        })
                }
            }
            var nextFolderUsage = dbSessionRetryable(tableClient) {
                rwTxRetryableOptimized(
                    { folderUsageDao.getKeysForOlderEpochsFirstPage(txSession, epoch.key.tenantId,
                        epoch.key.resourceId, epoch.epoch, 250) },
                    { page -> page },
                    { page ->
                        if (page.keys.isEmpty()) {
                            txSession.commitTransaction().awaitSingleOrNull()
                        } else {
                            staleFolderUsagesCounter.addAndGet(page.keys.size.toLong())
                            folderUsageDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                        }
                        page.nextFrom
                    })
            }
            while (nextFolderUsage != null) {
                nextFolderUsage = dbSessionRetryable(tableClient) {
                    rwTxRetryableOptimized(
                        { folderUsageDao.getKeysForOlderEpochsNextPage(txSession,
                            nextFolderUsage!!, 250) },
                        { page -> page },
                        { page ->
                            if (page.keys.isEmpty()) {
                                txSession.commitTransaction().awaitSingleOrNull()
                            } else {
                                staleFolderUsagesCounter.addAndGet(page.keys.size.toLong())
                                folderUsageDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                            }
                            page.nextFrom
                        })
                }
            }
        }
        return CleanupStats(staleServiceUsagesCounter.get(), staleAccountUsagesCounter.get(), staleFolderUsagesCounter.get())
    }

    private suspend fun doSyncWithValidation(user: YaUserDetails, locale: Locale, clock: Clock): Result<Unit> = binding {
        validatePermissions(user, locale).bind()
        launchInBackground { doSync(clock) }
        Result.success(Unit)
    }

    private fun launchInBackground(block: suspend () -> Unit) {
        try {
            backgroundHeavyDispatcher.launch {
                try {
                    block()
                } catch (e: Exception) {
                    logger.error(e) { "Error during YT usage sync" }
                    if (e is CancellationException) {
                        throw e
                    }
                }
            }
        } catch (e: Exception) {
            logger.error(e) { "Error during YT usage sync" }
        }
    }

    private fun validatePermissions(user: YaUserDetails, locale: Locale): Result<Unit> {
        if (user.user.isEmpty || !user.user.get().dAdmin) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.forbidden(messages
                .getMessage("errors.access.denied", null, locale))).build())
        }
        return Result.success(Unit)
    }

    private suspend fun getAllServiceSlugNodes(): List<ServiceNode> {
        return dbSessionRetryable(tableClient) {
            servicesDao.getAllServiceNodes(session).awaitSingle()
        }!!
    }

    private fun prepareHierarchy(nodes: List<ServiceNode>): ServiceHierarchy {
        val roots = mutableListOf<ServiceId>()
        val childrenByParent = mutableMapOf<ServiceId, MutableSet<ServiceId>>()
        nodes.forEach {
            if (it.parentId == null) {
                roots.add(it.id)
            } else {
                childrenByParent.computeIfAbsent(it.parentId) { mutableSetOf() }.add(it.id)
            }
        }
        return ServiceHierarchy(roots, childrenByParent, nodes.size)
    }

    private suspend fun getAccounts(syncParameters: SyncParameters, cluster: String): List<ServiceAccountKeys> {
        val accountsSpaceId = syncParameters.cpuAccountsSpacesByCluster[cluster]?.id ?: return emptyList()
        val providerId = syncParameters.ytProvider.id
        return dbSessionRetryable(tableClient) {
            accountsDao.getAllNonDeletedServiceAccountKeysByProviderAccountsSpaces(roStaleSingleRetryableCommit(),
                Tenants.DEFAULT_TENANT_ID, providerId, setOf(accountsSpaceId)).awaitSingle()
        }!!
    }

    private fun prepareServiceAccounts(accounts: List<ServiceAccountKeys>): Map<ServiceId, List<ServiceAccountKeys>> {
        val result = mutableMapOf<ServiceId, MutableList<ServiceAccountKeys>>()
        accounts.forEach { account ->
            result.computeIfAbsent(account.serviceId) { mutableListOf() }.add(account)
        }
        return result
    }

    private suspend fun loadAccountsTimeSeriesNoDownSampling(cluster: String,
                                                             accounts: Collection<ExternalAccountKey>,
                                                             from: Instant,
                                                             to: Instant): Map<ExternalAccountKey, Map<EpochSeconds, BigDecimal>> {
        if (accounts.isEmpty()) {
            return emptyMap()
        }
        val query = if (accounts.size == 1) {
            val account = accounts.first()
            "{project='yt', cluster='$cluster', service='pools', pool='$account', " +
                "sensor='resource_usage.cpu', tree='physical'}"
        } else {
            val accountsPattern = accounts.joinToString(separator = "|", prefix = "^(", postfix = ")$") { Pattern.quote(it) }
            "{project='yt', cluster='$cluster', service='pools', pool=~'$accountsPattern', " +
                "sensor='resource_usage.cpu', tree='physical'}"
        }
        val queryResult = meter({ withContext(Dispatchers.Default) { solomonClient
            .readData(ReadMetricsDataRequest.newBuilder()
                .setProjectId("yt")
                .setDownsampling(Downsampling.newBuilder()
                    .setDisabled(true)
                    .build())
                .setFromTime(Timestamps.fromMillis(from.toEpochMilli()))
                .setToTime(Timestamps.fromMillis(to.toEpochMilli()))
                .setForceZoneId("Z")
                .setUseOldFormat(true)
                .setQuery(query)
                .build()) } }, "YT usage synchronization, Solomon request $query")
        when (queryResult) {
            is SolomonResponse.Success<ReadMetricsDataResponse> -> {
                val accountsSet = accounts.toSet()
                val solomonData = queryResult.value
                val resultMap = mutableMapOf<ExternalAccountKey, Map<EpochSeconds, BigDecimal>>()
                if (solomonData.hasTimeseriesVector()) {
                    solomonData.timeseriesVector.valuesList.forEach { timeSeries ->
                        processSolomonTimeSeries(timeSeries, resultMap, accountsSet)
                    }
                } else if (solomonData.hasTimeseries()) {
                    processSolomonTimeSeries(solomonData.timeseries, resultMap, accountsSet)
                } else {
                    throw RuntimeException("No time series in Solomon response")
                }
                return resultMap
            }
            is SolomonResponse.Error<ReadMetricsDataResponse> -> {
                logger.error { "Failed Solomon query $query: $queryResult" }
                throw RuntimeException("Failed Solomon query $query: $queryResult")
            }
            is SolomonResponse.Failure<ReadMetricsDataResponse> -> {
                logger.error(queryResult.error) { "Failed Solomon query $query" }
                throw queryResult.error
            }
        }
    }

    private fun prepareUsageInterval(clock: Clock): UsageInterval {
        val now = Instant.now(clock)
        val todayStartOfDayUtc = now.atZone(ZoneOffset.UTC).toLocalDate().atStartOfDay(ZoneOffset.UTC).toInstant()
        val weekAgoStartOfDayUtc = now.atZone(ZoneOffset.UTC).toLocalDate().minusDays(7)
            .atStartOfDay(ZoneOffset.UTC).toInstant()
        return UsageInterval(weekAgoStartOfDayUtc, todayStartOfDayUtc)
    }

    private fun processSolomonTimeSeries(timeSeries: Timeseries,
                                         result: MutableMap<ExternalAccountKey, Map<EpochSeconds, BigDecimal>>,
                                         accounts: Set<ExternalAccountKey>) {
        val timestamps = timeSeries.timestampsList
        val labels = timeSeries.labelsMap
        val account = labels["pool"] ?: throw RuntimeException("Time series without pool label in Solomon response")
        if (!accounts.contains(account)) {
            throw RuntimeException("Unexpected account $account in solomon response, expecting on of: $accounts")
        }
        val usageByTimestamp = mutableMapOf<EpochSeconds, BigDecimal>()
        if (timeSeries.hasDoubleValues()) {
            timeSeries.doubleValues.valuesList.forEachIndexed { index, value ->
                if (index >= timestamps.size) {
                    throw RuntimeException("Time series timestamps list is too short")
                }
                val timestamp = timestamps[index]
                if (value.isFinite()) {
                    usageByTimestamp[Timestamps.toSeconds(timestamp)] = BigDecimal.valueOf(value)
                }
            }
        } else if (timeSeries.hasInt64Values()) {
            timeSeries.int64Values.valuesList.forEachIndexed { index, value ->
                if (index >= timestamps.size) {
                    throw RuntimeException("Time series timestamps list is too short")
                }
                val timestamp = timestamps[index]
                usageByTimestamp[Timestamps.toSeconds(timestamp)] = BigDecimal.valueOf(value)
            }
        } else {
            throw RuntimeException("No data in time series in Solomon response")
        }
        result[account] = usageByTimestamp
    }

    private fun processAccountsUsages(timeSeriesPerAccount: Map<ExternalAccountKey, Map<EpochSeconds, BigDecimal>>,
                                      now: Instant,
                                      resource: ResourceModel,
                                      epoch: Long,
                                      parameters: SyncParameters,
                                      accountsByExternalKey: Map<ExternalAccountKey, ServiceAccountKeys>,
                                      accountUsages: MutableList<AccountUsageModel>): GroupedAccountUsages {
        val aggregatedTimeSeries = mutableMapOf<EpochSeconds, BigInteger>()
        val timeSeriesPerFolder = mutableMapOf<FolderId, Map<EpochSeconds, BigInteger>>()
        timeSeriesPerAccount.forEach { (externalAccountKey, accountTimeSeries) ->
            if (accountTimeSeries.isEmpty()) {
                return@forEach
            }
            val convertedTimeSeries = convertTimeSeries(accountTimeSeries,
                parameters.cpuMetricUnit, parameters.cpuStrongBaseUnit)
            if (convertedTimeSeries.values.all { it.compareTo(BigInteger.ZERO) == 0 }) {
                return@forEach
            }
            val account = accountsByExternalKey[externalAccountKey]
            if (account != null) {
                accountUsages.add(AccountUsageModel(
                    key = AccountUsageKey(
                        tenantId = account.tenantId,
                        accountId = account.accountId,
                        resourceId = resource.id
                    ),
                    lastUpdate = now,
                    epoch = epoch,
                    ownUsage = prepareUsageAmount(convertedTimeSeries, parameters)
                ))
                val folderId = account.folderId
                val folderAccumulator = timeSeriesPerFolder.getOrDefault(folderId, mapOf())
                timeSeriesPerFolder[folderId] = sumTimeSeries(folderAccumulator, convertedTimeSeries)
            }
            accumulateTimeSeries(aggregatedTimeSeries, convertedTimeSeries)
        }
        return GroupedAccountUsages(aggregatedTimeSeries, timeSeriesPerFolder)
    }

    private fun prepareUsageAmount(timeSeries: Map<EpochSeconds, BigInteger>, parameters: SyncParameters): UsageAmount {
        val average = mean(timeSeries.values)
        val minMedianMax = minMedianMax(timeSeries.values)
        val variance = variance(timeSeries.values, average)
        val accumulated = accumulate(timeSeries, parameters.cpuMetricGridSpacingSeconds)
        val histogram = histogram(timeSeries.values, minMedianMax.first, minMedianMax.third)
        val sortedEntries = timeSeries.entries.sortedBy { it.key }
        val valuesX = sortedEntries.map { it.key }
        val valuesY = sortedEntries.map { it.value }
        return UsageAmount(
            value = null,
            average = roundToIntegerHalfUp(average),
            min = minMedianMax.first,
            max = minMedianMax.third,
            median = roundToIntegerHalfUp(minMedianMax.second),
            variance = roundToIntegerHalfUp(variance),
            accumulated = roundToIntegerHalfUp(accumulated.first),
            accumulatedDuration = accumulated.second,
            histogram = histogram,
            values = null,
            valuesX = valuesX,
            valuesY = valuesY,
            unused = null
        )
    }

    private suspend fun traverseForAggregates(epoch: Long,
                                              hierarchy: ServiceHierarchy,
                                              now: Instant,
                                              root: ServiceId,
                                              usageInterval: UsageInterval,
                                              serviceAccounts: Map<ServiceId, List<ServiceAccountKeys>>,
                                              cluster: String,
                                              accountUsage: MutableList<AccountUsageModel>,
                                              serviceUsage: MutableList<ServiceUsageModel>,
                                              folderUsage: MutableList<FolderUsageModel>,
                                              resource: ResourceModel,
                                              syncParameters: SyncParameters) {
        val stack = ArrayDeque<TraversalStackFrame>()
        stack.push(
            TraversalStackFrame(
                root, (hierarchy.childrenByParent[root] ?: setOf()).toMutableSet(), mutableMapOf(), null
            )
        )
        var counter = 0
        val failsafeLimit = hierarchy.nodesCount.toLong() * 3
        while (!stack.isEmpty()) {
            val currentStackFrame = stack.peek()
            if (currentStackFrame.unvisitedChildren.isEmpty()) {
                // Visit this node
                val currentServiceAccounts = serviceAccounts[currentStackFrame.id] ?: listOf()
                val currentServiceAccountsExternalKeys = currentServiceAccounts.mapNotNull { it.externalAccountKey }
                val accountsByExternalKey = currentServiceAccounts.asSequence()
                    .filter { it.externalAccountKey != null }.associateBy { it.externalAccountKey!! }
                val accountsTimeSeries = loadAccountsTimeSeriesNoDownSampling(cluster, currentServiceAccountsExternalKeys,
                    usageInterval.from, usageInterval.to)
                val nodeAggregates = if (currentStackFrame.visitedChildren.isEmpty()) {
                    // Leaf node
                    aggregateLeafServiceResources(accountsTimeSeries, accountsByExternalKey, epoch,
                        currentStackFrame.id, now, resource, syncParameters, accountUsage, serviceUsage, folderUsage)
                } else {
                    // Non-leaf node
                    aggregateServiceResources(accountsTimeSeries, accountsByExternalKey,
                        currentStackFrame.visitedChildren.values, epoch, currentStackFrame.id, now,
                        resource, syncParameters, accountUsage, serviceUsage, folderUsage)
                }
                if (currentStackFrame.parentFrame != null) {
                    // Non-root node
                    currentStackFrame.parentFrame.unvisitedChildren.remove(currentStackFrame.id)
                    currentStackFrame.parentFrame.visitedChildren[currentStackFrame.id] = nodeAggregates
                }
                stack.pop()
            } else {
                // Go deeper to next child
                val nextChildNode = currentStackFrame.unvisitedChildren.first()
                stack.push(
                    TraversalStackFrame(nextChildNode,
                        (hierarchy.childrenByParent[nextChildNode] ?: setOf()).toMutableSet(), mutableMapOf(),
                        currentStackFrame
                    )
                )
            }
            counter++
            if (counter > failsafeLimit) {
                throw IllegalStateException("Too many tree traversal iterations")
            }
        }
    }

    private fun aggregateServiceResources(timeSeries: Map<ExternalAccountKey, Map<EpochSeconds, BigDecimal>>,
                                          accountsByExternalKey: Map<ExternalAccountKey, ServiceAccountKeys>,
                                          childAggregates: Collection<Map<EpochSeconds, BigInteger>>,
                                          epoch: Long,
                                          serviceId: ServiceId,
                                          now: Instant,
                                          resource: ResourceModel,
                                          syncParameters: SyncParameters,
                                          accountUsages: MutableList<AccountUsageModel>,
                                          serviceUsages: MutableList<ServiceUsageModel>,
                                          folderUsages: MutableList<FolderUsageModel>
    ): Map<EpochSeconds, BigInteger> {
        if (timeSeries.isEmpty() && childAggregates.isEmpty()) {
            return mapOf()
        }
        val aggregatedTimeSeries = processAccountsUsages(timeSeries, now, resource, epoch, syncParameters,
            accountsByExternalKey, accountUsages)
        val serviceOwnUsage = if (aggregatedTimeSeries.serviceUsage.isEmpty()
            || aggregatedTimeSeries.serviceUsage.values.all { it.compareTo(BigInteger.ZERO) == 0 }) {
            null
        } else {
            prepareUsageAmount(aggregatedTimeSeries.serviceUsage, syncParameters)
        }
        val serviceSubtreeTimeSeries = sumTimeSeries(childAggregates)
        val serviceSubtreeUsage = if (serviceSubtreeTimeSeries.isEmpty()
            || serviceSubtreeTimeSeries.values.all { it.compareTo(BigInteger.ZERO) == 0 }) {
            null
        } else {
            prepareUsageAmount(serviceSubtreeTimeSeries, syncParameters)
        }
        val serviceTotalTimeSeries = sumTimeSeries(aggregatedTimeSeries.serviceUsage, serviceSubtreeTimeSeries)
        val serviceTotalUsage = if (serviceSubtreeUsage == null || serviceTotalTimeSeries.isEmpty()
            || serviceTotalTimeSeries.values.all { it.compareTo(BigInteger.ZERO) == 0 }) {
            null
        } else {
            prepareUsageAmount(serviceTotalTimeSeries, syncParameters)
        }
        if (serviceOwnUsage != null || serviceSubtreeUsage != null || serviceTotalUsage != null) {
            serviceUsages.add(ServiceUsageModel(
                key = ServiceUsageKey(
                    tenantId = resource.tenantId,
                    serviceId = serviceId,
                    providerId = syncParameters.ytProvider.id,
                    resourceId = resource.id
                ),
                lastUpdate = now,
                epoch = epoch,
                usage = ServiceUsageAmounts(
                    own = serviceOwnUsage,
                    subtree = serviceSubtreeUsage,
                    total = serviceTotalUsage
                )
            ))
        }
        aggregateFoldersUsage(aggregatedTimeSeries.folderUsages, syncParameters, folderUsages, resource, now, epoch)
        return serviceTotalTimeSeries
    }

    private fun aggregateLeafServiceResources(timeSeries: Map<ExternalAccountKey, Map<EpochSeconds, BigDecimal>>,
                                              accountsByExternalKey: Map<ExternalAccountKey, ServiceAccountKeys>,
                                              epoch: Long,
                                              serviceId: ServiceId,
                                              now: Instant,
                                              resource: ResourceModel,
                                              syncParameters: SyncParameters,
                                              accountUsages: MutableList<AccountUsageModel>,
                                              serviceUsages: MutableList<ServiceUsageModel>,
                                              folderUsages: MutableList<FolderUsageModel>
    ): Map<EpochSeconds, BigInteger> {
        if (timeSeries.isEmpty()) {
            return mapOf()
        }
        val aggregatedTimeSeries = processAccountsUsages(timeSeries, now, resource, epoch, syncParameters,
            accountsByExternalKey, accountUsages)
        if (aggregatedTimeSeries.serviceUsage.isNotEmpty()
            && !aggregatedTimeSeries.serviceUsage.values.all { it.compareTo(BigInteger.ZERO) == 0 }) {
            val serviceOwnUsage = prepareUsageAmount(aggregatedTimeSeries.serviceUsage, syncParameters)
            serviceUsages.add(ServiceUsageModel(
                key = ServiceUsageKey(
                    tenantId = resource.tenantId,
                    serviceId = serviceId,
                    providerId = syncParameters.ytProvider.id,
                    resourceId = resource.id
                ),
                lastUpdate = now,
                epoch = epoch,
                usage = ServiceUsageAmounts(
                    own = serviceOwnUsage,
                    subtree = null,
                    total = null
                )
            ))
        }
        aggregateFoldersUsage(aggregatedTimeSeries.folderUsages, syncParameters, folderUsages, resource, now, epoch)
        return aggregatedTimeSeries.serviceUsage
    }

    private fun aggregateFoldersUsage(timeSeriesByFolder: Map<FolderId, Map<EpochSeconds, BigInteger>>,
                                      syncParameters: SyncParameters,
                                      folderUsages: MutableList<FolderUsageModel>,
                                      resource: ResourceModel,
                                      now: Instant,
                                      epoch: Long) {
        if (timeSeriesByFolder.isNotEmpty()) {
            timeSeriesByFolder.forEach { (folderId, timeSeries) ->
                if (timeSeries.isNotEmpty() && !timeSeries.values.all { it.compareTo(BigInteger.ZERO) == 0 }) {
                    val folderOwnUsage = prepareUsageAmount(timeSeries, syncParameters)
                    folderUsages.add(FolderUsageModel(
                            key = FolderUsageKey(
                                tenantId = resource.tenantId,
                                folderId = folderId,
                                resourceId = resource.id
                            ),
                            lastUpdate = now,
                            epoch = epoch,
                            ownUsage = folderOwnUsage
                    ))
                }
            }
        }
    }

    private suspend fun prepareSyncParameters(): SyncParameters? {
        val knownProviders = runtimeSettingsService.getKnownProvidersImmediate(Tenants.DEFAULT_TENANT_ID)
        if (knownProviders?.yt == null) {
            logger.warn { "YT is not defined as known provider, YT usage synchronization will be skipped" }
            return null
        }
        val ytUsageSyncSettings = runtimeSettingsService.getYtUsageSyncSettingsImmediate(Tenants.DEFAULT_TENANT_ID)
        if (ytUsageSyncSettings?.cpuStrongResourceType == null || ytUsageSyncSettings.poolTreeSegmentation == null
            || ytUsageSyncSettings.physicalPoolTreeSegment == null || ytUsageSyncSettings.clusterSegmentation == null
            || ytUsageSyncSettings.scopeSegmentation == null || ytUsageSyncSettings.computeScopeSegment == null
            || ytUsageSyncSettings.syncEnabled == null || ytUsageSyncSettings.cpuMetricGridSpacingSeconds == null
            || ytUsageSyncSettings.cpuMetricUnitsEnsemble == null || ytUsageSyncSettings.cpuMetricUnit == null) {
            logger.warn { "YT usage sync settings are undefined or incomplete, YT usage synchronization will be skipped" }
            return null
        }
        val ytProviderO = dbSessionRetryable(tableClient) {
            providersDao.getById(roStaleSingleRetryableCommit(), knownProviders.yt,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (ytProviderO.isEmpty || ytProviderO.get().isDeleted) {
            logger.warn { "YT provider not found, YT usage synchronization will be skipped" }
            return null
        }
        val ytProvider = ytProviderO.get()
        val cpuStrongResourceTypeO = dbSessionRetryable(tableClient) {
            resourceTypesDao.getById(roStaleSingleRetryableCommit(), ytUsageSyncSettings.cpuStrongResourceType,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (cpuStrongResourceTypeO.isEmpty || cpuStrongResourceTypeO.get().isDeleted) {
            logger.warn { "YT strong_cpu resource type not found, YT usage synchronization will be skipped" }
            return null
        }
        val cpuStrongResourceType = cpuStrongResourceTypeO.get()
        if (cpuStrongResourceType.providerId != ytProvider.id) {
            logger.warn { "Inconsistent strong_cpu resource type, YT usage synchronization will be skipped" }
            return null
        }
        val segmentationIds = listOf(Tuples.of(ytUsageSyncSettings.clusterSegmentation, Tenants.DEFAULT_TENANT_ID),
            Tuples.of(ytUsageSyncSettings.scopeSegmentation, Tenants.DEFAULT_TENANT_ID),
            Tuples.of(ytUsageSyncSettings.poolTreeSegmentation, Tenants.DEFAULT_TENANT_ID))
        val segmentations = dbSessionRetryable(tableClient) {
            resourceSegmentationsDao.getByIds(roStaleSingleRetryableCommit(), segmentationIds).awaitSingle()
        }!!.associateBy { it.id }
        if (segmentations[ytUsageSyncSettings.clusterSegmentation]?.isDeleted != false
            || segmentations[ytUsageSyncSettings.scopeSegmentation]?.isDeleted != false
            || segmentations[ytUsageSyncSettings.poolTreeSegmentation]?.isDeleted != false) {
            logger.warn { "YT resource segmentations not found, YT usage synchronization will be skipped" }
            return null
        }
        val clusterSegmentation = segmentations[ytUsageSyncSettings.clusterSegmentation]!!
        val scopeSegmentation = segmentations[ytUsageSyncSettings.scopeSegmentation]!!
        val poolTreeSegmentation = segmentations[ytUsageSyncSettings.poolTreeSegmentation]!!
        if (clusterSegmentation.providerId != ytProvider.id || scopeSegmentation.providerId != ytProvider.id
            || poolTreeSegmentation.providerId != ytProvider.id) {
            logger.warn { "Inconsistent YT resource segmentations, YT usage synchronization will be skipped" }
            return null
        }
        val segmentIds = listOf(Tuples.of(ytUsageSyncSettings.computeScopeSegment, Tenants.DEFAULT_TENANT_ID),
            Tuples.of(ytUsageSyncSettings.physicalPoolTreeSegment, Tenants.DEFAULT_TENANT_ID))
        val segments = dbSessionRetryable(tableClient) {
            resourceSegmentsDao.getByIds(roStaleSingleRetryableCommit(), segmentIds).awaitSingle()
        }!!.associateBy { it.id }
        if (segments[ytUsageSyncSettings.computeScopeSegment]?.isDeleted != false
            || segments[ytUsageSyncSettings.physicalPoolTreeSegment]?.isDeleted != false) {
            logger.warn { "YT resource segments not found, YT usage synchronization will be skipped" }
            return null
        }
        val computeScopeSegment = segments[ytUsageSyncSettings.computeScopeSegment]!!
        val physicalPoolTreeSegment = segments[ytUsageSyncSettings.physicalPoolTreeSegment]!!
        if (computeScopeSegment.segmentationId != scopeSegmentation.id
            || physicalPoolTreeSegment.segmentationId != poolTreeSegmentation.id) {
            logger.warn { "Inconsistent YT resource segments, YT usage synchronization will be skipped" }
            return null
        }
        val ytResources = dbSessionRetryable(tableClient) {
            resourcesDao.getAllByProviderResourceType(roStaleSingleRetryableCommit(), ytProvider.id,
                cpuStrongResourceType.id, Tenants.DEFAULT_TENANT_ID, false).awaitSingle()
        }!!.filter { !it.isDeleted }
        val computeScopeKey = ResourceSegmentSettingsModel(scopeSegmentation.id, computeScopeSegment.id)
        val physicalPoolTreeKey = ResourceSegmentSettingsModel(poolTreeSegmentation.id, physicalPoolTreeSegment.id)
        val ytCpuStrongResources = ytResources.filter { resource ->
            resource.segments.contains(computeScopeKey) && resource.segments.contains(physicalPoolTreeKey)
            && resource.segments.any { it.segmentationId == clusterSegmentation.id }
        }
        if (ytCpuStrongResources.isEmpty()) {
            logger.warn { "No strong_cpu resources in YT, YT usage synchronization will be skipped" }
            return null
        }
        val resourceClusterSegmentIds = ytCpuStrongResources.flatMap { r -> r.segments
            .filter { s -> s.segmentationId == clusterSegmentation.id }.map { s -> s.segmentId } }.distinct()
        val resourceClusterSegments = dbSessionRetryable(tableClient) {
            resourceClusterSegmentIds.chunked(1000).map { p -> resourceSegmentsDao.getByIds(
                roStaleSingleRetryableCommit(), p.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }
                .flatten()
        }!!.associateBy { it.id }
        val ytCpuStrongResourcesByCluster = ytCpuStrongResources.associateBy { r -> resourceClusterSegments[r.segments
            .first { s -> s.segmentationId == clusterSegmentation.id }.segmentId]!!.key }
        val ytAccountsSpaces = dbSessionRetryable(tableClient) {
            accountsSpacesDao.getAllByProvider(roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID,
                ytProvider.id).awaitSingle()
        }!!.get().filter { !it.isDeleted }
        val ytCpuAccountsSpaces = ytAccountsSpaces.filter { space ->
            space.segments.contains(computeScopeKey) && space.segments.contains(physicalPoolTreeKey)
            && space.segments.any { it.segmentationId == clusterSegmentation.id }
        }
        if (ytCpuAccountsSpaces.isEmpty()) {
            logger.warn { "No CPU accounts spaces in YT, YT usage synchronization will be skipped" }
            return null
        }
        val spaceClusterSegmentIds = ytCpuAccountsSpaces.flatMap { r -> r.segments
            .filter { s -> s.segmentationId == clusterSegmentation.id }.map { s -> s.segmentId } }.distinct()
        val spaceClusterSegments = dbSessionRetryable(tableClient) {
            spaceClusterSegmentIds.chunked(1000).map { p -> resourceSegmentsDao.getByIds(
                roStaleSingleRetryableCommit(), p.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }
                .flatten()
        }!!.associateBy { it.id }
        val ytCpuAccountsSpacesByCluster = ytCpuAccountsSpaces.associateBy { r -> spaceClusterSegments[r.segments
            .first { s -> s.segmentationId == clusterSegmentation.id }.segmentId]!!.key }
        val cpuMetricUnitsEnsembleO = dbSessionRetryable(tableClient) {
            unitsEnsemblesDao.getById(roStaleSingleRetryableCommit(), ytUsageSyncSettings.cpuMetricUnitsEnsemble,
                Tenants.DEFAULT_TENANT_ID).awaitSingle()
        }!!
        if (cpuMetricUnitsEnsembleO.isEmpty || cpuMetricUnitsEnsembleO.get().isDeleted) {
            logger.warn { "CPU metric units ensemble not found, YT usage synchronization will be skipped" }
            return null
        }
        val cpuMetricUnitsEnsemble = cpuMetricUnitsEnsembleO.get()
        if (cpuMetricUnitsEnsemble.id != cpuStrongResourceType.unitsEnsembleId) {
            logger.warn { "Inconsistent CPU metric units ensemble, YT usage synchronization will be skipped" }
            return null
        }
        val cpuMetricUnitO = cpuMetricUnitsEnsemble.unitById(ytUsageSyncSettings.cpuMetricUnit)
        if (cpuMetricUnitO.isEmpty || cpuMetricUnitO.get().isDeleted) {
            logger.warn { "CPU metric unit not found, YT usage synchronization will be skipped" }
            return null
        }
        val cpuMetricUnit = cpuMetricUnitO.get()
        if (ytUsageSyncSettings.cpuMetricGridSpacingSeconds <= 0) {
            logger.warn { "CPU metric grid spacing is invalid, YT usage synchronization will be skipped" }
            return null
        }
        val cpuStrongBaseUnit = if (cpuStrongResourceType.baseUnitId != null) {
            cpuMetricUnitsEnsemble.unitById(cpuStrongResourceType.baseUnitId).orElseThrow()
        } else {
            UnitsComparator.getBaseUnit(cpuMetricUnitsEnsemble)
        }
        return SyncParameters(ytCpuStrongResourcesByCluster, ytCpuAccountsSpacesByCluster, ytProvider,
            ytUsageSyncSettings.syncEnabled, cpuMetricUnitsEnsemble, cpuMetricUnit,
            ytUsageSyncSettings.cpuMetricGridSpacingSeconds, cpuStrongBaseUnit)
    }

    private inline fun <T> meter(statement: () -> T, label: String): T {
        return elapsed(statement)
            { millis, success -> logger.info { "$label: duration = $millis ms, success = $success" } }
    }

    private data class SyncParameters(
        // Resource type = strong_cpu, scope = compute, pool tree = physical, map by cluster key
        val cpuStrongResourcesByCluster: Map<String, ResourceModel>,
        // Scope = compute, pool tree = physical, map by cluster key
        val cpuAccountsSpacesByCluster: Map<String, AccountSpaceModel>,
        val ytProvider: ProviderModel,
        val enabled: Boolean,
        val cpuMetricUnitsEnsemble: UnitsEnsembleModel,
        val cpuMetricUnit: UnitModel,
        val cpuMetricGridSpacingSeconds: Long,
        val cpuStrongBaseUnit: UnitModel
    )

    private data class ServiceHierarchy(
        val roots: List<ServiceId>,
        val childrenByParent: Map<ServiceId, Set<ServiceId>>,
        val nodesCount: Int
    )

    private data class UsageInterval(
        val from: Instant,
        val to: Instant
    )

    private data class AccumulatedUsages(
        val accountsUsages: List<AccountUsageModel>,
        val serviceUsages: List<ServiceUsageModel>,
        val folderUsages: List<FolderUsageModel>
    )

    private data class CleanupStats(
        val staleServiceUsages: Long,
        val staleAccountUsages: Long,
        val staleFolderUsages: Long
    )

    private data class TraversalStackFrame(
        val id: ServiceId,
        val unvisitedChildren: MutableSet<ServiceId>,
        val visitedChildren: MutableMap<ServiceId, Map<EpochSeconds, BigInteger>>,
        val parentFrame: TraversalStackFrame?
    )

    private data class GroupedAccountUsages(
        val serviceUsage: Map<EpochSeconds, BigInteger>,
        val folderUsages: Map<FolderId, Map<EpochSeconds, BigInteger>>
    )

}
