package ru.yandex.intranet.d.services.validators;

import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.services.ServiceReadOnlyState;
import ru.yandex.intranet.d.model.services.ServiceState;
import ru.yandex.intranet.d.model.services.ServiceWithStatesModel;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.ResultTx;
import ru.yandex.intranet.d.util.result.TypedError;

/**
 * ABC service ID validator.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 */
@Component
public class AbcServiceValidator {
    public static final Set<ServiceState> ALLOWED_SERVICE_STATES = Set.of(
            ServiceState.DEVELOP,
            ServiceState.SUPPORTED,
            ServiceState.NEED_INFO
    );
    public static final Set<ServiceState> ALLOWED_SERVICE_STATES_FOR_QUOTA_EXPORT = Set.of(
            ServiceState.DEVELOP,
            ServiceState.SUPPORTED,
            ServiceState.NEED_INFO,
            ServiceState.CLOSED,
            ServiceState.DELETED
    );
    public static final Set<ServiceReadOnlyState> ALLOWED_SERVICE_READONLY_STATES = Set.of(
            ServiceReadOnlyState.MOVING,
            ServiceReadOnlyState.RENAMING
    );
    public static final Set<ServiceReadOnlyState> ALLOWED_SERVICE_READONLY_STATES_FOR_QUOTA_EXPORT = Set.of(
            ServiceReadOnlyState.MOVING,
            ServiceReadOnlyState.RENAMING,
            ServiceReadOnlyState.DELETING,
            ServiceReadOnlyState.CLOSING
    );

    private final MessageSource messages;
    private final ServicesDao servicesDao;

    public AbcServiceValidator(@Qualifier("messageSource") MessageSource messages, ServicesDao servicesDao) {
        this.messages = messages;
        this.servicesDao = servicesDao;
    }

    @SuppressWarnings("OptionalUsedAsFieldOrParameterType")
    public Mono<ResultTx<ServiceWithStatesModel>> validateAbcService(
            Optional<Long> abcServiceId, Locale locale, YdbTxSession session, String fieldKey) {
        return validateAbcService(abcServiceId, locale, session, fieldKey, ALLOWED_SERVICE_STATES_FOR_QUOTA_EXPORT,
                ALLOWED_SERVICE_READONLY_STATES_FOR_QUOTA_EXPORT, true);
    }

    @SuppressWarnings("OptionalUsedAsFieldOrParameterType")
    public Mono<ResultTx<ServiceWithStatesModel>> validateAbcService(
            Optional<Long> abcServiceId, Locale locale, YdbTxSession session, String fieldKey,
            Set<ServiceState> allowedServiceStates, Set<ServiceReadOnlyState> allowedServiceReadOnlyStates,
            boolean allowedNonExportable) {
        if (abcServiceId.isEmpty()) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
                    .build();
            return Mono.just(ResultTx.failure(error, null));
        } else if (abcServiceId.get() > Integer.MAX_VALUE
                || abcServiceId.get() < Integer.MIN_VALUE) {
            ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.number.out.of.range", null, locale)))
                    .build();
            return Mono.just(ResultTx.failure(error, null));
        }
        return servicesDao.getServiceStatesById(session, abcServiceId.get()).map(serviceTx -> {
            String transactionId = serviceTx.getTransactionId();
            Optional<ServiceWithStatesModel> service = serviceTx.get();
            if (service.isEmpty()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                        .getMessage("errors.service.not.found", null, locale)))
                        .build();
                return ResultTx.failure(error, transactionId);
            }
            if (!allowedNonExportable && !service.get().isExportable()) {
                ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.notFound(messages
                        .getMessage("errors.service.is.non.exportable", null, locale)))
                        .build();
                return ResultTx.failure(error, transactionId);
            }
            boolean validState = allowedServiceStates.contains(service.get().getState());
            boolean validReadOnlyState = service.get().getReadOnlyState() == null ||
                            allowedServiceReadOnlyStates.contains(service.get().getReadOnlyState());
            if (!validState || !validReadOnlyState) {
                ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.notFound(messages
                        .getMessage("errors.service.bad.status", null, locale)))
                        .build();
                return ResultTx.failure(error, transactionId);
            }
            return ResultTx.success(service.get(), transactionId);
        });
    }

    public Mono<Result<List<ServiceWithStatesModel>>> validateAbcServices(
            List<Long> abcServiceIds, Locale locale, YdbTxSession session, Set<ServiceState> allowedServiceStates,
            Set<ServiceReadOnlyState> allowedServiceReadOnlyStates, boolean allowedNonExportable
    ) {
        return Flux.fromIterable(Lists.partition(abcServiceIds, 500))
                .concatMap(v -> servicesDao.getServiceStatesByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()))
                .map(services -> {
                    for (ServiceWithStatesModel service : services) {
                        if (!allowedNonExportable && !service.isExportable()) {
                            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                                    .getMessage("errors.service.is.non.exportable", null, locale)))
                                    .build();
                            return Result.failure(error);
                        }
                        boolean validState = allowedServiceStates.contains(service.getState());
                        boolean validReadOnlyState = service.getReadOnlyState() == null ||
                                allowedServiceReadOnlyStates.contains(service.getReadOnlyState());
                        if (!validState || !validReadOnlyState) {
                            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                                    .getMessage("errors.service.bad.status", null, locale)))
                                    .build();
                            return Result.failure(error);
                        }
                    }
                    return Result.success(services);
                });
    }
}
