package ru.yandex.intranet.d.services.validators;

import java.util.Locale;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;

/**
 * Provider Id validator.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 21.01.2021
 */
@Component
public class ProviderValidator {
    private final MessageSource messages;
    private final ProvidersLoader providersLoader;

    public ProviderValidator(@Qualifier("messageSource") MessageSource messages, ProvidersLoader providersLoader) {
        this.messages = messages;
        this.providersLoader = providersLoader;
    }

    public Mono<Result<ProviderModel>> validateProvider(TenantId tenantId, String providerId, Locale locale) {
        if (!Uuids.isValidUuid(providerId)) {
            ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)))
                    .build();
            return Mono.just(Result.failure(error));
        }
        return providersLoader.getProviderByIdImmediate(providerId, tenantId).map(provider -> {
            if (provider.isEmpty() || provider.get().isDeleted()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                        .getMessage("errors.provider.not.found", null, locale)))
                        .build();
                return Result.failure(error);
            }
            return Result.success(provider.get());
        });
    }

}
