package ru.yandex.intranet.d.services.validators;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.resources.segmentations.ResourceSegmentationsDao;
import ru.yandex.intranet.d.dao.resources.segments.ResourceSegmentsDao;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;

/**
 * SegmentationsValidator.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 05.02.2021
 */
@Component
public class SegmentationsValidator {
    private final MessageSource messages;
    private final ResourceSegmentationsDao resourceSegmentationsDao;
    private final ResourceSegmentsDao resourceSegmentsDao;

    public SegmentationsValidator(
            @Qualifier("messageSource") MessageSource messages,
            ResourceSegmentationsDao resourceSegmentationsDao,
            ResourceSegmentsDao resourceSegmentsDao
    ) {
        this.messages = messages;
        this.resourceSegmentationsDao = resourceSegmentationsDao;
        this.resourceSegmentsDao = resourceSegmentsDao;
    }

    @SuppressWarnings("ParameterNumber")
    public <T> Mono<Result<List<Tuple2<ResourceSegmentationModel, ResourceSegmentModel>>>> validateSegmentations(
            YdbTxSession session,
            Supplier<Optional<List<T>>> segmentationsSupplier,
            Function<T, Optional<String>> segmentationIdGetter,
            Function<T, Optional<String>> segmentIdGetter,
            String providerId,
            Locale locale,
            String fieldKey,
            boolean required
    ) {
        Optional<List<T>> values = segmentationsSupplier.get();
        if (values.isEmpty() || values.get().isEmpty()) {
            if (required) {
                ErrorCollection error = ErrorCollection.builder().addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)))
                        .build();
                return Mono.just(Result.failure(error));
            } else {
                return Mono.just(Result.success(List.of()));
            }
        }
        ErrorCollection.Builder errorsBuilder = ErrorCollection.builder();
        List<Tuple2<Integer, String>> segmentationIdsToLoad = new ArrayList<>();
        List<Tuple2<Integer, String>> segmentIdsToLoad = new ArrayList<>();
        List<Tuple2<Integer, Tuple2<String, String>>> indexedSegments = new ArrayList<>();
        collectSegmentsToLoad(values.get(), segmentationIdsToLoad, segmentIdsToLoad, indexedSegments,
                segmentationIdGetter, segmentIdGetter, locale, fieldKey, errorsBuilder);
        ErrorCollection errors = errorsBuilder.build();
        List<Tuple2<String, TenantId>> segmentationIds = segmentationIdsToLoad.stream()
                .map(t -> Tuples.of(t.getT2(), Tenants.DEFAULT_TENANT_ID)).distinct().collect(Collectors.toList());
        List<Tuple2<String, TenantId>> segmentIds = segmentIdsToLoad.stream()
                .map(t -> Tuples.of(t.getT2(), Tenants.DEFAULT_TENANT_ID)).distinct().collect(Collectors.toList());
        return resourceSegmentationsDao.getByIds(session, segmentationIds).flatMap(segmentations ->
                resourceSegmentsDao.getByIds(session, segmentIds)
                        .map(segments -> fillSegmentations(segmentationIdsToLoad, segmentIdsToLoad, indexedSegments,
                                segmentations, segments, errors, providerId, locale, fieldKey)));
    }

    @SuppressWarnings("ParameterNumber")
    private Result<List<Tuple2<ResourceSegmentationModel, ResourceSegmentModel>>> fillSegmentations(
            List<Tuple2<Integer, String>> segmentationIdsToLoad,
            List<Tuple2<Integer, String>> segmentIdsToLoad,
            List<Tuple2<Integer, Tuple2<String, String>>> indexedSegments,
            List<ResourceSegmentationModel> segmentations,
            List<ResourceSegmentModel> segments,
            ErrorCollection knownErrors,
            String providerId,
            Locale locale,
            String fieldKey) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        errors.add(knownErrors);
        List<Tuple2<ResourceSegmentationModel, ResourceSegmentModel>> result = new ArrayList<>();
        Map<String, ResourceSegmentationModel> segmentationsById = segmentations.stream()
                .collect(Collectors.toMap(ResourceSegmentationModel::getId, s -> s));
        Map<String, ResourceSegmentModel> segmentsById = segments.stream()
                .collect(Collectors.toMap(ResourceSegmentModel::getId, s -> s));
        segmentationIdsToLoad.forEach(t -> {
            ResourceSegmentationModel segmentation = segmentationsById.get(t.getT2());
            if (segmentation == null || segmentation.isDeleted()
                    || !segmentation.getProviderId().equals(providerId)) {
                errors.addError(fieldKey + "." + t.getT1() + ".segmentationId", TypedError.invalid(messages
                        .getMessage("errors.resource.segmentation.not.found", null, locale)));
            }
        });
        segmentIdsToLoad.forEach(t -> {
            ResourceSegmentModel segment = segmentsById.get(t.getT2());
            if (segment == null || segment.isDeleted()) {
                errors.addError(fieldKey + "." + t.getT1() + ".segmentId", TypedError.invalid(messages
                        .getMessage("errors.resource.segment.not.found", null, locale)));
            }
        });
        indexedSegments.forEach(t -> {
            String segmentationId = t.getT2().getT1();
            String segmentId = t.getT2().getT2();
            ResourceSegmentationModel segmentation = segmentationsById.get(segmentationId);
            ResourceSegmentModel segment = segmentsById.get(segmentId);
            if (segmentation == null || segmentation.isDeleted()
                    || !segmentation.getProviderId().equals(providerId) || segment == null || segment.isDeleted()) {
                return;
            }
            if (!segmentation.getId().equals(segment.getSegmentationId())) {
                errors.addError(fieldKey + "." + t.getT1() + ".segmentId", TypedError.invalid(messages
                        .getMessage("errors.resource.segment.not.found", null, locale)));
            }
        });
        indexedSegments.stream().map(Tuple2::getT2).distinct().forEach(t -> {
            String segmentationId = t.getT1();
            String segmentId = t.getT2();
            ResourceSegmentationModel segmentation = segmentationsById.get(segmentationId);
            ResourceSegmentModel segment = segmentsById.get(segmentId);
            if (segmentation == null || segmentation.isDeleted()
                    || !segmentation.getProviderId().equals(providerId) || segment == null || segment.isDeleted()
                    || !segmentation.getId().equals(segment.getSegmentationId())) {
                return;
            }
            result.add(Tuples.of(segmentation, segment));
        });
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(result);
    }

    @SuppressWarnings("ParameterNumber")
    private <T> void collectSegmentsToLoad(List<T> values,
                                           List<Tuple2<Integer, String>> segmentationIdsToLoad,
                                           List<Tuple2<Integer, String>> segmentIdsToLoad,
                                           List<Tuple2<Integer, Tuple2<String, String>>> indexedSegments,
                                           Function<T, Optional<String>> segmentationIdGetter,
                                           Function<T, Optional<String>> segmentIdGetter,
                                           Locale locale, String fieldKey, ErrorCollection.Builder errors) {
        for (int i = 0; i < values.size(); i++) {
            T value = values.get(i);
            if (value == null) {
                errors.addError(fieldKey + "." + i, TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
                continue;
            }
            boolean hasSegmentation = false;
            boolean hasSegment = false;
            if (segmentationIdGetter.apply(value).isEmpty()) {
                errors.addError(fieldKey + "." + i + ".segmentationId", TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
            } else if (segmentationIdGetter.apply(value).get().isBlank()) {
                errors.addError(fieldKey + "." + i + ".segmentationId", TypedError.invalid(messages
                        .getMessage("errors.non.blank.text.is.required", null, locale)));
            } else if (!Uuids.isValidUuid(segmentationIdGetter.apply(value).get())) {
                errors.addError(fieldKey + "." + i + ".segmentationId", TypedError.invalid(messages
                        .getMessage("errors.resource.segmentation.not.found", null, locale)));
            } else {
                segmentationIdsToLoad.add(Tuples.of(i, segmentationIdGetter.apply(value).get()));
                hasSegmentation = true;
            }
            if (segmentIdGetter.apply(value).isEmpty()) {
                errors.addError(fieldKey + "." + i + ".segmentId", TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)));
            } else if (segmentIdGetter.apply(value).get().isBlank()) {
                errors.addError(fieldKey + "." + i + ".segmentId", TypedError.invalid(messages
                        .getMessage("errors.non.blank.text.is.required", null, locale)));
            } else if (!Uuids.isValidUuid(segmentIdGetter.apply(value).get())) {
                errors.addError(fieldKey + "." + i + ".segmentId", TypedError.invalid(messages
                        .getMessage("errors.resource.segment.not.found", null, locale)));
            } else {
                segmentIdsToLoad.add(Tuples.of(i, segmentIdGetter.apply(value).get()));
                hasSegment = true;
            }
            if (hasSegmentation && hasSegment) {
                indexedSegments.add(Tuples.of(i, Tuples.of(segmentationIdGetter.apply(value).get(),
                        segmentIdGetter.apply(value).get())));
            }
        }
        Map<String, Long> segmentationCounts = values.stream().filter(Objects::nonNull)
                .flatMap(v -> segmentationIdGetter.apply(v).stream())
                .collect(Collectors.toMap(Function.identity(), v -> 1L, Long::sum));
        if (segmentationCounts.values().stream().anyMatch(v -> v > 1)) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.no.more.than.one.segment.per.segmentation.per.resource.is.allowed",
                            null, locale)));
        }
    }
}
