package ru.yandex.intranet.d.services.validators;

import java.text.Normalizer;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;

import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;

/**
 * Validator for texts.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 */
@Component
public class TextValidator {
    private final MessageSource messages;

    public TextValidator(@Qualifier("messageSource") MessageSource messages) {
        this.messages = messages;
    }

    public TextValidator validateAndSet(
            Supplier<Optional<String>> getter,
            Consumer<String> setter,
            ErrorCollection.Builder errors,
            String fieldKey,
            int maxLength,
            Locale locale
    ) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (text.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            setter.accept(text.get());
        }
        return this;
    }

    public TextValidator validateAndSetTags(
            Set<String> tags,
            Consumer<Set<String>> setter,
            ErrorCollection.Builder errors,
            String fieldKey,
            int maxLength,
            Locale locale
    ) {
        if (tags == null) {
            return this;
        }
        Set<String> result = tags.stream().map(String::strip).collect(Collectors.toSet());
        boolean valid = true;
        for (String tag : result) {
            if (tag.isBlank()) {
                valid = false;
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.non.blank.text.is.required", null, locale)));
            } else if (tag.length() > maxLength) {
                valid = false;
                errors.addError(fieldKey, TypedError.invalid(messages
                        .getMessage("errors.text.is.too.long", null, locale)));
            }
            String normalizedTag = Normalizer.normalize(tag, Normalizer.Form.NFKC).toLowerCase();
            if (!normalizedTag.equals(tag)) {
                valid = false;
                errors
                        .addError(fieldKey, TypedError.invalid(messages.getMessage(
                                "errors.text.is.not.normalized", null, locale
                        )))
                        .addDetail(fieldKey, Map.of(
                                "wrong_value", tag,
                                "suggest_value", normalizedTag
                        ));
            }
        }
        if (valid) {
            setter.accept(result);
        }
        return this;
    }

    public Result<Void> validateId(String id, Locale locale, String errorMessageCode) {
        if (!Uuids.isValidUuid(id)) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.notFound(messages
                    .getMessage(errorMessageCode, null, locale)))
                    .build());
        }
        return Result.success(null);
    }

    public Wrapper with(ErrorCollection.Builder errors, Locale locale) {
        return new Wrapper(this, errors, locale);
    }

    public static class Wrapper {
        private final TextValidator validator;
        private final ErrorCollection.Builder errors;
        private final Locale locale;

        private Wrapper(
                TextValidator validator,
                ErrorCollection.Builder errors,
                Locale locale
        ) {
            this.validator = validator;
            this.errors = errors;
            this.locale = locale;
        }

        public Wrapper validateAndSet(
                Supplier<Optional<String>> getter,
                Consumer<String> setter,
                String fieldKey,
                int maxLength
        ) {
            validator.validateAndSet(getter, setter, errors, fieldKey, maxLength, locale);
            return this;
        }
    }
}
