package ru.yandex.intranet.d.tms;

import java.time.Clock;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Profile;

import ru.yandex.direct.hourglass.HourglassProperties;
import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.InstanceMeta;
import ru.yandex.direct.hourglass.MonitoringWriter;
import ru.yandex.direct.hourglass.RandomChooser;
import ru.yandex.direct.hourglass.SchedulerInstancePinger;
import ru.yandex.direct.hourglass.SchedulerService;
import ru.yandex.direct.hourglass.TaskThreadPool;
import ru.yandex.direct.hourglass.implementations.InstanceIdImpl;
import ru.yandex.direct.hourglass.implementations.InstanceMetaImpl;
import ru.yandex.direct.hourglass.implementations.MD5Hash;
import ru.yandex.direct.hourglass.implementations.SchedulerInstancePingerImpl;
import ru.yandex.direct.hourglass.implementations.SchedulerServiceImpl;
import ru.yandex.direct.hourglass.implementations.TaskThreadPoolImpl;
import ru.yandex.direct.hourglass.implementations.ThreadsHierarchy;
import ru.yandex.direct.hourglass.implementations.randomchoosers.RendezvousRandomChooser;
import ru.yandex.direct.hourglass.implementations.updateschedule.MainScheduleVersionExtractorImpl;
import ru.yandex.direct.hourglass.implementations.updateschedule.ScheduleUpdateServiceImpl;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.direct.hourglass.storage.Storage;
import ru.yandex.direct.hourglass.updateschedule.MainScheduleVersionExtractor;
import ru.yandex.direct.hourglass.updateschedule.ScheduleUpdateService;
import ru.yandex.direct.hourglass.updateschedule.SchedulerInstancesRepository;
import ru.yandex.direct.scheduler.AlwaysTrue;
import ru.yandex.direct.scheduler.JobInterceptorsList;
import ru.yandex.direct.scheduler.hourglass.ScheduleRecordListProvider;
import ru.yandex.direct.scheduler.hourglass.TaskFactory;
import ru.yandex.direct.scheduler.hourglass.TaskListProvider;
import ru.yandex.direct.scheduler.hourglass.implementations.HourglassScheduler;
import ru.yandex.direct.scheduler.hourglass.implementations.JobScheduleInfoFactory;
import ru.yandex.direct.scheduler.hourglass.implementations.ScheduleInfoProcessor;
import ru.yandex.direct.scheduler.hourglass.implementations.ScheduleRecordListProviderImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.TaskFactoryImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.TaskListProviderImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.TaskParameterizer;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.ScheduleInfoConverterImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.modifiers.NextRunModifierFactoryImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.modifiers.RandomStartTime;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.strategies.ByCronExpressionStrategy;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.strategies.NextRunStrategiesFactoryImpl;
import ru.yandex.direct.scheduler.hourglass.implementations.schedule.strategies.PeriodicStrategy;
import ru.yandex.direct.scheduler.hourglass.schedule.ScheduleInfoConverter;
import ru.yandex.direct.scheduler.hourglass.schedule.modifiers.NextRunModifier;
import ru.yandex.direct.scheduler.hourglass.schedule.modifiers.NextRunModifierFactory;
import ru.yandex.direct.scheduler.hourglass.schedule.strategies.NextRunCalcStrategy;
import ru.yandex.direct.scheduler.hourglass.schedule.strategies.NextRunStrategiesFactory;
import ru.yandex.direct.scheduler.support.JobExecutingMonitor;
import ru.yandex.direct.utils.SystemUtils;

/**
 * Hourglass configuration.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 */
@Configuration
@Import({
        HourglassPropertiesConfiguration.class,
        HourglassYdbStorageConfiguration.class
})
@Profile({"local", "dev", "load-testing", "testing", "production"})
public class HourglassConfiguration {
    private static final Duration JOB_EXECUTION_MONITOR_INTERVAL = Duration.ofSeconds(10);

    @Bean
    public InstanceId instanceId() {
        return new InstanceIdImpl();
    }

    @Bean
    public InstanceMeta instanceMeta() {
        var host = SystemUtils.strictHostname();
//                .replace(".vm.search.yandex.net", "")
//                .replace(".gencfg-c.yandex.net", "");
        var startTime = LocalDateTime.now(Clock.systemUTC()).toString();
        return new InstanceMetaImpl(host, startTime);
    }

//    @Bean
//    public HourglassJugglerSender hourglassJugglerSender(JugglerSender jugglerSender) {
//        return new HourglassJugglerSender(jugglerSender);
//    }
//
//    @Bean
//    @Conditional(OnlyForTestingCondition.class)
//    public UptimeSchedulerPingListener uptimeSchedulerPingListener(HourglassJugglerSender hourglassJugglerSender) {
//        return new UptimeSchedulerPingListener(hourglassJugglerSender);
//    }
//
//    @Bean
//    public CheckInSchdedulerPingListener checkInSchdedulerPingListener(HourglassJugglerSender hourglassJugglerSender,
//                                                                       InstanceId instanceId,
//                                                                       InstanceMeta instanceMeta) {
//        return new CheckInSchdedulerPingListener(hourglassJugglerSender, instanceId, instanceMeta);
//    }
//
//    @Bean
//    public SchedulerPingMonitoring schedulerPingMonitoring(List<SchedulerPingListener> schedulerPingListenerList) {
//        return new SchedulerPingMonitoring(schedulerPingListenerList);
//    }
//
//    @Bean
//    public MetricRegistry metricRegistry() {
//        Labels solomonRegistryLabels = Labels.of("hourglass_scheduler", "HourglassApp");
//        return SOLOMON_REGISTRY.subRegistry(solomonRegistryLabels);
//    }
//
//    @Bean
//    public MonitoringWriter monitoringWriter(MetricRegistry metricRegistry,
//                                             SchedulerPingMonitoring schedulerPingMonitoring) {
//        return new MonitoringWriterImpl(metricRegistry, schedulerPingMonitoring);
//    }

    @Bean
    public MonitoringWriter monitoringWriter() {
        return new MonitoringWriterDummy();
    }

    @Bean
    public ThreadsHierarchy threadsHierarchy() {
        return new ThreadsHierarchy();
    }

    @Bean
    public TaskThreadPool taskThreadPool(ThreadsHierarchy threadsHierarchy, HourglassProperties hourglassProperties) {
        return new TaskThreadPoolImpl(hourglassProperties.getThreadPoolSize(), threadsHierarchy);
    }

    @Bean
    public RandomChooser<PrimaryId> randomChooser(
            @Value("${scheduler.version}") String version,
            SchedulerInstancesRepository schedulerInstancesRepository,
            InstanceId instanceId
    ) {
        return new RendezvousRandomChooser<>(version, schedulerInstancesRepository, instanceId, new MD5Hash());
    }

    @Bean
    @Lazy
    @SuppressWarnings("ParameterNumber")
    public SchedulerService schedulerService(
            Storage storage,
            InstanceId instanceId,
            RandomChooser<PrimaryId> randomChooser,
            ThreadsHierarchy threadsHierarchy,
            TaskThreadPool taskThreadPool,
            HourglassProperties hourglassProperties,
            SchedulerInstancePinger schedulerInstancePinger,
//            TraceHelper traceHelper,
            MonitoringWriter monitoringWriter
    ) {
        return new SchedulerServiceImpl(
                storage,
                threadsHierarchy,
                taskThreadPool,
                instanceId,
                randomChooser,
                hourglassProperties,
                schedulerInstancePinger,
                monitoringWriter
        );
//        return new SchedulerTracingWrapper(storage,
//                threadsHierarchy,
//                taskThreadPool,
//                instanceId,
//                randomChooser,
//                hourglassProperties,
//                monitoringWriter,
//                schedulerInstancePinger,
//                traceHelper);
    }

    @Bean
    public JobScheduleInfoFactory taskScheduleGetter(ApplicationContext context) {
        return new JobScheduleInfoFactory(context);
    }


    @Bean
    public TaskParameterizer taskParameterizer(ApplicationContext context,
                                               @Value("${scheduler.totalShards}") int totalShards) {
        Collection<Integer> shardIndexes = IntStream.range(0, totalShards).boxed().collect(Collectors.toList());
        return new TaskParameterizer(context, shardIndexes);
    }

    @Bean
    public TaskListProvider taskListProvider(ApplicationContext context) {
        return new TaskListProviderImpl(context);
    }

    @Bean
    public NextRunStrategiesFactory nextRunStrategiesFactory() {
        List<NextRunCalcStrategy> nextRunCalcStrategies = List.of(new PeriodicStrategy(),
                new ByCronExpressionStrategy());
        return new NextRunStrategiesFactoryImpl(nextRunCalcStrategies);
    }

    @Bean
    RandomStartTime randomStartTime() {
        return new RandomStartTime();
    }

    @Bean
    public NextRunModifierFactory nextRunModifierFactory(List<NextRunModifier> modifiers) {
        return new NextRunModifierFactoryImpl(modifiers);
    }

    @Bean
    public ScheduleInfoProcessor scheduleInfoProcessor(
            NextRunStrategiesFactory nextRunStrategiesFactory,
            NextRunModifierFactory nextRunModifierFactory
    ) {
        return new ScheduleInfoProcessor(nextRunStrategiesFactory, nextRunModifierFactory);
    }

    @Bean
    public ScheduleInfoConverter scheduleInfoConverter() {
        return new ScheduleInfoConverterImpl();
    }

    @Bean
    JobExecutingMonitor jobExecutingMonitor() {
        return new JobExecutingMonitorDummy();
    }

    @Bean
    JobInterceptorsList jobInterceptorsList() {
        return new JobInterceptorsList();
    }

    @Bean
    public TaskFactory taskFactory(
            ApplicationContext context,
            TaskListProvider taskListProvider,
            InstanceId instanceId,
            JobInterceptorsList jobInterceptorsList,
            JobExecutingMonitor jobExecutingMonitor,
            ScheduleInfoConverter scheduleInfoConverter,
            ScheduleInfoProcessor scheduleInfoProcessor
    ) {
        jobExecutingMonitor.setSchedulerName(instanceId.toString());
        jobExecutingMonitor.setInterval(JOB_EXECUTION_MONITOR_INTERVAL);
        return new TaskFactoryImpl(context, taskListProvider, jobInterceptorsList, jobExecutingMonitor,
                scheduleInfoConverter, scheduleInfoProcessor);
    }

    @Bean
    public ScheduleRecordListProvider scheduleRecordListProvider(
            TaskListProvider taskListProvider,
            ScheduleInfoConverter scheduleInfoConverter,
            TaskParameterizer taskParameterizer,
            JobScheduleInfoFactory jobScheduleInfoFactory
    ) {
        return new ScheduleRecordListProviderImpl(
                taskListProvider, scheduleInfoConverter, taskParameterizer,
                jobScheduleInfoFactory
        );
    }

    @Bean
    @Lazy
    public HourglassScheduler hourglassScheduler(
            ScheduleRecordListProvider scheduleRecordListProvider,
            TaskFactory taskFactory,
            SchedulerService schedulerService,
            ThreadsHierarchy threadsHierarchy,
//            @Qualifier("schedulerVersionWatcher") LiveResourceWatcher watcher,
//            VersionMismatchTerminator mismatchTerminator,
            ScheduleUpdateService scheduleUpdateService
    ) {
        HourglassScheduler hourglassScheduler = new HourglassScheduler(
                scheduleRecordListProvider,
                taskFactory,
                schedulerService,
                scheduleUpdateService
        );
        threadsHierarchy.addUncaughtExceptionHandler((a, b) -> hourglassScheduler.stop());
//        watcher.addListener(event -> mismatchTerminator.terminateIfVersionMismatch(event.getCurrentContent()));
//        watcher.watch();
        return hourglassScheduler;
    }

    @Bean
    public MainScheduleVersionExtractor mainScheduleVersionExtractor(
//            @Value("${scheduler.versionFile}") String mainVersionPath
            @Value("${scheduler.version}") String version
    ) {
//        var liveResource = LiveResourceFactory.get(mainVersionPath);
//        return new MainScheduleVersionExtractorImpl(liveResource::getContent);
        return new MainScheduleVersionExtractorImpl(() -> version);
    }

    @Bean
    @Lazy
    public SchedulerInstancePinger schedulerInstancePinger(
            InstanceId instanceId,
            InstanceMeta instanceMeta,
            SchedulerInstancesRepository schedulerInstancesRepository,
            @Value("${scheduler.version}") String version,
            MonitoringWriter monitoringWriter
    ) {
        return new SchedulerInstancePingerImpl(instanceId, version, schedulerInstancesRepository, instanceMeta,
                monitoringWriter);
    }

    @Bean
    @SuppressWarnings("ParameterNumber")
    public ScheduleUpdateService scheduleUpdateService(
            MainScheduleVersionExtractor mainScheduleVersionExtractor,
            SchedulerInstancesRepository schedulerInstancesRepository,
            Storage storage,
            ThreadsHierarchy threadsHierarchy,
            MonitoringWriter monitoringWriter,
            InstanceId instanceId,
            @Value("${scheduler.version}") String version,
            HourglassProperties hourglassProperties
    ) {
        return new ScheduleUpdateServiceImpl.Builder()
                .withStorage(storage)
                .withMainVersionExtractor(mainScheduleVersionExtractor)
                .withSchedulerInstancesRepository(schedulerInstancesRepository)
                .withInstanceId(instanceId)
                .withCurrentVersion(version)
                .withThreadsHierarchy(threadsHierarchy)
                .withMonitoringWriter(monitoringWriter)
                .withFrequency(hourglassProperties.getScheduleUpdateInterval())
                .build();
    }

    @Bean
    AlwaysTrue alwaysTrue() {
        return new AlwaysTrue();
    }
}
