package ru.yandex.intranet.d.tms;

import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.PostConstruct;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;

import ru.yandex.direct.scheduler.hourglass.implementations.HourglassScheduler;

/**
 * Нужен для асинхронного запуска планировщика.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 */
@Component
@Profile({"local", "dev", "load-testing", "testing", "production"})
public class HourglassSchedulerStarterImpl implements HourglassSchedulerStarter {
    private static final long INITIAL_DELAY_MS = 1000;
    private static final long RETRY_INTERVAL_MS = 60000;

    private static final Logger LOG = LoggerFactory.getLogger(HourglassSchedulerStarterImpl.class);

    private final ApplicationContext context;
    private final boolean schedulerEnable;
    private volatile boolean ready = false;
    private final AtomicLong retryCount = new AtomicLong(0);

    public HourglassSchedulerStarterImpl(
            ApplicationContext context,
            @Value("${scheduler.enable}") boolean schedulerEnable
    ) {
        this.context = context;
        this.schedulerEnable = schedulerEnable;
    }

    @PostConstruct
    public void startScheduler() {
        if (!schedulerEnable) {
            LOG.info("Scheduler disabled.");
            return;
        }
        Thread thread = new Thread(() -> {
            HourglassScheduler scheduler = null;
            try {
                Thread.sleep(INITIAL_DELAY_MS);
                while (scheduler == null) {
                    LOG.info("Starting scheduler...");
                    try {
                        scheduler = context.getBean(HourglassScheduler.class);
                    } catch (Exception e) {
                        retryCount.incrementAndGet();
                        LOG.error("Scheduler starting error, waiting for retry.", e);
                        Thread.sleep(RETRY_INTERVAL_MS);
                    }
                }
                scheduler.saveSchedule();
                scheduler.start();
                ready = true;
            } catch (InterruptedException e) {
                LOG.error("Scheduler starting interrupted", e);
                Thread.currentThread().interrupt();
            }
        });
        thread.setName(HourglassSchedulerStarterImpl.class.getSimpleName() + "-Thread");
        thread.setDaemon(true);
        thread.start();
    }

    @Override
    public boolean isReady() {
        return ready;
    }

    @Override
    public long getRetryCount() {
        return retryCount.get();
    }
}
