package ru.yandex.intranet.d.tms.jobs;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.MessageSource;
import org.thymeleaf.TemplateEngine;
import org.thymeleaf.context.Context;
import reactor.core.publisher.Mono;

import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.BaseDirectJob;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.transfers.QuotaTransfer;
import ru.yandex.intranet.d.model.transfers.ResourceQuotaTransfer;
import ru.yandex.intranet.d.model.transfers.TransferRequestModel;
import ru.yandex.intranet.d.model.transfers.TransferRequestType;
import ru.yandex.intranet.d.model.units.DecimalWithUnit;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.notifications.MailNotification;
import ru.yandex.intranet.d.services.notifications.NotificationMailSender;
import ru.yandex.intranet.d.services.quotas.QuotasHelper;
import ru.yandex.intranet.d.services.quotas.QuotasMoveStatisticResult;
import ru.yandex.intranet.d.services.quotas.QuotasMoveStatisticService;
import ru.yandex.intranet.d.services.transfer.TextFormatterService;
import ru.yandex.intranet.d.util.FrontStringUtil;

/**
 * Send mail about quotas move
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Hourglass(periodInSeconds = 86400)
public class QuotasMoveStatisticJob extends BaseDirectJob {

    private static final Logger LOG = LoggerFactory.getLogger(QuotasMoveStatisticJob.class);

    private final QuotasMoveStatisticService quotasMoveStatisticService;
    private final MessageSource emailMessageSource;
    private final TemplateEngine emailTemplateEngine;
    private final TextFormatterService textFormatterService;
    private final NotificationMailSender mailSender;
    private final List<Long> notificationServiceIds;
    private final String notificationDestination;
    private final String fromAddress;

    @SuppressWarnings("ParameterNumber")
    public QuotasMoveStatisticJob(QuotasMoveStatisticService quotasMoveStatisticService,
                                  @Qualifier("emailMessageSource") MessageSource emailMessageSource,
                                  TemplateEngine emailTemplateEngine,
                                  TextFormatterService textFormatterService,
                                  NotificationMailSender mailSender,
                                  @Value("${notifications.quotaMove.serviceIds}") List<Long> notificationServiceIds,
                                  @Value("${notifications.quotaMove.destination}") String notificationDestination,
                                  @Value("${notifications.mail.from}") String fromAddress) {
        this.quotasMoveStatisticService = quotasMoveStatisticService;
        this.emailMessageSource = emailMessageSource;
        this.emailTemplateEngine = emailTemplateEngine;
        this.textFormatterService = textFormatterService;
        this.mailSender = mailSender;
        this.notificationServiceIds = notificationServiceIds;
        this.notificationDestination = notificationDestination;
        this.fromAddress = fromAddress;
    }

    @Override
    public void execute() {
        LOG.info("Running task for send quotas move statistic...");
        Instant now = Instant.now();
        quotasMoveStatisticService.getQuotasMoveStatistic(now.minus(Duration.ofDays(1)), now,
                notificationServiceIds, Tenants.DEFAULT_TENANT_ID, Locale.ENGLISH)
                .map(res -> res.andThen(quotasMoveStatistic -> {
                    if (quotasMoveStatistic.getTransfersApplied().isEmpty() &&
                            quotasMoveStatistic.getTransfersCreated().isEmpty()) {
                        return res;
                    }
                    MailNotification mailNotification = MailNotification.builder()
                            .htmlBody(generateHtmlQuotasMoveStatisticNotification(quotasMoveStatistic))
                            .plainTextBody(generateTextQuotasMoveStatisticNotification(quotasMoveStatistic))
                            .subject("Quota movements to service subtree for the last day")
                            .from(fromAddress)
                            .to(notificationDestination)
                            .actualTo(null)
                            .fromName(emailMessageSource.getMessage(
                                    "notification.mail.from.name", null, Locale.ENGLISH))
                            .build();
                    mailSender.sendBackground(mailNotification);
                    return res;
                }))
                .onErrorResume(e -> {
                    LOG.error("Failed task for calculate quotas move statistic", e);
                    return Mono.empty();
                })
                .block();
        LOG.info("Task for send quotas move statistic finished");
    }

    public String generateHtmlQuotasMoveStatisticNotification(QuotasMoveStatisticResult quotasMoveStatistic) {
        Context context = new Context(Locale.ENGLISH);
        context.setVariable("transfersCreated", getListTransferDescriptionHtml(
                quotasMoveStatistic.getTransfersCreated(),
                quotasMoveStatistic.getResourceById(),
                quotasMoveStatistic.getUnitsEnsembleById()));
        context.setVariable("transfersApplied", getListTransferDescriptionHtml(
                quotasMoveStatistic.getTransfersApplied(),
                quotasMoveStatistic.getResourceById(),
                quotasMoveStatistic.getUnitsEnsembleById()));
        return emailTemplateEngine.process("html/quotas_move_statistic.html", context);
    }

    public String generateTextQuotasMoveStatisticNotification(QuotasMoveStatisticResult quotasMoveStatistic) {
        Context context = new Context(Locale.ENGLISH);
        context.setVariable("transfersCreated", getListTransferDescriptionText(
                quotasMoveStatistic.getTransfersCreated(),
                quotasMoveStatistic.getResourceById(),
                quotasMoveStatistic.getUnitsEnsembleById()));
        context.setVariable("transfersApplied", getListTransferDescriptionText(
                quotasMoveStatistic.getTransfersApplied(),
                quotasMoveStatistic.getResourceById(),
                quotasMoveStatistic.getUnitsEnsembleById()));
        return emailTemplateEngine.process("text/quotas_move_statistic.txt", context);
    }

    private String getListTransferDescriptionText(List<TransferRequestModel> transferRequests,
                                                  Map<String, ResourceModel> resourceById,
                                                  Map<String, UnitsEnsembleModel> unitsEnsembleById) {
        return transferRequests.stream()
                .map(t -> {
                    String description = t.getSummary().get() + ": "
                            + textFormatterService.buildTransferUrl(t.getId()) + "\n";
                    if (t.getType() == TransferRequestType.QUOTA_TRANSFER) {
                        if (t.getParameters().getQuotaTransfers().size() == 2) {
                            Set<QuotaTransfer> quotaTransfers = new HashSet<>(t.getParameters().getQuotaTransfers());
                            QuotaTransfer sourceTransfer = t.getParameters().getQuotaTransfers().stream()
                                    .filter(q -> transferHasNegativeQuota(q.getTransfers()))
                                    .findFirst().orElseThrow();
                            quotaTransfers.remove(sourceTransfer);
                            description += quotaTransfers.stream()
                                    .map(q -> convertResourceTransfersToString(
                                            q.getTransfers(), resourceById, unitsEnsembleById))
                                    .findFirst()
                                    .orElse("");
                        }
                    }
                    if (t.getType() == TransferRequestType.PROVISION_TRANSFER) {
                        description += t.getParameters().getProvisionTransfers().stream()
                                .map(p -> convertResourceTransfersToString(
                                        p.getDestinationAccountTransfers(), resourceById, unitsEnsembleById))
                                .findFirst()
                                .orElse("");
                    }
                    return description;
                })
                .collect(Collectors.joining("\n"));
    }

    private List<TransferHtml> getListTransferDescriptionHtml(List<TransferRequestModel> transferRequests,
                                                              Map<String, ResourceModel> resourceById,
                                                              Map<String, UnitsEnsembleModel> unitsEnsembleById) {
        return transferRequests.stream()
                .map(t -> {
                    StringBuilder resources = new StringBuilder();
                    if (t.getType() == TransferRequestType.QUOTA_TRANSFER) {
                        if (t.getParameters().getQuotaTransfers().size() == 2) {
                            Set<QuotaTransfer> quotaTransfers = new HashSet<>(t.getParameters().getQuotaTransfers());
                            QuotaTransfer sourceTransfer = t.getParameters().getQuotaTransfers().stream()
                                    .filter(q -> transferHasNegativeQuota(q.getTransfers()))
                                    .findFirst().orElseThrow();
                            quotaTransfers.remove(sourceTransfer);
                            resources.append(quotaTransfers.stream()
                                    .map(q -> convertResourceTransfersToString(
                                            q.getTransfers(), resourceById, unitsEnsembleById))
                                    .findFirst()
                                    .orElse(""));
                        }
                    }
                    if (t.getType() == TransferRequestType.PROVISION_TRANSFER) {
                        resources.append(t.getParameters().getProvisionTransfers().stream()
                                .map(p -> convertResourceTransfersToString(
                                        p.getDestinationAccountTransfers(), resourceById, unitsEnsembleById))
                                .findFirst()
                                .orElse(""));
                    }
                    TransferHtml transferHtml = new TransferHtml(t.getSummary().get(),
                            resources.toString(), textFormatterService.buildTransferUrl(t.getId()));
                    return transferHtml;
                })
                .toList();
    }

    private boolean transferHasNegativeQuota(Collection<ResourceQuotaTransfer> transfers) {
        return transfers.stream()
                .anyMatch(resourceQuotaTransfer -> resourceQuotaTransfer.getDelta() < 0);
    }

    private String convertResourceTransfersToString(
            Set<ResourceQuotaTransfer> resourceQuotaTransfers,
            Map<String, ResourceModel> resourceById,
            Map<String, UnitsEnsembleModel> unitsEnsembleById) {
        return resourceQuotaTransfers.stream()
                .map(resourceQuotaTransfer ->
                        convertResourceTransferToString(resourceQuotaTransfer, resourceById, unitsEnsembleById))
                .collect(Collectors.joining(", "));
    }

    private String convertResourceTransferToString(
            ResourceQuotaTransfer resourceQuotaTransfer,
            Map<String, ResourceModel> resourceById,
            Map<String, UnitsEnsembleModel> unitsEnsembleById) {
        ResourceModel resource = resourceById.get(resourceQuotaTransfer.getResourceId());
        UnitsEnsembleModel unitsEnsemble = unitsEnsembleById.get(resource.getUnitsEnsembleId());
        UnitModel unitModel = unitsEnsemble.getUnits().stream()
                .filter(u -> u.getId().equals(resource.getBaseUnitId())).findFirst().get();
        DecimalWithUnit decimalWithUnit = QuotasHelper.convertToReadable(
                BigDecimal.valueOf(resourceQuotaTransfer.getDelta()),
                unitsEnsemble.getUnits().stream()
                        .sorted(Comparator.comparingLong(UnitModel::getPower))
                        .toList(),
                unitModel);

        String name = resource.getNameRu();
        String delta = FrontStringUtil.toString(QuotasHelper.roundForDisplay(decimalWithUnit.getAmount()));
        String unit = decimalWithUnit.getUnit().getShortNameSingularEn();
        return name + ": " + delta + " " + unit;
    }

    public record TransferHtml(String summary, String resources, String link) { }
}
