package ru.yandex.intranet.d.tms.jobs;

import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.BaseDirectJob;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.users.AbcIntranetStaffDao;
import ru.yandex.intranet.d.dao.users.AbcServiceMemberDao;
import ru.yandex.intranet.d.dao.users.UsersDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.users.AbcServiceMemberModel;
import ru.yandex.intranet.d.model.users.AbcStaffAffiliation;
import ru.yandex.intranet.d.model.users.AbcUserModel;
import ru.yandex.intranet.d.model.users.ExtendedAbcUserModel;
import ru.yandex.intranet.d.model.users.StaffAffiliation;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.model.users.UserServiceRoles;

import static org.springframework.util.CollectionUtils.isEmpty;

/**
 * Синхронизация пользователей с таблицей ABC.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 */
@Hourglass(periodInSeconds = 60)
public class SyncAbcUsers extends BaseDirectJob {
    private static final Logger LOG = LoggerFactory.getLogger(SyncAbcUsers.class);
    private static final EnumSet<UserEqualsFields> USER_EQUALS_FIELDS = EnumSet.allOf(UserEqualsFields.class);
    private static final int BATCH_SIZE = 100;
    private static final Duration TIMEOUT = Duration.ofMinutes(30);

    private final YdbTableClient tableClient;
    private final AbcIntranetStaffDao abcDao;
    private final AbcServiceMemberDao serviceMemberDao;
    private final UsersDao usersDao;
    private final Map<Long, UserServiceRoles> roleByIdMap;

    @SuppressWarnings("ParameterNumber")
    public SyncAbcUsers(
            YdbTableClient tableClient,
            AbcIntranetStaffDao abcDao,
            UsersDao usersDao,
            AbcServiceMemberDao serviceMemberDao,
            @Value("${abc.roles.quotaManager}") long quotaManagerRoleId,
            @Value("${abc.roles.responsibleOfProvider}") long responsibleOfProviderRoleId,
            @Value("${abc.roles.serviceProductHead}") long serviceProductHeadRoleId,
            @Value("${abc.roles.serviceProductDeputyHead}") long serviceProductDeputyHeadRoleId,
            @Value("${abc.roles.serviceResponsible}") long serviceResponsibleRoleId
    ) {
        this.tableClient = tableClient;
        this.abcDao = abcDao;
        this.usersDao = usersDao;
        this.serviceMemberDao = serviceMemberDao;
        roleByIdMap = Map.of(
                quotaManagerRoleId, UserServiceRoles.QUOTA_MANAGER,
                responsibleOfProviderRoleId, UserServiceRoles.RESPONSIBLE_OF_PROVIDER,
                serviceProductHeadRoleId, UserServiceRoles.SERVICE_PRODUCT_HEAD,
                serviceProductDeputyHeadRoleId, UserServiceRoles.SERVICE_PRODUCT_DEPUTY_HEAD,
                serviceResponsibleRoleId, UserServiceRoles.SERVICE_RESPONSIBLE
        );
    }

    @Override
    public void execute() {
        LOG.info("Start ABC users synchronization...");
        Instant start = Instant.now();

        Long count = tableClient.usingSessionMonoRetryable(this::doSync)
                .block(TIMEOUT);

        Instant finish = Instant.now();
        LOG.info("ABC users synchronization finished. Proceed {} items for {} seconds.",
                count,
                Duration.between(start, finish).toMillis() / 1000.
        );
    }

    private Mono<Long> doSync(YdbSession ydbSession) {
        return abcDao
                .getAllRows(
                        ydbSession,
                        AbcIntranetStaffDao.Fields.ID,
                        AbcIntranetStaffDao.Fields.UID,
                        AbcIntranetStaffDao.Fields.LOGIN,
                        AbcIntranetStaffDao.Fields.IS_DISMISSED,
                        AbcIntranetStaffDao.Fields.IS_ROBOT,
                        AbcIntranetStaffDao.Fields.AFFILIATION,
                        AbcIntranetStaffDao.Fields.FIRST_NAME_EN,
                        AbcIntranetStaffDao.Fields.FIRST_NAME,
                        AbcIntranetStaffDao.Fields.LAST_NAME_EN,
                        AbcIntranetStaffDao.Fields.LAST_NAME,
                        AbcIntranetStaffDao.Fields.GENDER,
                        AbcIntranetStaffDao.Fields.WORK_EMAIL,
                        AbcIntranetStaffDao.Fields.LANG_UI,
                        AbcIntranetStaffDao.Fields.TZ
                )
                .buffer(BATCH_SIZE)
                .concatMap(abcUserModels -> ydbSession.usingCompTxMonoRetryable(
                        TransactionMode.SERIALIZABLE_READ_WRITE,
                        ts -> usersDao.getByPassportUidsTx(ts,
                                getUserUidTenantTuple2List(abcUserModels))
                                .map(WithTxId::asTuple),
                        (ts, userModelList) -> {
                            Map<String, UserModel> userModelsByUidMap = userModelList.stream()
                                    .collect(Collectors.toMap(userModel -> userModel.getPassportUid().orElseThrow(),
                                            Function.identity()));

                            return toExtendedAbcUser(ydbSession, abcUserModels)
                                    .map(extendedAbcUserModels -> extendedAbcUserModels.stream()
                                            .filter(abcUserModel -> notEquals(abcUserModel,
                                                    userModelsByUidMap.get(abcUserModel.getUid())))
                                            .map(abcUserModel -> abcToUser(abcUserModel,
                                                    userModelsByUidMap.get(abcUserModel.getUid())))
                                            .collect(Collectors.toList())
                                    );
                        },
                        this::upsertUsersByPassportUids
                ))
                .collect(Collectors.summingLong(value -> value));
    }

    private Mono<List<ExtendedAbcUserModel>>
    toExtendedAbcUser(YdbSession ydbSession, List<AbcUserModel> users) {
        Set<Long> userIds = users.stream()
                .map(AbcUserModel::getId)
                .collect(Collectors.toSet());

        return serviceMemberDao.getByUsersAndRoles(ydbSession.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                userIds, roleByIdMap.keySet())
                .map(abcServiceMemberModels -> toExtendedAbcUserModel(users, abcServiceMemberModels));
    }

    private List<ExtendedAbcUserModel> toExtendedAbcUserModel(List<AbcUserModel> users,
                                                              List<AbcServiceMemberModel> abcServiceMemberModels) {
        Map<Long, Map<UserServiceRoles, Set<Long>>> servicesByRoleMapByStaffIdMap =
                abcServiceMemberModels.stream()
                        .collect(Collectors.groupingBy(AbcServiceMemberModel::getStaffId,
                                Collectors.groupingBy(abcServiceMemberModel -> roleByIdMap.get(
                                        abcServiceMemberModel.getRoleId()),
                                        Collectors.mapping(AbcServiceMemberModel::getServiceId,
                                                Collectors.toSet())
                                )
                        ));

        return users.stream()
                .map(abcUserModel -> ExtendedAbcUserModel.builder()
                        .abcUserModel(abcUserModel)
                        .roles(servicesByRoleMapByStaffIdMap.getOrDefault(abcUserModel.getId(),
                                Collections.emptyMap()))
                        .build())
                .collect(Collectors.toList());
    }

    private List<Tuple2<String, TenantId>> getUserUidTenantTuple2List(List<? extends AbcUserModel> abcUserModels) {
        return abcUserModels.stream()
                .map(abcUserModel -> Tuples.of(abcUserModel.getUid(), Tenants.DEFAULT_TENANT_ID))
                .collect(Collectors.toList());
    }

    private Mono<Long> upsertUsersByPassportUids(YdbTxSession session, List<UserModel> batch) {
        if (isEmpty(batch)) {
            return session.commitTransaction().thenReturn(0L);
        }
        return usersDao
                .upsertUsersByPassportUidsRetryable(session, batch)
                .thenReturn((long) batch.size());
    }

    private boolean notEquals(ExtendedAbcUserModel abcUser, UserModel userModel) {
        if (userModel == null) {
            return true;
        }

        return !USER_EQUALS_FIELDS.stream()
                .allMatch(userEqualsFiled -> userEqualsFiled.isEquals(abcUser, userModel));
    }

    private UserModel abcToUser(ExtendedAbcUserModel abcUser, UserModel userModel) {
        String passportUid = abcUser.getUid();
        String passportLogin = abcUser.getLogin();
        Long staffId = abcUser.getId();
        Boolean staffDismissed = abcUser.isDismissed();
        Boolean staffRobot = abcUser.getRobot();
        StaffAffiliation staffAffiliation =
                AbcStaffAffiliation.valueOf(abcUser.getAffiliation().toUpperCase()).toStaffAffiliation();
        String firstNameEn = abcUser.getFirstNameEn();
        String firstNameRu = abcUser.getFirstName();
        String lastNameEn = abcUser.getLastNameEn();
        String lastNameRu = abcUser.getLastName();
        Map<UserServiceRoles, Set<Long>> roles = abcUser.getRoles();
        final String gender = String.valueOf(abcUser.getGender());
        String workEmail = abcUser.getWorkEmail();
        String langUi = abcUser.getLangUi();
        String timeZone = abcUser.getTz();

        if (userModel == null) { // new user
            String id = UUID.randomUUID().toString();
            TenantId tenantId = Tenants.DEFAULT_TENANT_ID;

            return UserModel.builder()
                    .id(id)
                    .tenantId(tenantId)
                    .passportUid(passportUid)
                    .passportLogin(passportLogin)
                    .staffId(staffId)
                    .staffDismissed(staffDismissed)
                    .staffRobot(staffRobot)
                    .staffAffiliation(staffAffiliation)
                    .firstNameEn(firstNameEn)
                    .firstNameRu(firstNameRu)
                    .lastNameEn(lastNameEn)
                    .lastNameRu(lastNameRu)
                    .deleted(false)
                    .dAdmin(false)
                    .roles(roles)
                    .gender(gender)
                    .workEmail(workEmail)
                    .langUi(langUi)
                    .timeZone(timeZone)
                    .build();
        }

        return userModel.copyBuilder()
                .passportLogin(passportLogin)
                .staffId(staffId)
                .staffDismissed(staffDismissed)
                .staffRobot(staffRobot)
                .staffAffiliation(staffAffiliation)
                .firstNameEn(firstNameEn)
                .firstNameRu(firstNameRu)
                .lastNameEn(lastNameEn)
                .lastNameRu(lastNameRu)
                .roles(roles)
                .gender(gender)
                .workEmail(workEmail)
                .langUi(langUi)
                .timeZone(timeZone)
                .build();
    }

    private enum UserEqualsFields {
        PASSPORT_LOGIN((abcUserModel, userModel) -> equalsO(abcUserModel.getLogin(), userModel.getPassportLogin())),
        STAFF_ID((abcUserModel, userModel) -> equalsO(abcUserModel.getId(), userModel.getStaffId())),
        STAFF_DISMISSED((abcUserModel, userModel) -> equalsO(abcUserModel.isDismissed(),
                userModel.getStaffDismissed())),
        STAFF_ROBOT((abcUserModel, userModel) -> equalsO(abcUserModel.getRobot(), userModel.getStaffRobot())),
        STAFF_AFFILIATION((abcUserModel, userModel) -> equalsO(AbcStaffAffiliation.valueOf(abcUserModel.getAffiliation()
                .toUpperCase()).toStaffAffiliation(), userModel.getStaffAffiliation())),
        FIRST_NAME_EN((abcUserModel, userModel) -> abcUserModel.getFirstNameEn().equals(userModel.getFirstNameEn())),
        FIRST_NAME_RU((abcUserModel, userModel) -> abcUserModel.getFirstName().equals(userModel.getFirstNameRu())),
        LAST_NAME_EN((abcUserModel, userModel) -> abcUserModel.getLastNameEn().equals(userModel.getLastNameEn())),
        LAST_NAME_RU((abcUserModel, userModel) -> abcUserModel.getLastName().equals(userModel.getLastNameRu())),
        ROLES((abcUserModel, userModel) -> abcUserModel.getRoles().equals(userModel.getRoles())),
        GENDER((abcUserModel, userModel) -> String.valueOf(abcUserModel.getGender()).equals(userModel.getGender())),
        WORK_EMAIL((abcUserModel, userModel) -> Objects.equals(abcUserModel.getWorkEmail(),
                userModel.getWorkEmail().orElse(null))),
        LANG_UI((abcUserModel, userModel) -> Objects.equals(abcUserModel.getLangUi(),
                userModel.getLangUi().orElse(null))),
        TIME_ZONE((abcUserModel, userModel) -> Objects.equals(abcUserModel.getTz(),
                userModel.getTimeZone().orElse(null)));

        @SuppressWarnings("ImmutableEnumChecker")
        private final BiFunction<ExtendedAbcUserModel, UserModel, Boolean> eqBiFunc;

        UserEqualsFields(BiFunction<ExtendedAbcUserModel, UserModel, Boolean> eqBiFunc) {
            this.eqBiFunc = eqBiFunc;
        }

        public boolean isEquals(ExtendedAbcUserModel abcUser, UserModel userModel) {
            return eqBiFunc.apply(abcUser, userModel);
        }
    }

    @SuppressWarnings("OptionalUsedAsFieldOrParameterType")
    private static <V> boolean equalsO(V fromAbcUser, Optional<V> fromUserModel) {
        return fromAbcUser.equals(fromUserModel.orElse(null));
    }
}
