package ru.yandex.intranet.d.tms.jobs;

import java.time.Clock;

import com.yandex.ydb.table.transaction.TransactionMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.i18n.Locales;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.sync.ProvidersSyncStatusModel;
import ru.yandex.intranet.d.services.sync.AccountsSyncService;
import ru.yandex.intranet.d.tms.SyncAccountsParametersSource;

/**
 * Cron job to sync providers accounts and quotas.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Hourglass(periodInSeconds = 1800)
@ParameterizedBy(parametersSource = SyncAccountsParametersSource.class)
public class SyncAccountsJob extends DirectParameterizedJob<String> {

    private static final Logger LOG = LoggerFactory.getLogger(SyncAccountsJob.class);

    private final YdbTableClient tableClient;
    private final ProvidersDao providersDao;
    private final AccountsSyncService accountsSyncService;

    public SyncAccountsJob(
            YdbTableClient tableClient,
            ProvidersDao providersDao,
            AccountsSyncService accountsSyncService
    ) {
        this.tableClient = tableClient;
        this.providersDao = providersDao;
        this.accountsSyncService = accountsSyncService;
    }

    @Override
    public void execute() {
        String providerId = getParam();
        syncOneProvider(providerId)
                .doOnError(e -> LOG.error("Failed to sync provider " + providerId, e))
                .onErrorResume(v -> Mono.empty())
                .block();
    }

    private Mono<ProvidersSyncStatusModel> syncOneProvider(String providerId) {
        return tableClient.usingSessionMonoRetryable(session -> session
                .usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, txSession ->
                        providersDao.getById(txSession, providerId, Tenants.DEFAULT_TENANT_ID))
                .flatMap(providerOptional -> providerOptional.map(provider -> {
                            if (!eligibleForSync(provider)) {
                                LOG.info("Sync disabled for provider {} ", provider.getKey());
                                return Mono.<ProvidersSyncStatusModel>empty();
                            }
                            LOG.info("Start sync provider {}", provider.getKey());
                            return accountsSyncService.syncOneProvider(provider, Locales.ENGLISH, Clock.systemUTC());
                        }).orElse(Mono.error(() -> new IllegalStateException("Provider not found " + providerId)))
                )
        );
    }

    private boolean eligibleForSync(ProviderModel provider) {
        return (provider.getRestApiUri().isPresent() || provider.getGrpcApiUri().isPresent())
                && provider.isSyncEnabled() && !provider.isDeleted();
    }
}
