package ru.yandex.intranet.d.util;

import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import javax.annotation.Nullable;

import org.springframework.context.MessageSource;

import ru.yandex.intranet.d.backend.service.recipe_proto.AggregationAlgorithm;
import ru.yandex.intranet.d.model.providers.AggregationQuotaQueryType;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.providers.AggregationAlgorithmDto;
import ru.yandex.intranet.d.web.model.providers.AggregationAlgorithmInputDto;
import ru.yandex.intranet.d.web.model.providers.AggregationQuotaQueryTypeInputDto;

/**
 * Aggregation algorithm helper.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class AggregationAlgorithmHelper {

    public static final boolean DEFAULT_DISABLED_FLAG = false;
    public static final long DEFAULT_UPSERT_PAGE_SIZE = 1000L;
    public static final AggregationQuotaQueryType DEFAULT_QUOTA_QUERY_TYPE = AggregationQuotaQueryType.PAGINATE;
    public static final boolean DEFAULT_CONCURRENT_AGGREGATES_UPSERT = false;
    public static final boolean DEFAULT_CONCURRENT_AGGREGATION = false;
    public static final long DEFAULT_RESOURCES_PAGE_SIZE = 3L;
    public static final boolean DEFAULT_PAGINATE_RESOURCES = true;
    public static final boolean DEFAULT_CONCURRENT_CLEANUP = false;

    private AggregationAlgorithmHelper() {
    }

    @Nullable
    public static AggregationAlgorithmInputDto fromAggregationAlgorithm(@Nullable AggregationAlgorithm algorithm) {
        if (algorithm == null) {
            return null;
        }
        return new AggregationAlgorithmInputDto(algorithm.getDisabled(), algorithm.getUpsertPageSize(),
                AggregationQuotaQueryTypeHelper.fromAggregationQuotaQueryType(algorithm.getQuotaQueryType()),
                algorithm.getConcurrentAggregatesUpsert(), algorithm.getConcurrentAggregation(),
                algorithm.getResourcesPageSize(), algorithm.getPaginateResources(), algorithm.getConcurrentCleanup());
    }

    @Nullable
    public static AggregationAlgorithm toAggregationAlgorithm(
            @Nullable ru.yandex.intranet.d.model.providers.AggregationAlgorithm algorithm) {
        if (algorithm == null) {
            return null;
        }
        return AggregationAlgorithm.newBuilder()
                .setDisabled(Objects.requireNonNullElse(algorithm.getDisabled(), DEFAULT_DISABLED_FLAG))
                .setUpsertPageSize(Objects.requireNonNullElse(algorithm.getUpsertPageSize(), DEFAULT_UPSERT_PAGE_SIZE))
                .setQuotaQueryType(AggregationQuotaQueryTypeHelper.toAggregationQuotaQueryType(
                        Objects.requireNonNullElse(algorithm.getQuotaQueryType(), DEFAULT_QUOTA_QUERY_TYPE)))
                .setConcurrentAggregatesUpsert(Objects.requireNonNullElse(algorithm.getConcurrentAggregatesUpsert(),
                        DEFAULT_CONCURRENT_AGGREGATES_UPSERT))
                .setConcurrentAggregation(Objects.requireNonNullElse(algorithm.getConcurrentAggregation(),
                        DEFAULT_CONCURRENT_AGGREGATION))
                .setResourcesPageSize(Objects.requireNonNullElse(algorithm.getResourcesPageSize(),
                        DEFAULT_RESOURCES_PAGE_SIZE))
                .setPaginateResources(Objects.requireNonNullElse(algorithm.getPaginateResources(),
                        DEFAULT_PAGINATE_RESOURCES))
                .setConcurrentCleanup(Objects.requireNonNullElse(algorithm.getConcurrentCleanup(),
                        DEFAULT_CONCURRENT_CLEANUP))
                .build();
    }

    @Nullable
    public static AggregationAlgorithmDto toAggregationAlgorithmDto(
            @Nullable ru.yandex.intranet.d.model.providers.AggregationAlgorithm algorithm) {
        if (algorithm == null) {
            return null;
        }
        return new AggregationAlgorithmDto(Objects.requireNonNullElse(algorithm.getDisabled(), DEFAULT_DISABLED_FLAG),
                Objects.requireNonNullElse(algorithm.getUpsertPageSize(), DEFAULT_UPSERT_PAGE_SIZE),
                AggregationQuotaQueryTypeHelper.toAggregationQuotaQueryTypeDto(Objects.requireNonNullElse(
                        algorithm.getQuotaQueryType(), DEFAULT_QUOTA_QUERY_TYPE)),
                Objects.requireNonNullElse(algorithm.getConcurrentAggregatesUpsert(),
                        DEFAULT_CONCURRENT_AGGREGATES_UPSERT),
                Objects.requireNonNullElse(algorithm.getConcurrentAggregation(), DEFAULT_CONCURRENT_AGGREGATION),
                Objects.requireNonNullElse(algorithm.getResourcesPageSize(), DEFAULT_RESOURCES_PAGE_SIZE),
                Objects.requireNonNullElse(algorithm.getPaginateResources(), DEFAULT_PAGINATE_RESOURCES),
                Objects.requireNonNullElse(algorithm.getConcurrentCleanup(), DEFAULT_CONCURRENT_CLEANUP));
    }

    @Nullable
    public static AggregationAlgorithmInputDto toAggregationAlgorithmInputDto(
            @Nullable AggregationAlgorithmDto algorithm) {
        if (algorithm == null) {
            return null;
        }
        return new AggregationAlgorithmInputDto(
                algorithm.getDisabled(),
                algorithm.getUpsertPageSize(),
                AggregationQuotaQueryTypeHelper.toAggregationQuotaQueryTypeInputDto(algorithm.getQuotaQueryType()),
                algorithm.getConcurrentAggregatesUpsert(),
                algorithm.getConcurrentAggregation(),
                algorithm.getResourcesPageSize(),
                algorithm.getPaginateResources(),
                algorithm.getConcurrentCleanup());
    }

    public static void validateAggregationAlgorithm(
            Supplier<Optional<AggregationAlgorithmInputDto>> getter,
            Consumer<ru.yandex.intranet.d.model.providers.AggregationAlgorithm> setter,
            ErrorCollection.Builder errors, String fieldKey, Locale locale, MessageSource messages) {
        Optional<AggregationAlgorithmInputDto> valueO = getter.get();
        if (valueO.isEmpty()) {
            setter.accept(null);
            return;
        }
        AggregationAlgorithmInputDto value = valueO.get();
        ErrorCollection.Builder localErrors = ErrorCollection.builder();
        if (value.getDisabled() == null) {
            localErrors.addError(fieldKey + ".disabled", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        }
        if (value.getUpsertPageSize() == null) {
            localErrors.addError(fieldKey + ".upsertPageSize", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (value.getUpsertPageSize() <= 0) {
            localErrors.addError(fieldKey + ".upsertPageSize", TypedError.invalid(messages
                    .getMessage("errors.number.must.be.positive", null, locale)));
        }
        if (value.getConcurrentAggregatesUpsert() == null) {
            localErrors.addError(fieldKey + ".concurrentAggregatesUpsert", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        }
        if (value.getConcurrentAggregation() == null) {
            localErrors.addError(fieldKey + ".concurrentAggregation", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        }
        if (value.getResourcesPageSize() == null) {
            localErrors.addError(fieldKey + ".resourcesPageSize", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (value.getResourcesPageSize() <= 0) {
            localErrors.addError(fieldKey + ".resourcesPageSize", TypedError.invalid(messages
                    .getMessage("errors.number.must.be.positive", null, locale)));
        }
        if (value.getPaginateResources() == null) {
            localErrors.addError(fieldKey + ".paginateResources", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        }
        if (value.getConcurrentCleanup() == null) {
            localErrors.addError(fieldKey + ".concurrentCleanup", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        }
        AggregationQuotaQueryType queryType = null;
        if (value.getQuotaQueryType() == null) {
            localErrors.addError(fieldKey + ".quotaQueryType", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            AggregationQuotaQueryTypeInputDto type = value.getQuotaQueryType();
            if (type == AggregationQuotaQueryTypeInputDto.UNKNOWN) {
                localErrors.addError(fieldKey + ".quotaQueryType", TypedError.invalid(messages
                        .getMessage("errors.invalid.value", null, locale)));
            } else {
                queryType = AggregationQuotaQueryTypeHelper.toAggregationQuotaQueryType(type);
            }
        }
        if (localErrors.hasAnyErrors()) {
            errors.add(localErrors);
            return;
        }
        setter.accept(new ru.yandex.intranet.d.model.providers.AggregationAlgorithm(value.getDisabled(),
                value.getUpsertPageSize(), queryType, value.getConcurrentAggregatesUpsert(),
                value.getConcurrentAggregation(), value.getResourcesPageSize(), value.getPaginateResources(),
                value.getConcurrentCleanup()));
    }

}
