package ru.yandex.intranet.d.util;

import java.util.Locale;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import javax.annotation.Nullable;

import org.springframework.context.MessageSource;

import ru.yandex.intranet.d.backend.service.proto.AggregationSettings;
import ru.yandex.intranet.d.backend.service.proto.TimeSeriesGrid;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.providers.AggregationSettingsDto;
import ru.yandex.intranet.d.web.model.providers.AggregationSettingsInputDto;
import ru.yandex.intranet.d.web.model.providers.FreeProvisionAggregationModeInputDto;
import ru.yandex.intranet.d.web.model.providers.UsageModeInputDto;

/**
 * Aggregation settings helper.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class AggregationSettingsHelper {

    private AggregationSettingsHelper() {
    }

    @Nullable
    public static AggregationSettingsInputDto fromAggregationSettings(@Nullable AggregationSettings settings) {
        if (settings == null) {
            return null;
        }
        return new AggregationSettingsInputDto(
                AggregationModeHelper.fromAggregationMode(settings.getFreeProvisionMode()),
                UsageModeHelper.fromUsageMode(settings.getUsageMode()),
                settings.hasTimeSeriesGrid() ? settings.getTimeSeriesGrid().getTimeSeriesGridSpacingSeconds() : null
        );
    }

    @Nullable
    public static AggregationSettings toAggregationSettings(
            @Nullable ru.yandex.intranet.d.model.providers.AggregationSettings settings) {
        if (settings == null) {
            return null;
        }
        AggregationSettings.Builder builder = AggregationSettings.newBuilder()
                .setFreeProvisionMode(AggregationModeHelper.toAggregationMode(settings.getFreeProvisionMode()))
                .setUsageMode(UsageModeHelper.toUsageMode(settings.getUsageMode()));
        if (settings.getTimeSeriesGridSpacingSeconds() != null) {
            builder.setTimeSeriesGrid(TimeSeriesGrid.newBuilder()
                    .setTimeSeriesGridSpacingSeconds(settings.getTimeSeriesGridSpacingSeconds())
                    .build());
        }
        return builder.build();
    }

    @Nullable
    public static AggregationSettingsDto toAggregationSettingsDto(
            @Nullable ru.yandex.intranet.d.model.providers.AggregationSettings settings) {
        if (settings == null) {
            return null;
        }
        return new AggregationSettingsDto(
                AggregationModeHelper.toAggregationModeDto(settings.getFreeProvisionMode()),
                UsageModeHelper.toUsageModeDto(settings.getUsageMode()),
                settings.getTimeSeriesGridSpacingSeconds()
        );
    }

    @Nullable
    public static AggregationSettingsInputDto toAggregationSettingsInputDto(@Nullable AggregationSettingsDto settings) {
        if (settings == null) {
            return null;
        }
        return new AggregationSettingsInputDto(
                AggregationModeHelper.toAggregationModeInputDto(settings.getFreeProvisionMode()),
                UsageModeHelper.toUsageModeInputDto(settings.getUsageMode()),
                settings.getTimeSeriesGridSpacingSeconds()
        );
    }

    public static void validateAggregationSettings(
            Supplier<Optional<AggregationSettingsInputDto>> getter,
            Consumer<ru.yandex.intranet.d.model.providers.AggregationSettings> setter,
            ErrorCollection.Builder errors, String fieldKey, Locale locale, MessageSource messages) {
        Optional<AggregationSettingsInputDto> valueO = getter.get();
        if (valueO.isEmpty()) {
            setter.accept(null);
            return;
        }
        AggregationSettingsInputDto value = valueO.get();
        boolean noErrors = true;
        FreeProvisionAggregationModeInputDto mode = null;
        if (value.getFreeProvisionMode() == null) {
            errors.addError(fieldKey + ".freeProvisionMode", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            noErrors = false;
        } else {
            mode = value.getFreeProvisionMode();
            if (mode == FreeProvisionAggregationModeInputDto.UNKNOWN) {
                errors.addError(fieldKey + ".freeProvisionMode", TypedError.invalid(messages
                        .getMessage("errors.invalid.value", null, locale)));
                noErrors = false;
            }
        }
        UsageModeInputDto usageMode = null;
        if (value.getUsageMode() == null) {
            errors.addError(fieldKey + ".usageMode", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            noErrors = false;
        } else {
            usageMode = value.getUsageMode();
            if (usageMode == UsageModeInputDto.UNKNOWN) {
                errors.addError(fieldKey + ".usageMode", TypedError.invalid(messages
                        .getMessage("errors.invalid.value", null, locale)));
                noErrors = false;
            }
        }
        if (value.getTimeSeriesGridSpacingSeconds() != null && value.getTimeSeriesGridSpacingSeconds() <= 0L) {
            errors.addError(fieldKey + ".timeSeriesGridSpacingSeconds", TypedError.invalid(messages
                    .getMessage("errors.number.must.be.positive", null, locale)));
            noErrors = false;
        }
        if (noErrors) {
            setter.accept(new ru.yandex.intranet.d.model.providers.AggregationSettings(
                    AggregationModeHelper.toAggregationMode(mode),
                    UsageModeHelper.toUsageMode(usageMode),
                    value.getTimeSeriesGridSpacingSeconds()
            ));
        }
    }

}
