package ru.yandex.intranet.d.util;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiConsumer;
import java.util.function.Predicate;

import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.publisher.SignalType;

/**
 * Metrics for async methods.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class AsyncMetrics {

    private AsyncMetrics() {
    }

    public static <T> Mono<T> metric(Mono<T> mono, BiConsumer<Long, Boolean> metricConsumer) {
        AtomicReference<Long> startTime = new AtomicReference<>();
        return mono.doOnSubscribe(s -> startTime.set(System.nanoTime()))
                .doFinally(s -> metricConsumer.accept(durationMillis(startTime), isSuccess(s)));
    }

    public static <T> Mono<T> metric(Mono<T> mono, Predicate<T> successCheck,
                                     BiConsumer<Long, Boolean> metricConsumer) {
        AtomicReference<Long> startTime = new AtomicReference<>();
        return mono.doOnSubscribe(s -> startTime.set(System.nanoTime()))
                .doOnSuccess(v -> metricConsumer.accept(durationMillis(startTime), successCheck.test(v)))
                .doOnError(e -> metricConsumer.accept(durationMillis(startTime), false))
                .doOnCancel(() -> metricConsumer.accept(durationMillis(startTime), false));
    }

    public static <T> Flux<T> metric(Flux<T> mono, BiConsumer<Long, Boolean> metricConsumer) {
        AtomicReference<Long> startTime = new AtomicReference<>();
        return mono.doOnSubscribe(s -> startTime.set(System.nanoTime()))
                .doFinally(s -> metricConsumer.accept(durationMillis(startTime), isSuccess(s)));
    }

    private static long durationMillis(AtomicReference<Long> startTime) {
        Long maybeStartTime = startTime.get();
        long currentTime = System.nanoTime();
        long startTimeValue = maybeStartTime != null ? maybeStartTime : currentTime;
        return TimeUnit.MILLISECONDS.convert(currentTime - startTimeValue, TimeUnit.NANOSECONDS);
    }

    private static boolean isSuccess(SignalType signalType) {
        return SignalType.ON_COMPLETE.equals(signalType);
    }

}
