package ru.yandex.intranet.d.util;

import java.util.Objects;
import java.util.function.Consumer;

import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;
import it.unimi.dsi.fastutil.longs.LongSet;

/**
 * Long to Long multimap.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 03.11.2020
 */
public class Long2LongMultimap {
    private final Long2ObjectOpenHashMap<LongSet> map = new Long2ObjectOpenHashMap<>();

    public synchronized Long2LongMultimap put(long key, long value) {
        map
                .computeIfAbsent(key, ignored -> new LongOpenHashSet())
                .add(value);
        return this;
    }

    public synchronized void forEach(LongLongSetBiConsumer consumer) {
        map.long2ObjectEntrySet().fastForEach(entry ->
                consumer.accept(entry.getLongKey(), entry.getValue())
        );
    }

    public synchronized void forEachBatch(int batchSize, Consumer<Long2LongMultimap> consumer) {
        Long2LongMultimap batch = new Long2LongMultimap();
        forEach((key, value) -> {
            batch.map.put(key, value);
            if (batch.map.size() >= batchSize) {
                consumer.accept(batch);
                batch.map.clear();
            }
        });
        if (batch.map.size() > 0) {
            consumer.accept(batch);
            batch.map.clear();
        }
    }

    public int size() {
        return map.size();
    }

    public LongSet get(long key) {
        return map.get(key);
    }

    public void resetAll(long key, LongOpenHashSet parents) {
        map.put(key, parents);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Long2LongMultimap)) {
            return false;
        }
        Long2LongMultimap that = (Long2LongMultimap) o;
        return that.map.equals(map);
    }

    @Override
    public int hashCode() {
        return Objects.hash(map);
    }

    @Override
    public String toString() {
        return map.toString();
    }

    @FunctionalInterface
    public interface LongLongSetBiConsumer {
        /**
         * Performs this operation on the given argument.
         *
         * @param value the input argument
         */
        void accept(long key, LongSet value);
    }
}
