package ru.yandex.intranet.d.util.bucket

import com.google.common.util.concurrent.ThreadFactoryBuilder
import kotlinx.coroutines.runBlocking
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.model.settings.RateLimiterSettings
import ru.yandex.intranet.d.services.settings.RuntimeSettingsService
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.ScheduledThreadPoolExecutor
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicBoolean
import javax.annotation.PostConstruct
import javax.annotation.PreDestroy

private val logger = KotlinLogging.logger {}

@Component
class RateLimiterBreaker(
    private val runtimeSettingsService: RuntimeSettingsService,
    @Value("\${limiter.settings.schedulerShutdownTimeoutMs}") private val schedulerShutdownTimeoutMs: Long
) {

    private val disabled: AtomicBoolean = AtomicBoolean(false)
    private val scheduler: ScheduledExecutorService

    init {
        val threadFactory = ThreadFactoryBuilder()
            .setDaemon(true)
            .setNameFormat("rate-limiter-refresh-pool-%d")
            .setUncaughtExceptionHandler { t: Thread, e: Throwable? -> logger.error(e) {"Uncaught exception in scheduler thread $t" } }
            .build()
        val scheduledThreadPoolExecutor = ScheduledThreadPoolExecutor(2, threadFactory)
        scheduledThreadPoolExecutor.removeOnCancelPolicy = true
        scheduler = scheduledThreadPoolExecutor
    }

    fun rateLimiterEnabled(): Boolean {
        return !disabled.get()
    }

    @PostConstruct
    fun postConstruct() {
        logger.debug { "Preparing rate limiter settings..." }
        var settingsLoaded = false
        try {
            val settings = loadSettings()
            settingsLoaded = if (settings != null) {
                disabled.set(settings.disabled)
                true
            } else {
                false
            }
        } catch (e: Exception) {
            logger.error(e) { "Failed to load rate limiter settings" }
        } finally {
            if (settingsLoaded) {
                logger.debug { "Rate limiter settings successfully loaded" }
                scheduler.schedule({ reloadSettings() }, 1, TimeUnit.MINUTES)
            } else {
                logger.debug {"Failed to load rate limiter settings, rescheduling..." }
                scheduler.schedule({ reloadSettings() }, 5, TimeUnit.SECONDS)
            }
        }
    }

    @PreDestroy
    fun preDestroy() {
        logger.debug { "Stopping rate limiter settings loader..." }
        scheduler.shutdown()
        try {
            scheduler.awaitTermination(schedulerShutdownTimeoutMs, TimeUnit.MILLISECONDS)
        } catch (e: InterruptedException) {
            Thread.currentThread().interrupt()
        }
        scheduler.shutdownNow()
        logger.debug { "Stopped rate limiter settings loader" }
    }

    private fun reloadSettings() {
        logger.debug { "Reloading rate limiter settings..." }
        var settingsLoaded = false
        try {
            val settings = loadSettings()
            settingsLoaded = if (settings != null) {
                disabled.set(settings.disabled)
                true
            } else {
                false
            }
        } catch (e: Exception) {
            logger.error(e) { "Failed to reload rate limiter settings" }
        } finally {
            if (settingsLoaded) {
                logger.debug {"Rate limiter settings successfully loaded" }
                scheduler.schedule({ reloadSettings() }, 1, TimeUnit.MINUTES)
            } else {
                logger.debug { "Failed to load rate limiter settings, rescheduling..." }
                scheduler.schedule({ reloadSettings() }, 5, TimeUnit.SECONDS)
            }
        }
    }

    private fun loadSettings(): RateLimiterSettings? {
        return runBlocking {
            try {
                runtimeSettingsService.getRateLimiterSettingsImmediate(Tenants.DEFAULT_TENANT_ID)
            } catch (e: Exception) {
                logger.error(e) { "Failed to load rate limiter settings" }
                null
            }
        }
    }

}
