package ru.yandex.intranet.d.util.dispatchers

import com.google.common.util.concurrent.ThreadFactoryBuilder
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.ExecutorCoroutineDispatcher
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.asCoroutineDispatcher
import kotlinx.coroutines.cancel
import kotlinx.coroutines.launch
import kotlinx.coroutines.slf4j.MDCContext
import mu.KotlinLogging
import org.springframework.context.annotation.Profile
import org.springframework.stereotype.Component
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.ScheduledThreadPoolExecutor
import java.util.concurrent.TimeUnit
import javax.annotation.PreDestroy

private val logger = KotlinLogging.logger {}

/**
 *
 * @author Mustakayev Marat <mmarat248@yandex-team.ru>
 */
@Component("operationRequestLogDispatcher")
@Profile("local", "dev", "testing", "load-testing", "production")
class OperationRequestLogDispatcherImpl: CustomDispatcher {
    private val scheduler: ScheduledExecutorService
    private val dispatcher: ExecutorCoroutineDispatcher
    private val coroutinesScope: CoroutineScope

    init {
        val threadFactory = ThreadFactoryBuilder()
            .setDaemon(true)
            .setNameFormat("operation-request-log-pool-%d")
            .setUncaughtExceptionHandler { t: Thread, e: Throwable? ->
                logger.error(e) { "Uncaught exception in operation request log thread $t" }
            }
            .build()
        val scheduledThreadPoolExecutor = ScheduledThreadPoolExecutor(
            (Runtime.getRuntime().availableProcessors() / 4).coerceAtLeast(2), threadFactory)
        scheduledThreadPoolExecutor.removeOnCancelPolicy = true
        scheduler = scheduledThreadPoolExecutor
        dispatcher = scheduledThreadPoolExecutor.asCoroutineDispatcher()
        val exceptionHandler = CoroutineExceptionHandler { _, exception ->
            logger.error(exception) { "Uncaught exception in operation request log pool" }
        }
        coroutinesScope = CoroutineScope(dispatcher + exceptionHandler + SupervisorJob())
    }

    @PreDestroy
    fun preDestroy() {
        logger.info("Stopping operation request log dispatcher...")
        coroutinesScope.cancel("Application context shutdown")
        dispatcher.close()
        try {
            scheduler.awaitTermination(1, TimeUnit.SECONDS)
        } catch (e: InterruptedException) {
            Thread.currentThread().interrupt()
        }
        scheduler.shutdownNow()
        logger.info("Stopped operation request log dispatcher")
    }

    override fun launch(block: suspend () -> Unit): Job {
        return coroutinesScope.launch(MDCContext()) { block() }
    }
}
