package ru.yandex.intranet.d.util.http;

import java.util.AbstractSet;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import io.netty.handler.codec.http.HttpHeaders;
import org.jetbrains.annotations.NotNull;
import org.springframework.lang.NonNull;
import org.springframework.util.CollectionUtils;
import org.springframework.util.MultiValueMap;

/**
 * Companion netty headers adapter for customized reactor client http connector
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class YaNettyHeadersAdapter implements MultiValueMap<String, String> {

    private final HttpHeaders headers;

    public YaNettyHeadersAdapter(HttpHeaders headers) {
        this.headers = headers;
    }

    @Override
    @Nullable
    public String getFirst(@NonNull String key) {
        return headers.get(key);
    }

    @Override
    public void add(@NonNull String key, @Nullable String value) {
        if (value != null) {
            headers.add(key, value);
        }
    }

    @Override
    public void addAll(@NonNull String key, @NonNull List<? extends String> values) {
        headers.add(key, values);
    }

    @Override
    public void addAll(MultiValueMap<String, String> values) {
        values.forEach(headers::add);
    }

    @Override
    public void set(@NonNull String key, @Nullable String value) {
        if (value != null) {
            headers.set(key, value);
        }
    }

    @Override
    public void setAll(Map<String, String> values) {
        values.forEach(this.headers::set);
    }

    @Override
    @NonNull
    public Map<String, String> toSingleValueMap() {
        Map<String, String> singleValueMap = CollectionUtils.newLinkedHashMap(headers.size());
        headers.entries().forEach(entry -> {
            if (!singleValueMap.containsKey(entry.getKey())) {
                singleValueMap.put(entry.getKey(), entry.getValue());
            }
        });
        return singleValueMap;
    }

    @Override
    public int size() {
        return headers.names().size();
    }

    @Override
    public boolean isEmpty() {
        return headers.isEmpty();
    }

    @Override
    public boolean containsKey(Object key) {
        return (key instanceof String && headers.contains((String) key));
    }

    @Override
    public boolean containsValue(Object value) {
        return (value instanceof String && headers.entries()
                .stream().anyMatch(entry -> value.equals(entry.getValue())));
    }

    @Override
    @Nullable
    public List<String> get(Object key) {
        if (containsKey(key)) {
            return headers.getAll((String) key);
        }
        return null;
    }

    @Override
    @Nullable
    public List<String> put(String key, @Nullable List<String> value) {
        List<String> previousValues = headers.getAll(key);
        headers.set(key, value);
        return previousValues;
    }

    @Override
    @Nullable
    public List<String> remove(Object key) {
        if (key instanceof String) {
            List<String> previousValues = headers.getAll((String) key);
            headers.remove((String) key);
            return previousValues;
        }
        return null;
    }

    @Override
    public void putAll(Map<? extends String, ? extends List<String>> m) {
        m.forEach(headers::set);
    }

    @Override
    public void clear() {
        headers.clear();
    }

    @NotNull
    @Override
    public Set<String> keySet() {
        return new HeaderNames();
    }

    @NotNull
    @Override
    public Collection<List<String>> values() {
        return headers.names().stream().map(headers::getAll).collect(Collectors.toList());
    }

    @NotNull
    @Override
    public Set<Entry<String, List<String>>> entrySet() {
        return new AbstractSet<>() {
            @Override
            public Iterator<Entry<String, List<String>>> iterator() {
                return new EntryIterator();
            }
            @Override
            public int size() {
                return headers.size();
            }
        };
    }

    @Override
    public String toString() {
        return org.springframework.http.HttpHeaders.formatHeaders(this);
    }

    private class EntryIterator implements Iterator<Entry<String, List<String>>> {

        private final Iterator<String> names = headers.names().iterator();

        @Override
        public boolean hasNext() {
            return names.hasNext();
        }

        @Override
        public Entry<String, List<String>> next() {
            return new HeaderEntry(names.next());
        }

    }


    private class HeaderEntry implements Entry<String, List<String>> {

        private final String key;

        private HeaderEntry(String key) {
            this.key = key;
        }

        @Override
        public String getKey() {
            return key;
        }

        @Override
        public List<String> getValue() {
            return headers.getAll(key);
        }

        @Override
        public List<String> setValue(List<String> value) {
            List<String> previousValues = headers.getAll(key);
            headers.set(key, value);
            return previousValues;
        }

    }

    private class HeaderNames extends AbstractSet<String> {

        @Override
        @NonNull
        public Iterator<String> iterator() {
            return new HeaderNamesIterator(headers.names().iterator());
        }

        @Override
        public int size() {
            return headers.names().size();
        }
    }

    private final class HeaderNamesIterator implements Iterator<String> {

        private final Iterator<String> iterator;

        @Nullable
        private String currentName;

        private HeaderNamesIterator(Iterator<String> iterator) {
            this.iterator = iterator;
        }

        @Override
        public boolean hasNext() {
            return iterator.hasNext();
        }

        @Override
        public String next() {
            currentName = iterator.next();
            return currentName;
        }

        @Override
        public void remove() {
            if (currentName == null) {
                throw new IllegalStateException("No current Header in iterator");
            }
            if (!headers.contains(currentName)) {
                throw new IllegalStateException("Header not present: " + currentName);
            }
            headers.remove(currentName);
        }

    }

}
