package ru.yandex.intranet.d.util.http;

import java.net.URI;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;

import org.springframework.http.HttpMethod;
import org.springframework.http.client.reactive.ClientHttpConnector;
import org.springframework.http.client.reactive.ClientHttpRequest;
import org.springframework.http.client.reactive.ClientHttpResponse;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;
import reactor.core.publisher.Mono;
import reactor.netty.NettyOutbound;
import reactor.netty.http.client.HttpClient;
import reactor.netty.http.client.HttpClientRequest;

/**
 * Customized reactor client http connector
 * Fixes unnecessary URI to string conversion which triggers a bug for URIs with single digit ports.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class YaReactorClientHttpConnector implements ClientHttpConnector {

    private final HttpClient httpClient;

    public YaReactorClientHttpConnector(HttpClient httpClient) {
        Assert.notNull(httpClient, "HttpClient is required");
        this.httpClient = httpClient;
    }

    @NonNull
    @Override
    public Mono<ClientHttpResponse> connect(
            @NonNull HttpMethod method,
            @NonNull URI uri,
            @NonNull Function<? super ClientHttpRequest, Mono<Void>> requestCallback) {
        AtomicReference<YaReactorClientHttpResponse> responseRef = new AtomicReference<>();
        return this.httpClient
                .request(io.netty.handler.codec.http.HttpMethod.valueOf(method.name()))
                .uri(uri)
                .send((request, outbound) -> requestCallback.apply(wrapRequest(method, uri, request, outbound)))
                .responseConnection((response, connection) -> {
                    responseRef.set(new YaReactorClientHttpResponse(response, connection));
                    return Mono.just((ClientHttpResponse) responseRef.get());
                })
                .next()
                .doOnCancel(() -> {
                    YaReactorClientHttpResponse response = responseRef.get();
                    if (response != null) {
                        response.releaseAfterCancel(method);
                    }
                });
    }

    private YaReactorClientHttpRequest wrapRequest(HttpMethod method, URI uri, HttpClientRequest request,
                                                   NettyOutbound nettyOutbound) {
        return new YaReactorClientHttpRequest(method, uri, request, nettyOutbound);
    }

}
