package ru.yandex.intranet.d.util.http;

import java.net.URI;
import java.nio.file.Path;
import java.util.Collection;

import io.netty.buffer.ByteBuf;
import io.netty.handler.codec.http.cookie.DefaultCookie;
import org.reactivestreams.Publisher;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferFactory;
import org.springframework.core.io.buffer.NettyDataBufferFactory;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ZeroCopyHttpOutputMessage;
import org.springframework.http.client.reactive.AbstractClientHttpRequest;
import org.springframework.lang.NonNull;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.netty.NettyOutbound;
import reactor.netty.http.client.HttpClientRequest;

/**
 * Companion http request for customized reactor client http connector
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class YaReactorClientHttpRequest extends AbstractClientHttpRequest implements ZeroCopyHttpOutputMessage {

    private final HttpMethod httpMethod;
    private final URI uri;
    private final HttpClientRequest request;
    private final NettyOutbound outbound;
    private final NettyDataBufferFactory bufferFactory;

    public YaReactorClientHttpRequest(HttpMethod method, URI uri, HttpClientRequest request, NettyOutbound outbound) {
        this.httpMethod = method;
        this.uri = uri;
        this.request = request;
        this.outbound = outbound;
        this.bufferFactory = new NettyDataBufferFactory(outbound.alloc());
    }

    @NonNull
    @Override
    public Mono<Void> writeWith(@NonNull Path file, long position, long count) {
        return doCommit(() -> outbound.sendFile(file, position, count).then());
    }

    @NonNull
    @Override
    public HttpMethod getMethod() {
        return httpMethod;
    }

    @NonNull
    @Override
    public URI getURI() {
        return uri;
    }

    @NonNull
    @Override
    @SuppressWarnings("unchecked")
    public <T> T getNativeRequest() {
        return (T) this.request;
    }

    @NonNull
    @Override
    public DataBufferFactory bufferFactory() {
        return bufferFactory;
    }

    @NonNull
    @Override
    public Mono<Void> writeWith(@NonNull Publisher<? extends DataBuffer> body) {
        return doCommit(() -> {
            if (body instanceof Mono) {
                Mono<ByteBuf> byteBufMono = Mono.from(body).map(NettyDataBufferFactory::toByteBuf);
                return outbound.send(byteBufMono).then();
            } else {
                Flux<ByteBuf> byteBufFlux = Flux.from(body).map(NettyDataBufferFactory::toByteBuf);
                return outbound.send(byteBufFlux).then();
            }
        });
    }

    @NonNull
    @Override
    public Mono<Void> writeAndFlushWith(@NonNull Publisher<? extends Publisher<? extends DataBuffer>> body) {
        Publisher<Publisher<ByteBuf>> byteBuffs = Flux.from(body).map(YaReactorClientHttpRequest::toByteBuffs);
        return doCommit(() -> outbound.sendGroups(byteBuffs).then());
    }

    @NonNull
    @Override
    public Mono<Void> setComplete() {
        return doCommit(outbound::then);
    }

    @Override
    protected void applyHeaders() {
        getHeaders().forEach((key, value) -> request.requestHeaders().set(key, value));
    }

    @Override
    protected void applyCookies() {
        getCookies().values().stream().flatMap(Collection::stream)
                .map(cookie -> new DefaultCookie(cookie.getName(), cookie.getValue()))
                .forEach(request::addCookie);
    }

    @NonNull
    @Override
    protected HttpHeaders initReadOnlyHeaders() {
        return HttpHeaders.readOnlyHttpHeaders(new YaNettyHeadersAdapter(this.request.requestHeaders()));
    }

    private static Publisher<ByteBuf> toByteBuffs(Publisher<? extends DataBuffer> dataBuffers) {
        return Flux.from(dataBuffers).map(NettyDataBufferFactory::toByteBuf);
    }

}
