package ru.yandex.intranet.d.util.http;

import java.util.Collection;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.Nullable;

import io.netty.handler.codec.http.cookie.Cookie;
import io.netty.handler.codec.http.cookie.DefaultCookie;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.NettyDataBufferFactory;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseCookie;
import org.springframework.http.client.reactive.ClientHttpResponse;
import org.springframework.lang.NonNull;
import org.springframework.util.ClassUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.util.ObjectUtils;
import reactor.core.publisher.Flux;
import reactor.netty.Connection;
import reactor.netty.NettyInbound;
import reactor.netty.http.client.HttpClientResponse;

/**
 * Companion http response for customized reactor client http connector
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class YaReactorClientHttpResponse implements ClientHttpResponse {

    private static final boolean REACTOR_NETTY_REQUEST_CHANNEL_OPERATION_ID_PRESENT = ClassUtils.isPresent(
            "reactor.netty.ChannelOperationsId", YaReactorClientHttpResponse.class.getClassLoader());

    private static final Logger LOG = LoggerFactory.getLogger(YaReactorClientHttpResponse.class);

    private final HttpClientResponse response;
    private final HttpHeaders headers;
    private final NettyInbound inbound;
    private final NettyDataBufferFactory bufferFactory;
    private final AtomicInteger state = new AtomicInteger();

    public YaReactorClientHttpResponse(HttpClientResponse response, Connection connection) {
        this.response = response;
        this.headers = HttpHeaders.readOnlyHttpHeaders(new YaNettyHeadersAdapter(response.responseHeaders()));
        this.inbound = connection.inbound();
        this.bufferFactory = new NettyDataBufferFactory(connection.outbound().alloc());
    }

    @NonNull
    @Override
    public String getId() {
        String id = null;
        if (REACTOR_NETTY_REQUEST_CHANNEL_OPERATION_ID_PRESENT) {
            id = extractChannelOperationId(response);
        }
        if (id == null && response instanceof Connection) {
            id = ((Connection) response).channel().id().asShortText();
        }
        return (id != null ? id : ObjectUtils.getIdentityHexString(this));
    }

    @NonNull
    @Override
    public HttpStatus getStatusCode() {
        return HttpStatus.valueOf(getRawStatusCode());
    }

    @Override
    public int getRawStatusCode() {
        return response.status().code();
    }

    @NonNull
    @Override
    public MultiValueMap<String, ResponseCookie> getCookies() {
        MultiValueMap<String, ResponseCookie> result = new LinkedMultiValueMap<>();
        response.cookies().values().stream().flatMap(Collection::stream).forEach(c ->
                result.add(c.name(), ResponseCookie.fromClientResponse(c.name(), c.value())
                        .domain(c.domain())
                        .path(c.path())
                        .maxAge(c.maxAge())
                        .secure(c.isSecure())
                        .httpOnly(c.isHttpOnly())
                        .sameSite(getSameSite(c))
                        .build()));
        return CollectionUtils.unmodifiableMultiValueMap(result);
    }

    @NonNull
    @Override
    public Flux<DataBuffer> getBody() {
        return inbound.receive().doOnSubscribe(s -> {
            if (state.compareAndSet(0, 1)) {
                return;
            }
            if (state.get() == 2) {
                throw new IllegalStateException(
                        "The client response body has been released already due to cancellation.");
            }
        }).map(byteBuf -> {
            byteBuf.retain();
            return bufferFactory.wrap(byteBuf);
        });
    }

    @NonNull
    @Override
    public HttpHeaders getHeaders() {
        return headers;
    }

    @Override
    public String toString() {
        return "YaReactorClientHttpResponse{" +
                "request=[" + response.method().name() + " " + response.uri() + "]," +
                "status=" + getRawStatusCode() + '}';
    }

    void releaseAfterCancel(HttpMethod method) {
        if (mayHaveBody(method) && state.compareAndSet(0, 2)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("[" + getId() + "]" + "Releasing body, not yet subscribed.");
            }
            inbound.receive().doOnNext(byteBuf -> { }).subscribe(byteBuf -> { }, ex -> { });
        }
    }

    private boolean mayHaveBody(HttpMethod method) {
        int code = getRawStatusCode();
        return !((code >= 100 && code < 200) || code == 204 || code == 205 ||
                method.equals(HttpMethod.HEAD) || getHeaders().getContentLength() == 0);
    }

    @Nullable
    private String extractChannelOperationId(HttpClientResponse response) {
        if (response instanceof reactor.netty.ChannelOperationsId) {
            return LOG.isDebugEnabled()
                    ? ((reactor.netty.ChannelOperationsId) response).asLongText()
                    : ((reactor.netty.ChannelOperationsId) response).asShortText();
        }
        return null;
    }

    @Nullable
    private String getSameSite(Cookie cookie) {
        if (cookie instanceof DefaultCookie) {
            DefaultCookie defaultCookie = (DefaultCookie) cookie;
            if (defaultCookie.sameSite() != null) {
                return defaultCookie.sameSite().name();
            }
        }
        return null;
    }

}
