package ru.yandex.intranet.d.util.paging;

import java.util.Locale;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.databind.ObjectReader;
import org.springframework.context.MessageSource;

import ru.yandex.intranet.d.util.JsonReader;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;

/**
 * Page request.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class PageRequest {

    private static final int DEFAULT_LIMIT = 100;
    private static final long MAX_LIMIT = 100;

    private final String continuationToken;
    private final long limit;

    public PageRequest(String continuationToken, Long limit) {
        this.continuationToken = continuationToken;
        this.limit = limit != null ? limit : DEFAULT_LIMIT;
    }

    public <T> Result<Validated<T>> validate(JsonReader<T> tokenReader, MessageSource messages, Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (limit > MAX_LIMIT) {
            errors.addError(TypedError.badRequest(messages
                    .getMessage("errors.limit.is.too.large", null, locale)));
        }
        if (limit <= 0) {
            errors.addError(TypedError.badRequest(messages
                    .getMessage("errors.limit.is.too.small", null, locale)));
        }
        T validatedContinuationToken;
        if (continuationToken != null) {
            Optional<T> decodedToken = ContinuationTokens.decode(continuationToken, tokenReader);
            if (decodedToken.isEmpty()) {
                errors.addError(TypedError.badRequest(messages
                        .getMessage("errors.invalid.continuation.token", null, locale)));
            }
            validatedContinuationToken = decodedToken.orElse(null);
        } else {
            validatedContinuationToken = null;
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(new Validated<>(validatedContinuationToken, (int) limit));
    }

    public <T> Result<Validated<T>> validate(ObjectReader tokenReader, MessageSource messages, Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (limit > MAX_LIMIT) {
            errors.addError(TypedError.badRequest(messages
                    .getMessage("errors.limit.is.too.large", null, locale)));
        }
        if (limit <= 0) {
            errors.addError(TypedError.badRequest(messages
                    .getMessage("errors.limit.is.too.small", null, locale)));
        }
        T validatedContinuationToken;
        if (continuationToken != null) {
            Optional<T> decodedToken = ContinuationTokens.decode(continuationToken, tokenReader);
            if (decodedToken.isEmpty()) {
                errors.addError(TypedError.badRequest(messages
                        .getMessage("errors.invalid.continuation.token", null, locale)));
            }
            validatedContinuationToken = decodedToken.orElse(null);
        } else {
            validatedContinuationToken = null;
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(new Validated<>(validatedContinuationToken, (int) limit));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PageRequest that = (PageRequest) o;
        return limit == that.limit &&
                Objects.equals(continuationToken, that.continuationToken);
    }

    @Override
    public int hashCode() {
        return Objects.hash(continuationToken, limit);
    }

    @Override
    public String toString() {
        return "PageRequest{" +
                "continuationToken='" + continuationToken + '\'' +
                ", limit=" + limit +
                '}';
    }

    public static final class Validated<T> {

        private final T continuationToken;
        private final int limit;

        public Validated(T continuationToken, int limit) {
            this.continuationToken = continuationToken;
            this.limit = limit;
        }

        public Optional<T> getContinuationToken() {
            return Optional.ofNullable(continuationToken);
        }

        public int getLimit() {
            return limit;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Validated<?> validated = (Validated<?>) o;
            return limit == validated.limit &&
                    Objects.equals(continuationToken, validated.continuationToken);
        }

        @Override
        public int hashCode() {
            return Objects.hash(continuationToken, limit);
        }

        @Override
        public String toString() {
            return "Validated{" +
                    "continuationToken=" + continuationToken +
                    ", limit=" + limit +
                    '}';
        }

    }

}
