package ru.yandex.intranet.d.util.result;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;


/**
 * Error collection.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ErrorCollection {

    private static final ErrorCollection EMPTY = new ErrorCollection(Collections.emptySet(), Collections.emptyMap(),
            Collections.emptyMap());

    private final Set<TypedError> errors;
    private final Map<String, Set<TypedError>> fieldErrors;
    private final Map<String, Set<Object>> details;

    private ErrorCollection(Set<TypedError> errors, Map<String, Set<TypedError>> fieldErrors,
                            Map<String, Set<Object>> details) {
        this.errors = errors;
        this.fieldErrors = fieldErrors;
        this.details = details;
    }

    public static ErrorCollection empty() {
        return EMPTY;
    }

    public static Builder builder() {
        return new Builder();
    }

    public Set<TypedError> getErrors() {
        return errors;
    }

    public Map<String, Set<TypedError>> getFieldErrors() {
        return fieldErrors;
    }

    public Map<String, Set<Object>> getDetails() {
        return details;
    }

    public boolean hasAnyErrors() {
        return !errors.isEmpty()
                || (!fieldErrors.isEmpty() && fieldErrors.values().stream().anyMatch(e -> !e.isEmpty()));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final ErrorCollection that = (ErrorCollection) o;
        return errors.equals(that.errors) &&
                fieldErrors.equals(that.fieldErrors) &&
                details.equals(that.details);
    }

    @Override
    public int hashCode() {
        return Objects.hash(errors, fieldErrors, details);
    }

    @Override
    public String toString() {
        return "ErrorCollection{" +
                "errors=" + errors +
                ", fieldErrors=" + fieldErrors +
                ", details=" + details +
                '}';
    }

    public static final class Builder {

        private final Set<TypedError> errors = new HashSet<>();
        private final Map<String, Set<TypedError>> fieldErrors = new HashMap<>();
        private final Map<String, Set<Object>> details = new HashMap<>();

        private Builder() {
        }

        public Builder addError(TypedError error) {
            Objects.requireNonNull(error, "Error is required.");
            errors.add(error);
            return this;
        }

        public Builder addError(String field, TypedError error) {
            Objects.requireNonNull(field, "Field is required.");
            Objects.requireNonNull(error, "Error is required.");
            fieldErrors.computeIfAbsent(field, k -> new HashSet<>()).add(error);
            return this;
        }

        public Builder addDetail(String key, Object value) {
            Objects.requireNonNull(key, "Key is required.");
            Objects.requireNonNull(value, "Value is required.");
            details.computeIfAbsent(key, k -> new HashSet<>()).add(value);
            return this;
        }

        public Builder add(ErrorCollection errorCollection) {
            Objects.requireNonNull(errorCollection, "Error collection is required.");
            errors.addAll(errorCollection.getErrors());
            errorCollection.getFieldErrors()
                    .forEach((k, v) -> fieldErrors.computeIfAbsent(k, x -> new HashSet<>()).addAll(v));
            errorCollection.getDetails()
                    .forEach((k, v) -> details.computeIfAbsent(k, x -> new HashSet<>()).addAll(v));
            return this;
        }

        public Builder add(ErrorCollection.Builder errorCollectionBuilder) {
            Objects.requireNonNull(errorCollectionBuilder, "Error collection is required.");
            errors.addAll(errorCollectionBuilder.errors);
            errorCollectionBuilder.fieldErrors
                    .forEach((k, v) -> fieldErrors.computeIfAbsent(k, x -> new HashSet<>()).addAll(v));
            errorCollectionBuilder.details
                    .forEach((k, v) -> details.computeIfAbsent(k, x -> new HashSet<>()).addAll(v));
            return this;
        }

        public ErrorCollection build() {
            return new ErrorCollection(errors, fieldErrors, details);
        }

        public boolean hasAnyErrors() {
            return !errors.isEmpty()
                    || (!fieldErrors.isEmpty() && fieldErrors.values().stream().anyMatch(e -> !e.isEmpty()));
        }

        public boolean hasAnyDetails() {
            return !details.isEmpty();
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Builder builder = (Builder) o;
            return Objects.equals(errors, builder.errors) &&
                    Objects.equals(fieldErrors, builder.fieldErrors) &&
                    Objects.equals(details, builder.details);
        }

        @Override
        public int hashCode() {
            return Objects.hash(errors, fieldErrors, details);
        }

        @Override
        public String toString() {
            return "Builder{" +
                    "errors=" + errors +
                    ", fieldErrors=" + fieldErrors +
                    '}';
        }

    }

}
